/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.common.protocol.message;

import static org.mule.munit.common.protocol.message.MessageField.*;
import static org.mule.munit.common.protocol.message.MessageID.AFTER_SUITE_END;
import static org.mule.munit.common.protocol.message.MessageID.AFTER_SUITE_START;
import static org.mule.munit.common.protocol.message.MessageID.BEFORE_SUITE_END;
import static org.mule.munit.common.protocol.message.MessageID.BEFORE_SUITE_START;
import static org.mule.munit.common.protocol.message.MessageID.CONTAINER_START_FAILURE;
import static org.mule.munit.common.protocol.message.MessageID.CONTAINER_SUITE_START_FAILURE;
import static org.mule.munit.common.protocol.message.MessageID.COVERAGE_REPORT;
import static org.mule.munit.common.protocol.message.MessageID.RUN_FINISH;
import static org.mule.munit.common.protocol.message.MessageID.RUN_START;
import static org.mule.munit.common.protocol.message.MessageID.SUITE_END;
import static org.mule.munit.common.protocol.message.MessageID.SUITE_START;
import static org.mule.munit.common.protocol.message.MessageID.TEST_END;
import static org.mule.munit.common.protocol.message.MessageID.TEST_START;
import static org.mule.munit.common.protocol.message.MessageID.UNEXPECTED_ERROR;
import static org.mule.munit.common.protocol.message.MessageID.UNEXPECTED_SUITE_ERROR;

import org.mule.munit.common.protocol.listeners.RunEventListener;

import com.google.gson.Gson;

/**
 * The goal of this class is to parse a string representing a {@link RunMessage} and react to it by calling the different methods
 * of the interface {@link RunEventListener}
 *
 * @author Mulesoft Inc.
 * @since 2.0.0
 */
public class RunMessageParser {

  private RunEventListener listener;

  public RunMessageParser(RunEventListener listener) {
    this.listener = listener;
  }

  public void parseAndNotify(String message) {
    RunMessage cmd = new Gson().fromJson(message, RunMessage.class);
    switch (cmd.getId()) {
      case RUN_START:
        listener.notifyRunStart();
        break;
      case SUITE_START:
        listener.notifySuiteStart(cmd.get(MUNIT_SUITE_KEY), cmd.get(PARAMETERIZATION_KEY),
                                  Integer.parseInt(cmd.get(NUMBER_OF_TESTS_KEY)));
        break;
      case CONTAINER_START_FAILURE:
        listener.notifyContainerFailure(cmd.get(STACK_TRACE_KEY));
        break;
      case CONTAINER_SUITE_START_FAILURE:
        listener.notifyContainerFailure(cmd.get(MUNIT_SUITE_KEY), cmd.get(PARAMETERIZATION_KEY), cmd.get(STACK_TRACE_KEY));
        break;
      case BEFORE_SUITE_START:
        listener.notifyBeforeSuiteStart(cmd.get(NAME_KEY));
        break;
      case BEFORE_SUITE_END:
        listener.notifyBeforeSuiteEnd(cmd.get(NAME_KEY), cmd.get(STACK_TRACE_KEY), cmd.getStatus(STATUS_KEY));
        break;
      case TEST_START:
        listener.notifyTestStart(cmd.get(NAME_KEY), cmd.get(DESCRIPTION_KEY), cmd.getBoolean(IGNORED_KEY), cmd.get(FILE_NAME));
        break;
      case TEST_END:
        listener.notifyTestEnd(cmd.get(NAME_KEY), cmd.get(STACK_TRACE_KEY), cmd.getStatus(STATUS_KEY),
                               cmd.getLong(ELAPSED_TIME_KEY), cmd.get(FILE_NAME));
        break;
      case AFTER_SUITE_START:
        listener.notifyAfterSuiteStart(cmd.get(NAME_KEY));
        break;
      case AFTER_SUITE_END:
        listener.notifyAfterSuiteEnd(cmd.get(NAME_KEY), cmd.get(STACK_TRACE_KEY), cmd.getStatus(STATUS_KEY));
        break;
      case SUITE_END:
        listener.notifySuiteEnd(cmd.get(MUNIT_SUITE_KEY), cmd.get(PARAMETERIZATION_KEY), cmd.getLong(ELAPSED_TIME_KEY));
        break;
      case RUN_FINISH:
        listener.notifyRunFinish();
        break;
      case COVERAGE_REPORT:
        listener.notifyCoverageReport(cmd.get(COVERAGE_REPORT_KEY));
        break;
      case UNEXPECTED_SUITE_ERROR:
        listener.notifySuiteUnexpectedError(cmd.get(NAME_KEY), cmd.get(STACK_TRACE_KEY));
        break;
      case UNEXPECTED_ERROR:
        listener.notifyUnexpectedError(cmd.get(STACK_TRACE_KEY));
        break;
      default:
        listener.notifyUnknown(message);
        break;

    }
  }

}
