/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.common.api.model;

import java.util.Objects;

import org.mule.metadata.api.annotation.TypeAliasAnnotation;
import org.mule.metadata.api.annotation.TypeIdAnnotation;
import org.mule.metadata.api.builder.BaseTypeBuilder;
import org.mule.metadata.api.builder.ObjectTypeBuilder;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.java.api.annotation.ClassInformationAnnotation;
import org.mule.runtime.api.metadata.resolving.InputStaticTypeResolver;
import org.mule.runtime.extension.api.annotation.Alias;
import org.mule.runtime.extension.api.annotation.dsl.xml.ParameterDsl;
import org.mule.runtime.extension.api.annotation.param.Parameter;
import org.mule.runtime.extension.api.annotation.param.display.Placement;
import org.mule.runtime.extension.api.declaration.type.annotation.TypeDslAnnotation;

import static org.mule.metadata.java.api.JavaTypeLoader.JAVA;


/**
 * Definition of the processor attribute that wants to be mocked.
 *
 * @author Mulesoft Inc.
 * @since 2.0.0
 */
@Alias(value = "with-attribute")
public class Attribute {

  /**
   * Definition of the attribute properties metadata.
   */
  public static class DefaultTypeResolver extends InputStaticTypeResolver {

    @Override
    public MetadataType getStaticMetadata() {
      ObjectTypeBuilder objectTypeBuilder = BaseTypeBuilder.create(JAVA).objectType()
          .with(new TypeAliasAnnotation("with-attribute"))
          .with(new TypeIdAnnotation("org.mule.munit.common.api.model.Attribute"))
          .with(new TypeDslAnnotation(true, false, null, null));

      objectTypeBuilder.addField().key("attributeName").value().stringType();
      objectTypeBuilder.addField().key("whereValue").value().stringType().with(new ClassInformationAnnotation(Object.class));

      return objectTypeBuilder.build();
    }
  }

  /** The name of the attribute of the processor */
  @Parameter
  @Placement(order = 0)
  private String attributeName;

  /** The object that need to match (can be a matcher expression) */
  @Parameter
  @ParameterDsl(allowReferences = false)
  @Placement(order = 1)
  private Object whereValue;

  /**
   * Factory method to simplify test coding
   *
   * @param name The Name of the attribute
   * @param whereValue Its value
   *
   * @return A new instance of the Attribute
   */
  public static Attribute create(String name, String whereValue) {
    Attribute attribute = new Attribute();
    attribute.setAttributeName(name);
    attribute.setWhereValue(whereValue);

    return attribute;
  }

  public String getAttributeName() {
    return attributeName;
  }

  public void setAttributeName(String attributeName) {
    this.attributeName = attributeName;
  }

  public Object getWhereValue() {
    return whereValue;
  }

  public void setWhereValue(Object whereValue) {
    this.whereValue = whereValue;
  }

  @Override
  public String toString() {
    return "Attribute{" +
        "attributeName='" + attributeName + '\'' +
        ", whereValue=" + whereValue +
        '}';
  }

  @Override
  public boolean equals(Object object) {
    if (!(object instanceof Attribute)) {
      return false;

    } else if (object == this) {
      return true;
    }

    Attribute that = (Attribute) object;

    return Objects.equals(this.getAttributeName(), that.getAttributeName()) &&
        Objects.equals(this.getWhereValue(), that.getWhereValue());
  }

  @Override
  public int hashCode() {
    return Objects.hash(getAttributeName(), getWhereValue());
  }
}
