"use strict";
/**
 * Copyright (c) Microsoft Corporation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.PlaywrightServer = void 0;
const debug = require("debug");
const http = require("http");
const WebSocket = require("ws");
const debugController_1 = require("../debug/debugController");
const dispatcher_1 = require("../dispatchers/dispatcher");
const playwrightDispatcher_1 = require("../dispatchers/playwrightDispatcher");
const playwright_1 = require("../server/playwright");
const processLauncher_1 = require("../server/processLauncher");
const tracer_1 = require("../trace/tracer");
const harTracer_1 = require("../trace/harTracer");
const debugLog = debug('pw:server');
debugController_1.installDebugController();
tracer_1.installTracer();
harTracer_1.installHarTracer();
class PlaywrightServer {
    listen(port) {
        this._server = http.createServer((request, response) => {
            response.end('Running');
        });
        this._server.on('error', error => debugLog(error));
        this._server.listen(port);
        debugLog('Listening on ' + port);
        const wsServer = new WebSocket.Server({ server: this._server, path: '/ws' });
        wsServer.on('connection', async (ws) => {
            if (this._client) {
                ws.close();
                return;
            }
            this._client = ws;
            debugLog('Incoming connection');
            const dispatcherConnection = new dispatcher_1.DispatcherConnection();
            ws.on('message', message => dispatcherConnection.dispatch(JSON.parse(message.toString())));
            ws.on('close', () => {
                debugLog('Client closed');
                this._onDisconnect();
            });
            ws.on('error', error => {
                debugLog('Client error ' + error);
                this._onDisconnect();
            });
            dispatcherConnection.onmessage = message => ws.send(JSON.stringify(message));
            const playwright = new playwright_1.Playwright(__dirname, require('../../browsers.json')['browsers']);
            new playwrightDispatcher_1.PlaywrightDispatcher(dispatcherConnection.rootDispatcher(), playwright);
        });
    }
    async close() {
        if (!this._server)
            return;
        debugLog('Closing server');
        await new Promise(f => this._server.close(f));
        await processLauncher_1.gracefullyCloseAll();
    }
    async _onDisconnect() {
        await processLauncher_1.gracefullyCloseAll();
        this._client = undefined;
    }
}
exports.PlaywrightServer = PlaywrightServer;
//# sourceMappingURL=playwrightServer.js.map