"use strict";
/**
 * Copyright (c) Microsoft Corporation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.videoById = exports.actionById = exports.readTraceFile = exports.trace = void 0;
exports.trace = require("../../trace/traceTypes");
function readTraceFile(events, traceModel, filePath) {
    const contextEntries = new Map();
    const pageEntries = new Map();
    for (const event of events) {
        switch (event.type) {
            case 'context-created': {
                contextEntries.set(event.contextId, {
                    filePath,
                    name: filePath.substring(filePath.lastIndexOf('/') + 1),
                    startTime: Number.MAX_VALUE,
                    endTime: Number.MIN_VALUE,
                    created: event,
                    destroyed: undefined,
                    pages: [],
                    resourcesByUrl: new Map(),
                });
                break;
            }
            case 'context-destroyed': {
                contextEntries.get(event.contextId).destroyed = event;
                break;
            }
            case 'page-created': {
                const pageEntry = {
                    created: event,
                    destroyed: undefined,
                    actions: [],
                    resources: [],
                    interestingEvents: [],
                };
                pageEntries.set(event.pageId, pageEntry);
                contextEntries.get(event.contextId).pages.push(pageEntry);
                break;
            }
            case 'page-destroyed': {
                pageEntries.get(event.pageId).destroyed = event;
                break;
            }
            case 'page-video': {
                const pageEntry = pageEntries.get(event.pageId);
                pageEntry.video = { video: event, videoId: event.contextId + '/' + event.pageId };
                break;
            }
            case 'action': {
                const pageEntry = pageEntries.get(event.pageId);
                const action = {
                    actionId: event.contextId + '/' + event.pageId + '/' + pageEntry.actions.length,
                    action: event,
                    resources: pageEntry.resources,
                };
                pageEntry.resources = [];
                pageEntry.actions.push(action);
                break;
            }
            case 'resource': {
                const contextEntry = contextEntries.get(event.contextId);
                const pageEntry = pageEntries.get(event.pageId);
                const action = pageEntry.actions[pageEntry.actions.length - 1];
                if (action)
                    action.resources.push(event);
                else
                    pageEntry.resources.push(event);
                let responseEvents = contextEntry.resourcesByUrl.get(event.url);
                if (!responseEvents) {
                    responseEvents = [];
                    contextEntry.resourcesByUrl.set(event.url, responseEvents);
                }
                responseEvents.push(event);
                break;
            }
            case 'dialog-opened':
            case 'dialog-closed':
            case 'navigation':
            case 'load': {
                const pageEntry = pageEntries.get(event.pageId);
                pageEntry.interestingEvents.push(event);
                break;
            }
        }
        const contextEntry = contextEntries.get(event.contextId);
        contextEntry.startTime = Math.min(contextEntry.startTime, event.timestamp);
        contextEntry.endTime = Math.max(contextEntry.endTime, event.timestamp);
    }
    traceModel.contexts.push(...contextEntries.values());
}
exports.readTraceFile = readTraceFile;
function actionById(traceModel, actionId) {
    const [contextId, pageId, actionIndex] = actionId.split('/');
    const context = traceModel.contexts.find(entry => entry.created.contextId === contextId);
    const page = context.pages.find(entry => entry.created.pageId === pageId);
    const action = page.actions[+actionIndex];
    return { context, page, action };
}
exports.actionById = actionById;
function videoById(traceModel, videoId) {
    const [contextId, pageId] = videoId.split('/');
    const context = traceModel.contexts.find(entry => entry.created.contextId === contextId);
    const page = context.pages.find(entry => entry.created.pageId === pageId);
    return { context, page };
}
exports.videoById = videoById;
//# sourceMappingURL=traceModel.js.map