/*
 * Decompiled with CFR 0.152.
 */
package com.microsoft.aad.adal;

import android.annotation.SuppressLint;
import android.annotation.TargetApi;
import android.content.Context;
import android.security.KeyPairGeneratorSpec;
import android.util.Base64;
import com.microsoft.aad.adal.ADALError;
import com.microsoft.aad.adal.AuthenticationSettings;
import com.microsoft.aad.adal.ClientAnalytics;
import com.microsoft.aad.adal.InstrumentationPropertiesBuilder;
import com.microsoft.aad.adal.Logger;
import com.microsoft.aad.adal.StringExtensions;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.math.BigInteger;
import java.security.DigestException;
import java.security.GeneralSecurityException;
import java.security.Key;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.security.cert.CertificateException;
import java.security.spec.AlgorithmParameterSpec;
import java.util.Calendar;
import java.util.Date;
import java.util.Locale;
import javax.crypto.Cipher;
import javax.crypto.KeyGenerator;
import javax.crypto.Mac;
import javax.crypto.SecretKey;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.SecretKeySpec;
import javax.security.auth.x500.X500Principal;

public class StorageHelper {
    private static final String TAG = "StorageHelper";
    private static final String HMAC_KEY_HASH_ALGORITHM = "SHA256";
    private static final String KEY_STORE_CERT_ALIAS = "AdalKey";
    private static final String ADALKS = "adalks";
    private static final String KEYSPEC_ALGORITHM = "AES";
    private static final String WRAP_ALGORITHM = "RSA/ECB/PKCS1Padding";
    private static final String CIPHER_ALGORITHM = "AES/CBC/PKCS5Padding";
    private static final String HMAC_ALGORITHM = "HmacSHA256";
    private static final int KEY_SIZE = 256;
    public static final int DATA_KEY_LENGTH = 16;
    public static final int HMAC_LENGTH = 32;
    public static final String VERSION_ANDROID_KEY_STORE = "A001";
    public static final String VERSION_USER_DEFINED = "U001";
    private static final int KEY_VERSION_BLOB_LENGTH = 4;
    private static final String ENCODE_VERSION = "E1";
    private static final int KEY_FILE_SIZE = 1024;
    private static final String ANDROID_KEY_STORE = "AndroidKeyStore";
    private final Context mContext;
    private final SecureRandom mRandom;
    private KeyPair mKeyPair;
    private String mBlobVersion;
    private SecretKey mKey = null;
    private SecretKey mHMACKey = null;
    private SecretKey mSecretKeyFromAndroidKeyStore = null;

    public StorageHelper(Context context) {
        this.mContext = context;
        this.mRandom = new SecureRandom();
    }

    public String encrypt(String clearText) throws GeneralSecurityException, IOException {
        Logger.v(TAG, "Starting encryption");
        if (StringExtensions.isNullOrBlank(clearText)) {
            throw new IllegalArgumentException("Input is empty or null");
        }
        this.mKey = this.loadSecretKeyForEncryption();
        this.mHMACKey = this.getHMacKey(this.mKey);
        Logger.v(TAG, "Encrypt version:" + this.mBlobVersion);
        byte[] blobVersion = this.mBlobVersion.getBytes("UTF_8");
        byte[] bytes = clearText.getBytes("UTF_8");
        byte[] iv = new byte[16];
        this.mRandom.nextBytes(iv);
        IvParameterSpec ivSpec = new IvParameterSpec(iv);
        Cipher cipher = Cipher.getInstance(CIPHER_ALGORITHM);
        Mac mac = Mac.getInstance(HMAC_ALGORITHM);
        cipher.init(1, (Key)this.mKey, ivSpec);
        byte[] encrypted = cipher.doFinal(bytes);
        mac.init(this.mHMACKey);
        mac.update(blobVersion);
        mac.update(encrypted);
        mac.update(iv);
        byte[] macDigest = mac.doFinal();
        byte[] blobVerAndEncryptedDataAndIVAndMacDigest = new byte[blobVersion.length + encrypted.length + iv.length + macDigest.length];
        System.arraycopy(blobVersion, 0, blobVerAndEncryptedDataAndIVAndMacDigest, 0, blobVersion.length);
        System.arraycopy(encrypted, 0, blobVerAndEncryptedDataAndIVAndMacDigest, blobVersion.length, encrypted.length);
        System.arraycopy(iv, 0, blobVerAndEncryptedDataAndIVAndMacDigest, blobVersion.length + encrypted.length, iv.length);
        System.arraycopy(macDigest, 0, blobVerAndEncryptedDataAndIVAndMacDigest, blobVersion.length + encrypted.length + iv.length, macDigest.length);
        String encryptedText = new String(Base64.encode((byte[])blobVerAndEncryptedDataAndIVAndMacDigest, (int)2), "UTF_8");
        Logger.v(TAG, "Finished encryption");
        return this.getEncodeVersionLengthPrefix() + ENCODE_VERSION + encryptedText;
    }

    public String decrypt(String encryptedBlob) throws GeneralSecurityException, IOException {
        Logger.v(TAG, "Starting decryption");
        if (StringExtensions.isNullOrBlank(encryptedBlob)) {
            throw new IllegalArgumentException("Input is empty or null");
        }
        int encodeVersionLength = encryptedBlob.charAt(0) - 97;
        if (encodeVersionLength <= 0) {
            throw new IllegalArgumentException(String.format("Encode version length: '%s' is not valid, it must be greater of equal to 0", encodeVersionLength));
        }
        if (!encryptedBlob.substring(1, 1 + encodeVersionLength).equals(ENCODE_VERSION)) {
            throw new IllegalArgumentException(String.format("Encode version received was: '%s', Encode version supported is: '%s'", encryptedBlob, ENCODE_VERSION));
        }
        byte[] bytes = Base64.decode((String)encryptedBlob.substring(1 + encodeVersionLength), (int)0);
        String keyVersion = new String(bytes, 0, 4, "UTF_8");
        Logger.v(TAG, "Encrypt version:" + keyVersion);
        SecretKey secretKey = this.loadSecretKeyForDecryption(keyVersion);
        SecretKey hmacKey = this.getHMacKey(secretKey);
        int ivIndex = bytes.length - 16 - 32;
        int macIndex = bytes.length - 32;
        int encryptedLength = ivIndex - 4;
        if (ivIndex < 0 || macIndex < 0 || encryptedLength < 0) {
            throw new IOException("Invalid byte array input for decryption.");
        }
        Cipher cipher = Cipher.getInstance(CIPHER_ALGORITHM);
        Mac mac = Mac.getInstance(HMAC_ALGORITHM);
        mac.init(hmacKey);
        mac.update(bytes, 0, macIndex);
        byte[] macDigest = mac.doFinal();
        this.assertHMac(bytes, macIndex, bytes.length, macDigest);
        cipher.init(2, (Key)secretKey, new IvParameterSpec(bytes, ivIndex, 16));
        String decrypted = new String(cipher.doFinal(bytes, 4, encryptedLength), "UTF_8");
        Logger.v(TAG, "Finished decryption");
        return decrypted;
    }

    synchronized SecretKey loadSecretKeyForEncryption() throws IOException, GeneralSecurityException {
        if (this.mKey != null && this.mHMACKey != null) {
            return this.mKey;
        }
        byte[] secretKeyData = AuthenticationSettings.INSTANCE.getSecretKeyData();
        this.mBlobVersion = secretKeyData == null ? VERSION_ANDROID_KEY_STORE : VERSION_USER_DEFINED;
        return this.getKeyOrCreate(this.mBlobVersion);
    }

    synchronized SecretKey loadSecretKeyForDecryption(String keyVersion) throws GeneralSecurityException, IOException {
        if (this.mSecretKeyFromAndroidKeyStore != null) {
            return this.mSecretKeyFromAndroidKeyStore;
        }
        return this.getKey(keyVersion);
    }

    private synchronized SecretKey getKeyOrCreate(String keyVersion) throws GeneralSecurityException, IOException {
        if (VERSION_USER_DEFINED.equals(keyVersion)) {
            return this.getSecretKey(AuthenticationSettings.INSTANCE.getSecretKeyData());
        }
        try {
            this.mSecretKeyFromAndroidKeyStore = this.getKey(keyVersion);
        }
        catch (IOException | GeneralSecurityException exception) {
            Logger.v(TAG, "Key does not exist in AndroidKeyStore, try to generate new keys.");
        }
        if (this.mSecretKeyFromAndroidKeyStore == null) {
            this.mKeyPair = this.generateKeyPairFromAndroidKeyStore();
            this.mSecretKeyFromAndroidKeyStore = this.generateSecretKey();
            byte[] keyWrapped = this.wrap(this.mSecretKeyFromAndroidKeyStore);
            this.writeKeyData(keyWrapped);
        }
        return this.mSecretKeyFromAndroidKeyStore;
    }

    private synchronized SecretKey getKey(String keyVersion) throws GeneralSecurityException, IOException {
        switch (keyVersion) {
            case "U001": {
                return this.getSecretKey(AuthenticationSettings.INSTANCE.getSecretKeyData());
            }
            case "A001": {
                this.mKeyPair = this.readKeyPair();
                this.mSecretKeyFromAndroidKeyStore = this.getUnwrappedSecretKey();
                return this.mSecretKeyFromAndroidKeyStore;
            }
        }
        throw new IOException("Unknown keyVersion.");
    }

    @TargetApi(value=18)
    private synchronized KeyPair generateKeyPairFromAndroidKeyStore() throws GeneralSecurityException, IOException {
        KeyStore keyStore = KeyStore.getInstance(ANDROID_KEY_STORE);
        keyStore.load(null);
        Logger.v(TAG, "Generate KeyPair from AndroidKeyStore");
        Calendar start = Calendar.getInstance();
        Calendar end = Calendar.getInstance();
        int certValidYears = 100;
        end.add(1, 100);
        KeyPairGenerator generator = KeyPairGenerator.getInstance("RSA", ANDROID_KEY_STORE);
        generator.initialize(this.getKeyPairGeneratorSpec(this.mContext, start.getTime(), end.getTime()));
        try {
            return generator.generateKeyPair();
        }
        catch (IllegalStateException exception) {
            ClientAnalytics.logEvent(new AndroidKeyStoreFailureEvent(new InstrumentationPropertiesBuilder(exception)));
            throw new KeyStoreException(exception);
        }
    }

    private synchronized KeyPair readKeyPair() throws GeneralSecurityException, IOException {
        KeyStore.PrivateKeyEntry entry;
        if (!this.doesKeyPairExist()) {
            throw new KeyStoreException("KeyPair entry does not exist.");
        }
        Logger.v(TAG, "Reading Key entry");
        KeyStore keyStore = KeyStore.getInstance(ANDROID_KEY_STORE);
        keyStore.load(null);
        try {
            entry = (KeyStore.PrivateKeyEntry)keyStore.getEntry(KEY_STORE_CERT_ALIAS, null);
        }
        catch (RuntimeException e) {
            ClientAnalytics.logEvent(new AndroidKeyStoreFailureEvent(new InstrumentationPropertiesBuilder(e)));
            throw new KeyStoreException(e);
        }
        return new KeyPair(entry.getCertificate().getPublicKey(), entry.getPrivateKey());
    }

    private synchronized boolean doesKeyPairExist() throws GeneralSecurityException, IOException {
        boolean isKeyStoreCertAliasExisted;
        KeyStore keyStore = KeyStore.getInstance(ANDROID_KEY_STORE);
        keyStore.load(null);
        try {
            isKeyStoreCertAliasExisted = keyStore.containsAlias(KEY_STORE_CERT_ALIAS);
        }
        catch (NullPointerException exception) {
            ClientAnalytics.logEvent(new AndroidKeyStoreFailureEvent(new InstrumentationPropertiesBuilder(exception)));
            throw new KeyStoreException(exception);
        }
        return isKeyStoreCertAliasExisted;
    }

    @TargetApi(value=18)
    private AlgorithmParameterSpec getKeyPairGeneratorSpec(Context context, Date start, Date end) {
        String certInfo = String.format(Locale.ROOT, "CN=%s, OU=%s", KEY_STORE_CERT_ALIAS, context.getPackageName());
        return new KeyPairGeneratorSpec.Builder(context).setAlias(KEY_STORE_CERT_ALIAS).setSubject(new X500Principal(certInfo)).setSerialNumber(BigInteger.ONE).setStartDate(start).setEndDate(end).build();
    }

    private SecretKey getSecretKey(byte[] rawBytes) {
        if (rawBytes == null) {
            throw new IllegalArgumentException("rawBytes");
        }
        return new SecretKeySpec(rawBytes, KEYSPEC_ALGORITHM);
    }

    private SecretKey getHMacKey(SecretKey key) throws NoSuchAlgorithmException {
        byte[] encodedKey = key.getEncoded();
        if (encodedKey != null) {
            MessageDigest digester = MessageDigest.getInstance(HMAC_KEY_HASH_ALGORITHM);
            return new SecretKeySpec(digester.digest(encodedKey), KEYSPEC_ALGORITHM);
        }
        return key;
    }

    private char getEncodeVersionLengthPrefix() {
        return (char)(97 + ENCODE_VERSION.length());
    }

    private void assertHMac(byte[] digest, int start, int end, byte[] calculated) throws DigestException {
        if (calculated.length != end - start) {
            throw new IllegalArgumentException("Unexpected HMAC length");
        }
        int result = 0;
        for (int i = start; i < end; ++i) {
            result = (byte)(result | calculated[i - start] ^ digest[i]);
        }
        if (result != 0) {
            throw new DigestException();
        }
    }

    private SecretKey generateSecretKey() throws NoSuchAlgorithmException {
        KeyGenerator keygen = KeyGenerator.getInstance(KEYSPEC_ALGORITHM);
        keygen.init(256, this.mRandom);
        return keygen.generateKey();
    }

    @TargetApi(value=18)
    private synchronized SecretKey getUnwrappedSecretKey() throws GeneralSecurityException, IOException {
        SecretKey unwrappedSecretKey;
        Logger.v(TAG, "Reading SecretKey");
        try {
            byte[] wrappedSecretKey = this.readKeyData();
            unwrappedSecretKey = this.unwrap(wrappedSecretKey);
            Logger.v(TAG, "Finished reading SecretKey");
        }
        catch (IOException | GeneralSecurityException ex) {
            Logger.e(TAG, "Unwrap failed for AndroidKeyStore", "", ADALError.ANDROIDKEYSTORE_FAILED, ex);
            this.mKeyPair = null;
            this.deleteKeyFile();
            this.resetKeyPairFromAndroidKeyStore();
            Logger.v(TAG, "Removed previous key pair info.");
            throw ex;
        }
        return unwrappedSecretKey;
    }

    private void deleteKeyFile() {
        File keyFile = new File(this.mContext.getDir(this.mContext.getPackageName(), 0), ADALKS);
        if (keyFile.exists()) {
            Logger.v(TAG, "Delete KeyFile");
            if (!keyFile.delete()) {
                Logger.v(TAG, "Delete KeyFile failed");
            }
        }
    }

    @TargetApi(value=18)
    private synchronized void resetKeyPairFromAndroidKeyStore() throws KeyStoreException, NoSuchAlgorithmException, CertificateException, IOException {
        KeyStore keyStore = KeyStore.getInstance(ANDROID_KEY_STORE);
        keyStore.load(null);
        keyStore.deleteEntry(KEY_STORE_CERT_ALIAS);
    }

    @TargetApi(value=18)
    @SuppressLint(value={"GetInstance"})
    private byte[] wrap(SecretKey key) throws GeneralSecurityException {
        Logger.v(TAG, "Wrap secret key.");
        Cipher wrapCipher = Cipher.getInstance(WRAP_ALGORITHM);
        wrapCipher.init(3, this.mKeyPair.getPublic());
        return wrapCipher.wrap(key);
    }

    @TargetApi(value=18)
    @SuppressLint(value={"GetInstance"})
    private SecretKey unwrap(byte[] keyBlob) throws GeneralSecurityException {
        Cipher wrapCipher = Cipher.getInstance(WRAP_ALGORITHM);
        wrapCipher.init(4, this.mKeyPair.getPrivate());
        try {
            return (SecretKey)wrapCipher.unwrap(keyBlob, KEYSPEC_ALGORITHM, 3);
        }
        catch (IllegalArgumentException exception) {
            ClientAnalytics.logEvent(new AndroidKeyStoreFailureEvent(new InstrumentationPropertiesBuilder(exception)));
            throw new KeyStoreException(exception);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void writeKeyData(byte[] data) throws IOException {
        Logger.v(TAG, "Writing key data to a file");
        File keyFile = new File(this.mContext.getDir(this.mContext.getPackageName(), 0), ADALKS);
        try (FileOutputStream out = new FileOutputStream(keyFile);){
            ((OutputStream)out).write(data);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private byte[] readKeyData() throws IOException {
        File keyFile = new File(this.mContext.getDir(this.mContext.getPackageName(), 0), ADALKS);
        if (!keyFile.exists()) {
            throw new IOException("Key file to read does not exist");
        }
        Logger.v(TAG, "Reading key data from a file");
        try (FileInputStream in = new FileInputStream(keyFile);){
            int count;
            ByteArrayOutputStream bytes = new ByteArrayOutputStream();
            byte[] buffer = new byte[1024];
            while ((count = ((InputStream)in).read(buffer)) != -1) {
                bytes.write(buffer, 0, count);
            }
            byte[] byArray = bytes.toByteArray();
            return byArray;
        }
    }

    private static final class AndroidKeyStoreFailureEvent
    extends ClientAnalytics.Event {
        private AndroidKeyStoreFailureEvent(InstrumentationPropertiesBuilder builder) {
            super(StorageHelper.ANDROID_KEY_STORE, builder.add("Result", "Fail").build());
        }
    }
}

