/*
 * Decompiled with CFR 0.152.
 */
package com.microsoft.aad.adal;

import android.annotation.TargetApi;
import android.content.Context;
import android.os.Build;
import android.security.KeyPairGeneratorSpec;
import android.util.Base64;
import com.microsoft.aad.adal.ADALError;
import com.microsoft.aad.adal.AuthenticationSettings;
import com.microsoft.aad.adal.Logger;
import com.microsoft.aad.adal.StringExtensions;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.math.BigInteger;
import java.security.DigestException;
import java.security.GeneralSecurityException;
import java.security.Key;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.security.UnrecoverableKeyException;
import java.security.cert.CertificateException;
import java.security.spec.AlgorithmParameterSpec;
import java.util.Calendar;
import java.util.Date;
import java.util.Locale;
import javax.crypto.Cipher;
import javax.crypto.KeyGenerator;
import javax.crypto.Mac;
import javax.crypto.SecretKey;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.SecretKeySpec;
import javax.security.auth.x500.X500Principal;

public class StorageHelper {
    private static final String MAC_KEY_HASH_ALGORITHM = "SHA256";
    private static final String KEY_STORE_CERT_ALIAS = "AdalKey";
    private static final String ADALKS = "adalks";
    private static final String KEYSPEC_ALGORITHM = "AES";
    private static final String WRAP_ALGORITHM = "RSA/ECB/PKCS1Padding";
    private static final String TAG = "StorageHelper";
    private static final String CIPHER_ALGORITHM = "AES/CBC/PKCS5Padding";
    private static final String MAC_ALGORITHM = "HmacSHA256";
    private final SecureRandom mRandom;
    private static final int KEY_SIZE = 256;
    public static final int DATA_KEY_LENGTH = 16;
    public static final int MAC_LENGTH = 32;
    private KeyPair mKeyPair;
    private Context mContext;
    public static final String VERSION_ANDROID_KEY_STORE = "A001";
    public static final String VERSION_USER_DEFINED = "U001";
    private static final int KEY_VERSION_BLOB_LENGTH = 4;
    private static final String ENCODE_VERSION = "E1";
    private static final Object LOCK_OBJ = new Object();
    private String mBlobVersion;
    private SecretKey mKey = null;
    private SecretKey mMacKey = null;
    private static SecretKey sSecretKeyFromAndroidKeyStore = null;

    public StorageHelper(Context ctx) {
        this.mContext = ctx;
        this.mRandom = new SecureRandom();
    }

    public synchronized SecretKey loadSecretKeyForAPI() throws IOException, GeneralSecurityException {
        if (this.mKey != null && this.mMacKey != null) {
            return this.mKey;
        }
        byte[] secretKeyData = AuthenticationSettings.INSTANCE.getSecretKeyData();
        if (secretKeyData == null && Build.VERSION.SDK_INT < 18) {
            throw new IllegalStateException("Secret key must be provided for API < 18. Use AuthenticationSettings.INSTANCE.setSecretKey()");
        }
        if (secretKeyData != null) {
            Logger.v(TAG, "Encryption will use secret key from Settings");
            this.mKey = this.getSecretKey(secretKeyData);
            this.mMacKey = this.getMacKey(this.mKey);
            this.mBlobVersion = VERSION_USER_DEFINED;
        } else {
            try {
                this.mKey = this.getSecretKeyFromAndroidKeyStore();
                this.mMacKey = this.getMacKey(this.mKey);
                this.mBlobVersion = VERSION_ANDROID_KEY_STORE;
            }
            catch (IOException | GeneralSecurityException e) {
                Logger.e(TAG, "Failed to get private key from AndroidKeyStore", "", ADALError.ANDROIDKEYSTORE_FAILED, e);
                throw e;
            }
        }
        return this.mKey;
    }

    private SecretKey getKeyForVersion(String keyVersion) throws GeneralSecurityException, IOException {
        if (keyVersion.equals(VERSION_USER_DEFINED)) {
            return this.getSecretKey(AuthenticationSettings.INSTANCE.getSecretKeyData());
        }
        if (keyVersion.equals(VERSION_ANDROID_KEY_STORE)) {
            if (Build.VERSION.SDK_INT >= 18) {
                try {
                    return this.getSecretKeyFromAndroidKeyStore();
                }
                catch (IOException | GeneralSecurityException e) {
                    Logger.e(TAG, "Failed to get private key from AndroidKeyStore", "", ADALError.ANDROIDKEYSTORE_FAILED, e);
                    throw e;
                }
            }
            throw new IllegalArgumentException(String.format("keyVersion '%s' is not supported in this SDK. AndroidKeyStore is supported API18 and above.", keyVersion));
        }
        throw new IllegalArgumentException("keyVersion = " + keyVersion);
    }

    private SecretKey getSecretKey(byte[] rawBytes) {
        if (rawBytes != null) {
            return new SecretKeySpec(rawBytes, KEYSPEC_ALGORITHM);
        }
        throw new IllegalArgumentException("rawBytes");
    }

    private SecretKey getMacKey(SecretKey key) throws NoSuchAlgorithmException {
        byte[] encodedKey = key.getEncoded();
        if (encodedKey != null) {
            MessageDigest digester = MessageDigest.getInstance(MAC_KEY_HASH_ALGORITHM);
            return new SecretKeySpec(digester.digest(encodedKey), KEYSPEC_ALGORITHM);
        }
        return key;
    }

    public String decrypt(String value) throws GeneralSecurityException, IOException {
        Logger.v(TAG, "Starting decryption");
        if (StringExtensions.IsNullOrBlank(value)) {
            throw new IllegalArgumentException("Input is empty or null");
        }
        int encodeVersionLength = value.charAt(0) - 97;
        if (encodeVersionLength <= 0) {
            throw new IllegalArgumentException(String.format("Encode version length: '%s' is not valid, it must be greater of equal to 0", encodeVersionLength));
        }
        if (!value.substring(1, 1 + encodeVersionLength).equals(ENCODE_VERSION)) {
            throw new IllegalArgumentException(String.format("Encode version received was: '%s', Encode version supported is: '%s'", value, ENCODE_VERSION));
        }
        byte[] bytes = Base64.decode((String)value.substring(1 + encodeVersionLength), (int)0);
        String keyVersionCheck = new String(bytes, 0, 4, "UTF_8");
        Logger.v(TAG, "Encrypt version:" + keyVersionCheck);
        SecretKey versionKey = this.getKeyForVersion(keyVersionCheck);
        SecretKey versionMacKey = this.getMacKey(versionKey);
        int ivIndex = bytes.length - 16 - 32;
        int macIndex = bytes.length - 32;
        int encryptedLength = ivIndex - 4;
        if (ivIndex < 0 || macIndex < 0 || encryptedLength < 0) {
            throw new IllegalArgumentException("Given value is smaller than the IV vector and MAC length");
        }
        Cipher cipher = Cipher.getInstance(CIPHER_ALGORITHM);
        Mac mac = Mac.getInstance(MAC_ALGORITHM);
        mac.init(versionMacKey);
        mac.update(bytes, 0, macIndex);
        byte[] macDigest = mac.doFinal();
        this.assertMac(bytes, macIndex, bytes.length, macDigest);
        cipher.init(2, (Key)versionKey, new IvParameterSpec(bytes, ivIndex, 16));
        String decrypted = new String(cipher.doFinal(bytes, 4, encryptedLength), "UTF_8");
        Logger.v(TAG, "Finished decryption");
        return decrypted;
    }

    private char getEncodeVersionLengthPrefix() {
        return (char)(97 + ENCODE_VERSION.length());
    }

    public String encrypt(String clearText) throws GeneralSecurityException, IOException {
        Logger.v(TAG, "Starting encryption");
        if (StringExtensions.IsNullOrBlank(clearText)) {
            throw new IllegalArgumentException("Input is empty or null");
        }
        this.loadSecretKeyForAPI();
        Logger.v(TAG, "Encrypt version:" + this.mBlobVersion);
        byte[] blobVersion = this.mBlobVersion.getBytes("UTF_8");
        byte[] bytes = clearText.getBytes("UTF_8");
        byte[] iv = new byte[16];
        this.mRandom.nextBytes(iv);
        IvParameterSpec ivSpec = new IvParameterSpec(iv);
        Cipher cipher = Cipher.getInstance(CIPHER_ALGORITHM);
        Mac mac = Mac.getInstance(MAC_ALGORITHM);
        cipher.init(1, (Key)this.mKey, ivSpec);
        byte[] encrypted = cipher.doFinal(bytes);
        mac.init(this.mMacKey);
        mac.update(blobVersion);
        mac.update(encrypted);
        mac.update(iv);
        byte[] macDigest = mac.doFinal();
        byte[] blobVerAndEncryptedDataAndIVAndMacDigest = new byte[blobVersion.length + encrypted.length + iv.length + macDigest.length];
        System.arraycopy(blobVersion, 0, blobVerAndEncryptedDataAndIVAndMacDigest, 0, blobVersion.length);
        System.arraycopy(encrypted, 0, blobVerAndEncryptedDataAndIVAndMacDigest, blobVersion.length, encrypted.length);
        System.arraycopy(iv, 0, blobVerAndEncryptedDataAndIVAndMacDigest, blobVersion.length + encrypted.length, iv.length);
        System.arraycopy(macDigest, 0, blobVerAndEncryptedDataAndIVAndMacDigest, blobVersion.length + encrypted.length + iv.length, macDigest.length);
        String encryptedText = new String(Base64.encode((byte[])blobVerAndEncryptedDataAndIVAndMacDigest, (int)2), "UTF_8");
        Logger.v(TAG, "Finished encryption");
        return this.getEncodeVersionLengthPrefix() + ENCODE_VERSION + encryptedText;
    }

    private void assertMac(byte[] digest, int start, int end, byte[] calculated) throws DigestException {
        if (calculated.length != end - start) {
            throw new IllegalArgumentException("Unexpected MAC length");
        }
        int result = 0;
        for (int i = start; i < end; ++i) {
            result = (byte)(result | calculated[i - start] ^ digest[i]);
        }
        if (result != 0) {
            throw new DigestException();
        }
    }

    private final SecretKey generateSecretKey() throws NoSuchAlgorithmException {
        KeyGenerator keygen = KeyGenerator.getInstance(KEYSPEC_ALGORITHM);
        keygen.init(256, this.mRandom);
        return keygen.generateKey();
    }

    @TargetApi(value=18)
    private final synchronized SecretKey getSecretKeyFromAndroidKeyStore() throws IOException, GeneralSecurityException {
        if (sSecretKeyFromAndroidKeyStore != null) {
            return sSecretKeyFromAndroidKeyStore;
        }
        File keyFile = new File(this.mContext.getDir(this.mContext.getPackageName(), 0), ADALKS);
        if (this.mKeyPair == null) {
            this.mKeyPair = this.getKeyPairFromAndroidKeyStore();
            Logger.v(TAG, "Retrived keypair from androidKeyStore");
        }
        Cipher wrapCipher = Cipher.getInstance(WRAP_ALGORITHM);
        if (!keyFile.exists()) {
            Logger.v(TAG, "Key file does not exists");
            SecretKey key = this.generateSecretKey();
            Logger.v(TAG, "Wrapping SecretKey");
            byte[] keyWrapped = this.wrap(wrapCipher, key);
            Logger.v(TAG, "Writing SecretKey");
            StorageHelper.writeKeyData(keyFile, keyWrapped);
            Logger.v(TAG, "Finished writing SecretKey");
        }
        Logger.v(TAG, "Reading SecretKey");
        try {
            byte[] encryptedKey = StorageHelper.readKeyData(keyFile);
            if (encryptedKey == null || encryptedKey.length == 0) {
                throw new UnrecoverableKeyException("Couldn't find encrypted key in file");
            }
            sSecretKeyFromAndroidKeyStore = this.unwrap(wrapCipher, encryptedKey);
            Logger.v(TAG, "Finished reading SecretKey");
        }
        catch (IOException | GeneralSecurityException ex) {
            Logger.e(TAG, "Unwrap failed for AndroidKeyStore", "", ADALError.ANDROIDKEYSTORE_FAILED, ex);
            this.mKeyPair = null;
            sSecretKeyFromAndroidKeyStore = null;
            this.deleteKeyFile();
            this.resetKeyPairFromAndroidKeyStore();
            Logger.v(TAG, "Removed previous key pair info.");
            throw ex;
        }
        return sSecretKeyFromAndroidKeyStore;
    }

    private void deleteKeyFile() {
        File keyFile = new File(this.mContext.getDir(this.mContext.getPackageName(), 0), ADALKS);
        if (keyFile.exists()) {
            Logger.v(TAG, "Delete KeyFile");
            keyFile.delete();
        }
    }

    @TargetApi(value=18)
    private synchronized KeyPair getKeyPairFromAndroidKeyStore() throws GeneralSecurityException, IOException {
        boolean isKeyStoreCertAliasExisted;
        KeyStore keyStore = KeyStore.getInstance("AndroidKeyStore");
        keyStore.load(null);
        try {
            isKeyStoreCertAliasExisted = keyStore.containsAlias(KEY_STORE_CERT_ALIAS);
        }
        catch (NullPointerException exception) {
            throw new KeyStoreException(exception);
        }
        if (!isKeyStoreCertAliasExisted) {
            Logger.v(TAG, "Key entry is not available");
            Calendar start = Calendar.getInstance();
            Calendar end = Calendar.getInstance();
            end.add(1, 100);
            KeyPairGenerator generator = KeyPairGenerator.getInstance("RSA", "AndroidKeyStore");
            generator.initialize(this.getKeyPairGeneratorSpec(this.mContext, start.getTime(), end.getTime()));
            try {
                generator.generateKeyPair();
            }
            catch (IllegalStateException exception) {
                throw new KeyStoreException(exception);
            }
            Logger.v(TAG, "Key entry is generated");
        } else {
            Logger.v(TAG, "Key entry is available");
        }
        Logger.v(TAG, "Reading Key entry");
        try {
            KeyStore.PrivateKeyEntry entry = (KeyStore.PrivateKeyEntry)keyStore.getEntry(KEY_STORE_CERT_ALIAS, null);
            return new KeyPair(entry.getCertificate().getPublicKey(), entry.getPrivateKey());
        }
        catch (RuntimeException e) {
            throw new KeyStoreException(e);
        }
    }

    @TargetApi(value=18)
    private AlgorithmParameterSpec getKeyPairGeneratorSpec(Context context, Date start, Date end) {
        String certInfo = String.format(Locale.ROOT, "CN=%s, OU=%s", KEY_STORE_CERT_ALIAS, context.getPackageName());
        return new KeyPairGeneratorSpec.Builder(context).setAlias(KEY_STORE_CERT_ALIAS).setSubject(new X500Principal(certInfo)).setSerialNumber(BigInteger.ONE).setStartDate(start).setEndDate(end).build();
    }

    @TargetApi(value=18)
    private synchronized void resetKeyPairFromAndroidKeyStore() throws KeyStoreException, NoSuchAlgorithmException, CertificateException, IOException {
        KeyStore keyStore = KeyStore.getInstance("AndroidKeyStore");
        keyStore.load(null);
        keyStore.deleteEntry(KEY_STORE_CERT_ALIAS);
    }

    @TargetApi(value=18)
    private byte[] wrap(Cipher wrapCipher, SecretKey key) throws GeneralSecurityException {
        wrapCipher.init(3, this.mKeyPair.getPublic());
        return wrapCipher.wrap(key);
    }

    @TargetApi(value=18)
    private SecretKey unwrap(Cipher wrapCipher, byte[] keyBlob) throws GeneralSecurityException {
        wrapCipher.init(4, this.mKeyPair.getPrivate());
        try {
            return (SecretKey)wrapCipher.unwrap(keyBlob, KEYSPEC_ALGORITHM, 3);
        }
        catch (IllegalArgumentException exception) {
            throw new KeyStoreException(exception);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static void writeKeyData(File file, byte[] data) throws IOException {
        Logger.v(TAG, "Writing key data to a file");
        try (FileOutputStream out = new FileOutputStream(file);){
            ((OutputStream)out).write(data);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static byte[] readKeyData(File file) throws IOException {
        Logger.v(TAG, "Reading key data from a file");
        try (FileInputStream in = new FileInputStream(file);){
            int count;
            ByteArrayOutputStream bytes = new ByteArrayOutputStream();
            byte[] buffer = new byte[1024];
            while ((count = ((InputStream)in).read(buffer)) != -1) {
                bytes.write(buffer, 0, count);
            }
            byte[] byArray = bytes.toByteArray();
            return byArray;
        }
    }
}

