/*
 * Decompiled with CFR 0.152.
 */
package com.linecorp.centraldogma.server.storage.encryption;

import com.linecorp.centraldogma.internal.shaded.guava.collect.ImmutableList;
import com.linecorp.centraldogma.internal.shaded.guava.collect.ImmutableMap;
import com.linecorp.centraldogma.internal.shaded.guava.primitives.Ints;
import com.linecorp.centraldogma.server.internal.storage.AesGcmSivCipher;
import com.linecorp.centraldogma.server.internal.storage.repository.git.rocksdb.GitObjectMetadata;
import com.linecorp.centraldogma.server.storage.encryption.EncryptionStorageException;
import com.linecorp.centraldogma.server.storage.encryption.EncryptionStorageManager;
import com.linecorp.centraldogma.server.storage.encryption.KeyManagementService;
import com.linecorp.centraldogma.server.storage.encryption.SecretKeyWithVersion;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.ServiceLoader;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import javax.annotation.Nullable;
import javax.crypto.SecretKey;
import javax.crypto.spec.SecretKeySpec;
import org.rocksdb.AbstractImmutableNativeReference;
import org.rocksdb.BlockBasedTableConfig;
import org.rocksdb.BloomFilter;
import org.rocksdb.ColumnFamilyDescriptor;
import org.rocksdb.ColumnFamilyHandle;
import org.rocksdb.ColumnFamilyOptions;
import org.rocksdb.CompressionType;
import org.rocksdb.DBOptions;
import org.rocksdb.Filter;
import org.rocksdb.ReadOptions;
import org.rocksdb.RocksDB;
import org.rocksdb.RocksDBException;
import org.rocksdb.RocksIterator;
import org.rocksdb.RocksObject;
import org.rocksdb.Snapshot;
import org.rocksdb.TableFormatConfig;
import org.rocksdb.WriteBatch;
import org.rocksdb.WriteOptions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

final class DefaultEncryptionStorageManager
implements EncryptionStorageManager {
    private static final Logger logger = LoggerFactory.getLogger(DefaultEncryptionStorageManager.class);
    private static final String WDEK_COLUMN_FAMILY = "wdek";
    private static final String ENCRYPTION_METADATA_COLUMN_FAMILY = "encryption_metadata";
    private static final String ENCRYPTED_OBJECT_COLUMN_FAMILY = "encrypted_object";
    private static final String ENCRYPTED_OBJECT_ID_COLUMN_FAMILY = "encrypted_object_id";
    private static final List<String> ALL_COLUMN_FAMILY_NAMES = ImmutableList.of((Object)"default", (Object)"wdek", (Object)"encryption_metadata", (Object)"encrypted_object", (Object)"encrypted_object_id");
    private static final int BATCH_WRITE_SIZE = 1000;
    static final String ROCKSDB_PATH = "_rocks";
    private final KeyManagementService keyManagementService;
    private final RocksDB rocksDb;
    private final DBOptions dbOptions;
    private final Map<String, ColumnFamilyHandle> columnFamilyHandlesMap;
    private final BloomFilter bloomFilter;

    DefaultEncryptionStorageManager(String rocksDbPath) {
        ImmutableList keyManagementServices = ImmutableList.copyOf(ServiceLoader.load(KeyManagementService.class, EncryptionStorageManager.class.getClassLoader()));
        if (keyManagementServices.size() != 1) {
            throw new IllegalStateException("A single KeyManagementService implementation must be provided. found: " + keyManagementServices);
        }
        this.keyManagementService = (KeyManagementService)keyManagementServices.get(0);
        RocksDB.loadLibrary();
        this.bloomFilter = new BloomFilter();
        HashMap<String, ColumnFamilyOptions> cfNameToOptions = new HashMap<String, ColumnFamilyOptions>();
        for (String string : ALL_COLUMN_FAMILY_NAMES) {
            cfNameToOptions.put(string, this.createColumnFamilyOptions(ENCRYPTION_METADATA_COLUMN_FAMILY.equals(string)));
        }
        ArrayList<ColumnFamilyDescriptor> cfDescriptors = new ArrayList<ColumnFamilyDescriptor>();
        for (String cfName : ALL_COLUMN_FAMILY_NAMES) {
            cfDescriptors.add(new ColumnFamilyDescriptor(cfName.getBytes(StandardCharsets.UTF_8), (ColumnFamilyOptions)cfNameToOptions.get(cfName)));
        }
        this.dbOptions = new DBOptions().setCreateIfMissing(true).setCreateMissingColumnFamilies(true);
        ArrayList arrayList = new ArrayList();
        try {
            this.rocksDb = RocksDB.open((DBOptions)this.dbOptions, (String)rocksDbPath, cfDescriptors, arrayList);
        }
        catch (RocksDBException e) {
            cfNameToOptions.values().forEach(AbstractImmutableNativeReference::close);
            this.dbOptions.close();
            throw new EncryptionStorageException("Failed to open RocksDB with column families at " + rocksDbPath, e);
        }
        ImmutableMap.Builder handlesMapBuilder = ImmutableMap.builder();
        for (ColumnFamilyHandle handle : arrayList) {
            try {
                handlesMapBuilder.put((Object)new String(handle.getName(), StandardCharsets.UTF_8), (Object)handle);
            }
            catch (RocksDBException e) {
                arrayList.forEach(DefaultEncryptionStorageManager::closeSilently);
                DefaultEncryptionStorageManager.closeSilently((RocksObject)this.rocksDb);
                cfNameToOptions.values().forEach(AbstractImmutableNativeReference::close);
                this.dbOptions.close();
                throw new EncryptionStorageException("Failed to get name for a column family handle", e);
            }
        }
        this.columnFamilyHandlesMap = handlesMapBuilder.build();
        for (String cfName : ALL_COLUMN_FAMILY_NAMES) {
            if (this.columnFamilyHandlesMap.containsKey(cfName)) continue;
            this.close();
            throw new EncryptionStorageException("Column family handle not found for: " + cfName);
        }
    }

    private static void closeSilently(RocksObject obj) {
        try {
            obj.close();
        }
        catch (Exception e) {
            logger.warn("Failed to close RocksObject silently", (Throwable)e);
        }
    }

    private ColumnFamilyOptions createColumnFamilyOptions(boolean withBloomFilter) {
        ColumnFamilyOptions columnFamilyOptions = new ColumnFamilyOptions().setCompressionType(CompressionType.NO_COMPRESSION);
        if (!withBloomFilter) {
            return columnFamilyOptions;
        }
        return columnFamilyOptions.setTableFormatConfig((TableFormatConfig)new BlockBasedTableConfig().setFilterPolicy((Filter)this.bloomFilter));
    }

    @Override
    public boolean enabled() {
        return true;
    }

    @Override
    public CompletableFuture<byte[]> generateWdek() {
        byte[] dek = AesGcmSivCipher.generateAes256Key();
        return this.keyManagementService.wrap(dek);
    }

    @Override
    public SecretKey getDek(String projectName, String repoName, int version) {
        byte[] key;
        byte[] wdek;
        ColumnFamilyHandle wdekCf = this.columnFamilyHandlesMap.get(WDEK_COLUMN_FAMILY);
        try {
            wdek = this.rocksDb.get(wdekCf, DefaultEncryptionStorageManager.repoWdekDbKey(projectName, repoName, version));
        }
        catch (RocksDBException e) {
            throw new EncryptionStorageException("Failed to get WDEK of " + DefaultEncryptionStorageManager.projectRepoVersion(projectName, repoName, version), e);
        }
        if (wdek == null) {
            throw new EncryptionStorageException("WDEK of " + DefaultEncryptionStorageManager.projectRepoVersion(projectName, repoName, version) + " does not exist");
        }
        try {
            key = this.keyManagementService.unwrap(wdek).get(10L, TimeUnit.SECONDS);
        }
        catch (Throwable t) {
            throw new EncryptionStorageException("Failed to unwrap WDEK of " + DefaultEncryptionStorageManager.projectRepoVersion(projectName, repoName, version), t);
        }
        return AesGcmSivCipher.aesSecretKey(key);
    }

    @Override
    public SecretKeyWithVersion getCurrentDek(String projectName, String repoName) {
        int version;
        Objects.requireNonNull(projectName, "projectName");
        Objects.requireNonNull(repoName, "repoName");
        ColumnFamilyHandle wdekCf = this.columnFamilyHandlesMap.get(WDEK_COLUMN_FAMILY);
        try {
            byte[] versionBytes = this.rocksDb.get(wdekCf, DefaultEncryptionStorageManager.repoCurrentWdekDbKey(projectName, repoName));
            if (versionBytes == null) {
                throw new EncryptionStorageException("Current WDEK of " + DefaultEncryptionStorageManager.projectRepo(projectName, repoName) + " does not exist");
            }
            try {
                version = Ints.fromByteArray((byte[])versionBytes);
            }
            catch (IllegalArgumentException e) {
                throw new EncryptionStorageException("Failed to parse the current WDEK version of " + DefaultEncryptionStorageManager.projectRepo(projectName, repoName) + ". The version bytes: " + Arrays.toString(versionBytes), e);
            }
        }
        catch (RocksDBException e) {
            throw new EncryptionStorageException("Failed to get the current WDEK of " + DefaultEncryptionStorageManager.projectRepo(projectName, repoName), e);
        }
        return new SecretKeyWithVersion(this.getDek(projectName, repoName, version), version);
    }

    @Override
    public void storeWdek(String projectName, String repoName, byte[] wdek) {
        Objects.requireNonNull(projectName, "projectName");
        Objects.requireNonNull(repoName, "repoName");
        Objects.requireNonNull(wdek, WDEK_COLUMN_FAMILY);
        boolean version = true;
        byte[] wdekKeyBytes = DefaultEncryptionStorageManager.repoWdekDbKey(projectName, repoName, 1);
        ColumnFamilyHandle wdekCf = this.columnFamilyHandlesMap.get(WDEK_COLUMN_FAMILY);
        try {
            byte[] existingWdek = this.rocksDb.get(wdekCf, wdekKeyBytes);
            if (existingWdek != null) {
                throw new EncryptionStorageException("WDEK of " + DefaultEncryptionStorageManager.projectRepoVersion(projectName, repoName, 1) + " already exists");
            }
        }
        catch (RocksDBException e) {
            throw new EncryptionStorageException("Failed to check the existence of WDEK for " + DefaultEncryptionStorageManager.projectRepoVersion(projectName, repoName, 1), e);
        }
        try (WriteBatch writeBatch = new WriteBatch();
             WriteOptions writeOptions = new WriteOptions();){
            writeOptions.setSync(true);
            writeBatch.put(wdekCf, wdekKeyBytes, wdek);
            writeBatch.put(wdekCf, DefaultEncryptionStorageManager.repoCurrentWdekDbKey(projectName, repoName), Ints.toByteArray((int)1));
            this.rocksDb.write(writeOptions, writeBatch);
        }
        catch (RocksDBException e) {
            throw new EncryptionStorageException("Failed to store WDEK of " + DefaultEncryptionStorageManager.projectRepoVersion(projectName, repoName, 1), e);
        }
    }

    private static String projectRepo(String projectName, String repoName) {
        return projectName + '/' + repoName;
    }

    private static String projectRepoVersion(String projectName, String repoName, int version) {
        return projectName + '/' + repoName + '/' + version;
    }

    private static byte[] repoWdekDbKey(String projectName, String repoName, int version) {
        return ("wdeks/" + projectName + '/' + repoName + '/' + version).getBytes(StandardCharsets.UTF_8);
    }

    private static byte[] repoCurrentWdekDbKey(String projectName, String repoName) {
        return ("wdeks/" + projectName + '/' + repoName + "/current").getBytes(StandardCharsets.UTF_8);
    }

    @Override
    public void removeWdek(String projectName, String repoName) {
        Objects.requireNonNull(projectName, "projectName");
        Objects.requireNonNull(repoName, "repoName");
        boolean version = true;
        try (WriteBatch writeBatch = new WriteBatch();
             WriteOptions writeOptions = new WriteOptions();){
            writeOptions.setSync(true);
            ColumnFamilyHandle wdekCf = this.columnFamilyHandlesMap.get(WDEK_COLUMN_FAMILY);
            writeBatch.delete(wdekCf, DefaultEncryptionStorageManager.repoWdekDbKey(projectName, repoName, 1));
            writeBatch.delete(wdekCf, DefaultEncryptionStorageManager.repoCurrentWdekDbKey(projectName, repoName));
            this.rocksDb.write(writeOptions, writeBatch);
        }
        catch (RocksDBException e) {
            throw new EncryptionStorageException("Failed to remove WDEK of " + DefaultEncryptionStorageManager.projectRepoVersion(projectName, repoName, 1), e);
        }
    }

    @Override
    public byte[] getObject(byte[] key, byte[] metadataKey) {
        Objects.requireNonNull(key, "key");
        Objects.requireNonNull(metadataKey, "metadataKey");
        try {
            return this.rocksDb.get(this.columnFamilyHandlesMap.get(ENCRYPTED_OBJECT_COLUMN_FAMILY), key);
        }
        catch (RocksDBException e) {
            throw new EncryptionStorageException("Failed to get object. metadata key: " + new String(metadataKey, StandardCharsets.UTF_8), e);
        }
    }

    @Override
    public byte[] getObjectId(byte[] key, byte[] metadataKey) {
        Objects.requireNonNull(key, "key");
        Objects.requireNonNull(metadataKey, "metadataKey");
        try {
            return this.rocksDb.get(this.columnFamilyHandlesMap.get(ENCRYPTED_OBJECT_ID_COLUMN_FAMILY), key);
        }
        catch (RocksDBException e) {
            throw new EncryptionStorageException("Failed to get object ID. metadata key: " + new String(metadataKey, StandardCharsets.UTF_8), e);
        }
    }

    @Override
    public byte[] getMetadata(byte[] metadataKey) {
        Objects.requireNonNull(metadataKey, "metadataKey");
        try {
            return this.rocksDb.get(this.columnFamilyHandlesMap.get(ENCRYPTION_METADATA_COLUMN_FAMILY), metadataKey);
        }
        catch (RocksDBException e) {
            throw new EncryptionStorageException("Failed to get metadata. key: " + new String(metadataKey, StandardCharsets.UTF_8), e);
        }
    }

    @Override
    public void putObject(byte[] metadataKey, byte[] metadataValue, byte[] key, byte[] value) {
        try (WriteBatch writeBatch = new WriteBatch();
             WriteOptions writeOptions = new WriteOptions();){
            writeOptions.setSync(true);
            writeBatch.put(this.columnFamilyHandlesMap.get(ENCRYPTION_METADATA_COLUMN_FAMILY), metadataKey, metadataValue);
            writeBatch.put(this.columnFamilyHandlesMap.get(ENCRYPTED_OBJECT_COLUMN_FAMILY), key, value);
            this.rocksDb.write(writeOptions, writeBatch);
        }
        catch (RocksDBException e) {
            throw new EncryptionStorageException("Failed to write object key-value with metadata. metadata key: " + new String(metadataKey, StandardCharsets.UTF_8), e);
        }
    }

    @Override
    public void putObjectId(byte[] metadataKey, byte[] metadataValue, byte[] key, byte[] value, @Nullable byte[] previousKeyToRemove) {
        try (WriteBatch writeBatch = new WriteBatch();
             WriteOptions writeOptions = new WriteOptions();){
            writeOptions.setSync(true);
            writeBatch.put(this.columnFamilyHandlesMap.get(ENCRYPTION_METADATA_COLUMN_FAMILY), metadataKey, metadataValue);
            ColumnFamilyHandle objectIdcolumnFamilyHandle = this.columnFamilyHandlesMap.get(ENCRYPTED_OBJECT_ID_COLUMN_FAMILY);
            writeBatch.put(objectIdcolumnFamilyHandle, key, value);
            if (previousKeyToRemove != null) {
                writeBatch.delete(objectIdcolumnFamilyHandle, previousKeyToRemove);
            }
            this.rocksDb.write(writeOptions, writeBatch);
        }
        catch (RocksDBException e) {
            throw new EncryptionStorageException("Failed to write object key-value with metadata. metadata key: " + new String(metadataKey, StandardCharsets.UTF_8), e);
        }
    }

    @Override
    public boolean containsMetadata(byte[] key) {
        Objects.requireNonNull(key, "key");
        try {
            byte[] value = this.rocksDb.get(this.columnFamilyHandlesMap.get(ENCRYPTION_METADATA_COLUMN_FAMILY), key);
            return value != null;
        }
        catch (RocksDBException e) {
            throw new EncryptionStorageException("Failed to check existence of metadata. key: " + new String(key, StandardCharsets.UTF_8), e);
        }
    }

    @Override
    public void deleteObjectId(byte[] metadataKey, byte[] key) {
        Objects.requireNonNull(metadataKey, "metadataKey");
        Objects.requireNonNull(key, "key");
        try (WriteBatch writeBatch = new WriteBatch();
             WriteOptions writeOptions = new WriteOptions();){
            writeOptions.setSync(true);
            writeBatch.delete(this.columnFamilyHandlesMap.get(ENCRYPTION_METADATA_COLUMN_FAMILY), metadataKey);
            writeBatch.delete(this.columnFamilyHandlesMap.get(ENCRYPTED_OBJECT_ID_COLUMN_FAMILY), key);
            this.rocksDb.write(writeOptions, writeBatch);
        }
        catch (RocksDBException e) {
            throw new EncryptionStorageException("Failed to delete object ID key-value with metadata. metadata key: " + new String(metadataKey, StandardCharsets.UTF_8), e);
        }
    }

    @Override
    public void deleteRepositoryData(String projectName, String repoName) {
        Objects.requireNonNull(projectName, "projectName");
        Objects.requireNonNull(repoName, "repoName");
        SecretKey dek = this.getDek(projectName, repoName, 1);
        String projectRepoPrefix = DefaultEncryptionStorageManager.projectRepo(projectName, repoName) + '/';
        byte[] projectRepoPrefixBytes = projectRepoPrefix.getBytes(StandardCharsets.UTF_8);
        byte[] objectKeyPrefixBytes = (projectRepoPrefix + "objs/").getBytes(StandardCharsets.UTF_8);
        byte[] refsKeyPrefixBytes = (projectRepoPrefix + "refs/").getBytes(StandardCharsets.UTF_8);
        byte[] headKeyBytes = (projectRepoPrefix + "HEAD").getBytes(StandardCharsets.UTF_8);
        byte[] rev2ShaPrefixBytes = (projectRepoPrefix + "rev2sha/").getBytes(StandardCharsets.UTF_8);
        int totalDeletedCount = 0;
        int operationsInCurrentBatch = 0;
        ColumnFamilyHandle metadataColumnFamilyHandle = this.columnFamilyHandlesMap.get(ENCRYPTION_METADATA_COLUMN_FAMILY);
        try (WriteBatch writeBatch = new WriteBatch();
             WriteOptions writeOptions = new WriteOptions();
             RocksIterator iterator = this.rocksDb.newIterator(metadataColumnFamilyHandle);){
            byte[] metadataKey;
            iterator.seek(projectRepoPrefixBytes);
            while (iterator.isValid() && DefaultEncryptionStorageManager.startsWith(metadataKey = iterator.key(), projectRepoPrefixBytes)) {
                byte[] key;
                byte[] nonce;
                byte[] idPart;
                byte[] metadataValue = iterator.value();
                if (DefaultEncryptionStorageManager.startsWith(metadataKey, objectKeyPrefixBytes)) {
                    if (metadataKey.length == objectKeyPrefixBytes.length + 20) {
                        idPart = Arrays.copyOfRange(metadataKey, objectKeyPrefixBytes.length, metadataKey.length);
                        if (metadataValue != null) {
                            SecretKeySpec objectDek;
                            GitObjectMetadata gitObjectMetadata = GitObjectMetadata.fromBytes(metadataValue);
                            try {
                                objectDek = gitObjectMetadata.objectDek(dek);
                            }
                            catch (Exception e) {
                                throw new EncryptionStorageException("Failed to get object dek for " + new String(metadataKey, StandardCharsets.UTF_8), e);
                            }
                            byte[] key2 = AesGcmSivCipher.encrypt(objectDek, gitObjectMetadata.nonce(), idPart);
                            writeBatch.delete(this.columnFamilyHandlesMap.get(ENCRYPTED_OBJECT_COLUMN_FAMILY), key2);
                            ++operationsInCurrentBatch;
                            ++totalDeletedCount;
                        } else {
                            logger.warn("Invalid metadata value for object key: {}", (Object)new String(metadataKey, StandardCharsets.UTF_8));
                        }
                    } else {
                        logger.warn("Invalid object metadata key length: {}", (Object)new String(metadataKey, StandardCharsets.UTF_8));
                    }
                } else if (DefaultEncryptionStorageManager.startsWith(metadataKey, rev2ShaPrefixBytes)) {
                    if (metadataKey.length == rev2ShaPrefixBytes.length + 4) {
                        idPart = Arrays.copyOfRange(metadataKey, rev2ShaPrefixBytes.length, metadataKey.length);
                        if (metadataValue != null && metadataValue.length == 16) {
                            nonce = new byte[12];
                            System.arraycopy(metadataValue, 4, nonce, 0, 12);
                            key = AesGcmSivCipher.encrypt(dek, nonce, idPart);
                            writeBatch.delete(this.columnFamilyHandlesMap.get(ENCRYPTED_OBJECT_ID_COLUMN_FAMILY), key);
                            ++operationsInCurrentBatch;
                            ++totalDeletedCount;
                        } else {
                            logger.warn("Invalid nonce (metadata value) for rev2sha key: {}", (Object)new String(metadataKey, StandardCharsets.UTF_8));
                        }
                    } else {
                        logger.warn("Invalid rev2sha metadata key length: {}", (Object)new String(metadataKey, StandardCharsets.UTF_8));
                    }
                } else if (DefaultEncryptionStorageManager.startsWith(metadataKey, headKeyBytes) || DefaultEncryptionStorageManager.startsWith(metadataKey, refsKeyPrefixBytes)) {
                    idPart = Arrays.copyOfRange(metadataKey, projectRepoPrefixBytes.length, metadataKey.length);
                    if (metadataValue != null && metadataValue.length == 16) {
                        nonce = new byte[12];
                        System.arraycopy(metadataValue, 4, nonce, 0, 12);
                        key = AesGcmSivCipher.encrypt(dek, nonce, idPart);
                        writeBatch.delete(this.columnFamilyHandlesMap.get(ENCRYPTED_OBJECT_ID_COLUMN_FAMILY), key);
                        ++operationsInCurrentBatch;
                        ++totalDeletedCount;
                    } else {
                        logger.warn("Invalid nonce (metadata value) for ref key: {}", (Object)new String(metadataKey, StandardCharsets.UTF_8));
                    }
                } else {
                    logger.warn("Unknown metadata key pattern for prefix {}: {}", (Object)projectRepoPrefix, (Object)new String(metadataKey, StandardCharsets.UTF_8));
                }
                writeBatch.delete(metadataColumnFamilyHandle, metadataKey);
                ++totalDeletedCount;
                if (++operationsInCurrentBatch >= 1000 && writeBatch.count() > 0) {
                    writeOptions.setSync(true);
                    this.rocksDb.write(writeOptions, writeBatch);
                    logger.info("Deleted {} entries for repository {}/{}. Total entries processed for deletion so far: {}.", new Object[]{operationsInCurrentBatch, projectName, repoName, totalDeletedCount});
                    writeBatch.clear();
                    operationsInCurrentBatch = 0;
                }
                iterator.next();
            }
            if (operationsInCurrentBatch > 0) {
                writeOptions.setSync(true);
                this.rocksDb.write(writeOptions, writeBatch);
                logger.info("Deleted {} entries for repository {}/{}. Total entries processed for deletion so far: {}.", new Object[]{operationsInCurrentBatch, projectName, repoName, totalDeletedCount});
            }
            if (totalDeletedCount > 0) {
                logger.info("Successfully deleted a total of {} entries for repository {}/{}", new Object[]{totalDeletedCount, projectName, repoName});
            } else {
                logger.info("No data found for repository {}/{}", (Object)projectName, (Object)repoName);
            }
        }
        catch (EncryptionStorageException | RocksDBException e) {
            throw new EncryptionStorageException("Failed to delete repository data for " + DefaultEncryptionStorageManager.projectRepo(projectName, repoName), e);
        }
        catch (Exception e) {
            throw new EncryptionStorageException("Unexpected error during repository data deletion for " + DefaultEncryptionStorageManager.projectRepo(projectName, repoName), e);
        }
        this.removeWdek(projectName, repoName);
    }

    private static boolean startsWith(byte[] array, byte[] prefix) {
        if (array.length < prefix.length) {
            return false;
        }
        for (int i = 0; i < prefix.length; ++i) {
            if (array[i] == prefix[i]) continue;
            return false;
        }
        return true;
    }

    @Override
    public Map<String, Map<String, byte[]>> getAllData() {
        HashMap<String, Map<String, byte[]>> allData = new HashMap<String, Map<String, byte[]>>();
        try (Snapshot snapshot = this.rocksDb.getSnapshot();
             ReadOptions readOptions = new ReadOptions().setSnapshot(snapshot);){
            for (Map.Entry<String, ColumnFamilyHandle> entry : this.columnFamilyHandlesMap.entrySet()) {
                String cfName = entry.getKey();
                ColumnFamilyHandle cfHandle = entry.getValue();
                HashMap<String, byte[]> cfData = new HashMap<String, byte[]>();
                try (RocksIterator iterator = this.rocksDb.newIterator(cfHandle, readOptions);){
                    iterator.seekToFirst();
                    while (iterator.isValid()) {
                        String key = new String(iterator.key(), StandardCharsets.UTF_8);
                        cfData.put(key, iterator.value());
                        iterator.next();
                    }
                }
                allData.put(cfName, cfData);
            }
        }
        return allData;
    }

    public void close() {
        this.bloomFilter.close();
        for (Map.Entry<String, ColumnFamilyHandle> entry : this.columnFamilyHandlesMap.entrySet()) {
            try {
                entry.getValue().close();
            }
            catch (Throwable t) {
                logger.warn("Failed to close column family handle: {}", (Object)entry.getKey(), (Object)t);
            }
        }
        try {
            this.rocksDb.close();
        }
        catch (Throwable t) {
            logger.warn("Failed to close RocksDB", t);
        }
        try {
            this.dbOptions.close();
        }
        catch (Throwable t) {
            logger.warn("Failed to close DBOptions", t);
        }
    }
}

