/**
 * SPDX-FileCopyrightText: (c) 2000 Liferay, Inc. https://liferay.com
 * SPDX-License-Identifier: LGPL-2.1-or-later OR LicenseRef-Liferay-DXP-EULA-2.0.0-2023-06
 */

package com.liferay.source.formatter.check;

import com.liferay.petra.string.StringBundler;
import com.liferay.portal.kernel.io.unsync.UnsyncBufferedReader;
import com.liferay.portal.kernel.io.unsync.UnsyncStringReader;
import com.liferay.portal.kernel.util.StringUtil;

import java.io.IOException;

/**
 * @author Hugo Huijser
 */
public class FTLWhitespaceCheck extends WhitespaceCheck {

	@Override
	protected String doProcess(
			String fileName, String absolutePath, String content)
		throws IOException {

		content = StringUtil.replace(content, " >\n", ">\n");

		content = _formatWhitespace(fileName, absolutePath, content);

		content = StringUtil.replace(content, "\n\n\n", "\n\n");

		return content;
	}

	@Override
	protected String formatDoubleSpace(String line) {
		String trimmedLine = StringUtil.trim(line);

		if (trimmedLine.startsWith("*")) {
			return line;
		}

		return super.formatDoubleSpace(line);
	}

	private String _formatWhitespace(
			String fileName, String absolutePath, String content)
		throws IOException {

		StringBundler sb = new StringBundler();

		try (UnsyncBufferedReader unsyncBufferedReader =
				new UnsyncBufferedReader(new UnsyncStringReader(content))) {

			boolean assignBlock = false;
			String line = null;

			while ((line = unsyncBufferedReader.readLine()) != null) {
				line = trimLine(fileName, absolutePath, line);

				String trimmedLine = StringUtil.trimLeading(line);

				if (trimmedLine.startsWith("<#assign ") || assignBlock) {
					line = formatWhitespace(line, trimmedLine, true);

					line = formatIncorrectSyntax(line, "+[", "+ [", false);
					line = formatIncorrectSyntax(line, "=[", "= [", false);
					line = formatIncorrectSyntax(line, "=(", "= (", false);
					line = formatIncorrectSyntax(line, "=.", "= .", false);
					line = formatIncorrectSyntax(line, "=\"", "= \"", false);
				}

				if (line.endsWith(">")) {
					if (line.endsWith("/>")) {
						if (!trimmedLine.equals("/>") &&
							!line.endsWith(" />")) {

							line = StringUtil.replaceLast(line, "/>", " />");
						}

						if (assignBlock) {
							assignBlock = false;
						}
					}
					else if (line.endsWith(" >")) {
						line = StringUtil.replaceLast(line, " >", ">");
					}
				}

				if (trimmedLine.equals("<#assign")) {
					assignBlock = true;
				}

				sb.append(line);
				sb.append("\n");
			}
		}

		content = sb.toString();

		if (content.endsWith("\n") && !fileName.endsWith("_js.ftl")) {
			content = content.substring(0, content.length() - 1);
		}

		return content;
	}

}