/**
 * SPDX-FileCopyrightText: (c) 2026 Liferay, Inc. https://liferay.com
 * SPDX-License-Identifier: LGPL-2.1-or-later OR LicenseRef-Liferay-DXP-EULA-2.0.0-2023-06
 */

import Label from '@clayui/label';
import {AssigneeValue} from '@liferay/object-dynamic-data-mapping-form-field-type';
import React from 'react';

import {patchTaskById} from '../../utils/api';
import {
	displayAssignSuccessToast,
	displayStateSuccessToast,
} from '../../utils/toastUtil';
import CustomAssignee from '../CustomAssignee';
import InfoSummary from '../InfoSummary';
import StateSelector, {State} from '../StateSelector';

import '../AssigneeTrigger.scss';

import {DateRenderer} from '@liferay/frontend-data-set-web';

interface TaskInfoSummaryProps {
	assignTo: AssigneeValue;
	dueDate: string;
	initialState: string;
	states: State[];
	tags: string[];
	taskId: string;
	title: string;
}

export default function TaskInfoSummary({
	assignTo,
	dueDate,
	initialState,
	states,
	tags,
	taskId,
	title,
}: TaskInfoSummaryProps) {
	return (
		<InfoSummary
			defaultOpen={true}
			items={[
				{
					label: 'State',
					value: (
						<StateSelector
							initialSelectedKey={initialState}
							onChange={async (key: string) => {
								const response = await patchTaskById({
									body: {state: key},
									taskId,
								});

								if (response.ok) {
									displayStateSuccessToast();
								}
							}}
							states={states}
						/>
					),
				},
				{
					label: 'Assignee',
					value: (
						<CustomAssignee
							onChange={async (value: AssigneeValue | {}) => {
								const response = await patchTaskById({
									body: {assignTo: value},
									taskId,
								});

								if (response.ok) {
									displayAssignSuccessToast(
										title,
										(value as AssigneeValue).name
									);
								}
							}}
							showLabel={false}
							value={assignTo}
						/>
					),
				},
				{
					label: 'Due Date',
					value: DateRenderer({value: dueDate}) ?? '',
				},
				{
					label: 'Tags',
					value: (
						<div>
							{tags.map((tag) => (
								<Label key={tag}>{tag}</Label>
							))}
						</div>
					),
				},
			]}
		/>
	);
}
