/**
 * SPDX-FileCopyrightText: (c) 2000 Liferay, Inc. https://liferay.com
 * SPDX-License-Identifier: LGPL-2.1-or-later OR LicenseRef-Liferay-DXP-EULA-2.0.0-2023-06
 */

package com.liferay.portal.workflow.metrics.internal.sla.processor;

import com.liferay.petra.string.CharPool;
import com.liferay.petra.string.StringPool;
import com.liferay.petra.string.StringUtil;
import com.liferay.portal.kernel.search.BooleanClauseOccur;
import com.liferay.portal.kernel.search.Field;
import com.liferay.portal.kernel.search.filter.BooleanFilter;
import com.liferay.portal.kernel.search.filter.ExistsFilter;
import com.liferay.portal.kernel.search.generic.BooleanQueryImpl;
import com.liferay.portal.kernel.util.GetterUtil;
import com.liferay.portal.kernel.util.ListUtil;
import com.liferay.portal.kernel.util.PropsKeys;
import com.liferay.portal.kernel.util.PropsUtil;
import com.liferay.portal.kernel.util.Time;
import com.liferay.portal.kernel.util.Validator;
import com.liferay.portal.search.document.Document;
import com.liferay.portal.search.engine.adapter.search.SearchRequestExecutor;
import com.liferay.portal.search.engine.adapter.search.SearchSearchRequest;
import com.liferay.portal.search.engine.adapter.search.SearchSearchResponse;
import com.liferay.portal.search.filter.DateRangeFilter;
import com.liferay.portal.search.filter.DateRangeFilterBuilder;
import com.liferay.portal.search.filter.FilterBuilders;
import com.liferay.portal.search.hits.SearchHit;
import com.liferay.portal.search.hits.SearchHits;
import com.liferay.portal.search.sort.SortOrder;
import com.liferay.portal.search.sort.Sorts;
import com.liferay.portal.workflow.metrics.model.WorkflowMetricsSLADefinitionVersion;
import com.liferay.portal.workflow.metrics.search.index.name.WorkflowMetricsIndexNameBuilder;
import com.liferay.portal.workflow.metrics.sla.calendar.WorkflowMetricsSLACalendar;
import com.liferay.portal.workflow.metrics.sla.calendar.WorkflowMetricsSLACalendarTracker;
import com.liferay.portal.workflow.metrics.sla.processor.WorkfowMetricsSLAStatus;

import java.time.Duration;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;

import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Stack;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;

/**
 * @author Rafael Praxedes
 */
@Component(immediate = true, service = WorkflowMetricsSLAProcessor.class)
public class WorkflowMetricsSLAProcessor {

	public Optional<WorkflowMetricsSLAProcessResult> process(
		long companyId, LocalDateTime createLocalDateTime, long instanceId,
		LocalDateTime nowLocalDateTime, long startNodeId,
		WorkflowMetricsSLADefinitionVersion
			workflowMetricsSLADefinitionVersion) {

		WorkflowMetricsSLAProcessResult lastWorkflowMetricsSLAProcessResult =
			fetchLastWorkflowMetricsSLAProcessResult(
				workflowMetricsSLADefinitionVersion, instanceId);

		long elapsedTime = 0;
		LocalDateTime lastCheckLocalDateTime = null;
		WorkfowMetricsSLAStatus workfowMetricsSLAStatus =
			WorkfowMetricsSLAStatus.NEW;

		if (lastWorkflowMetricsSLAProcessResult != null) {
			elapsedTime = lastWorkflowMetricsSLAProcessResult.getElapsedTime();

			lastCheckLocalDateTime =
				lastWorkflowMetricsSLAProcessResult.getLastCheckLocalDateTime();
			workfowMetricsSLAStatus =
				lastWorkflowMetricsSLAProcessResult.
					getWorkfowMetricsSLAStatus();

			if (lastCheckLocalDateTime.isAfter(nowLocalDateTime) ||
				Objects.equals(
					workfowMetricsSLAStatus,
					WorkfowMetricsSLAStatus.COMPLETED)) {

				return Optional.empty();
			}
		}

		List<Document> documents = getDocuments(
			companyId, instanceId, lastCheckLocalDateTime);
		WorkflowMetricsSLACalendar workflowMetricsSLACalendar =
			_workflowMetricsSLACalendarTracker.getWorkflowMetricsSLACalendar(
				workflowMetricsSLADefinitionVersion.getCalendarKey());
		WorkflowMetricsSLAStopwatch workflowMetricsSLAStopwatch =
			_createWorkflowMetricsSLAStopwatch(
				documents, createLocalDateTime, lastCheckLocalDateTime,
				startNodeId, workflowMetricsSLADefinitionVersion,
				workfowMetricsSLAStatus);

		if (!workflowMetricsSLAStopwatch.isEmpty()) {
			List<TaskInterval> taskIntervals = _toTaskIntervals(
				documents, lastCheckLocalDateTime, nowLocalDateTime);

			for (TaskInterval taskInterval : taskIntervals) {
				elapsedTime += _computeElapsedTime(
					taskInterval.getEndLocalDateTime(),
					taskInterval.getStartLocalDateTime(),
					workflowMetricsSLACalendar, workflowMetricsSLAStopwatch);
			}

			workfowMetricsSLAStatus =
				workflowMetricsSLAStopwatch.getWorkfowMetricsSLAStatus();
		}

		return Optional.of(
			_createWorkflowMetricsSLAProcessResult(
				companyId, documents, elapsedTime, instanceId, nowLocalDateTime,
				workflowMetricsSLACalendar, workflowMetricsSLADefinitionVersion,
				workfowMetricsSLAStatus));
	}

	protected WorkflowMetricsSLAProcessResult
		fetchLastWorkflowMetricsSLAProcessResult(
			WorkflowMetricsSLADefinitionVersion
				workflowMetricsSLADefinitionVersion,
			long instanceId) {

		SearchSearchRequest searchSearchRequest = new SearchSearchRequest();

		searchSearchRequest.setIndexNames(
			_slaProcessResultWorkflowMetricsIndexNameBuilder.getIndexName(
				workflowMetricsSLADefinitionVersion.getCompanyId()));
		searchSearchRequest.setQuery(
			new BooleanQueryImpl() {
				{
					setPreBooleanFilter(
						new BooleanFilter() {
							{
								addRequiredTerm(
									"companyId",
									workflowMetricsSLADefinitionVersion.
										getCompanyId());
								addRequiredTerm("deleted", false);
								addRequiredTerm("instanceId", instanceId);
								addRequiredTerm(
									"slaDefinitionId",
									workflowMetricsSLADefinitionVersion.
										getWorkflowMetricsSLADefinitionId());
							}
						});
				}
			});

		return Stream.of(
			_searchRequestExecutor.executeSearchRequest(searchSearchRequest)
		).map(
			SearchSearchResponse::getSearchHits
		).map(
			SearchHits::getSearchHits
		).flatMap(
			List::parallelStream
		).map(
			SearchHit::getDocument
		).findFirst(
		).map(
			document -> new WorkflowMetricsSLAProcessResult() {
				{
					setCompanyId(
						workflowMetricsSLADefinitionVersion.getCompanyId());
					setElapsedTime(document.getLong("elapsedTime"));
					setInstanceId(instanceId);
					setLastCheckLocalDateTime(
						LocalDateTime.parse(
							document.getString("lastCheckDate"),
							_dateTimeFormatter));
					setOnTime(
						GetterUtil.getBoolean(document.getValue("onTime")));
					setOverdueLocalDateTime(
						LocalDateTime.parse(
							document.getString("overdueDate"),
							_dateTimeFormatter));
					setProcessId(
						workflowMetricsSLADefinitionVersion.getProcessId());
					setRemainingTime(document.getLong("remainingTime"));
					setSLADefinitionId(
						workflowMetricsSLADefinitionVersion.
							getWorkflowMetricsSLADefinitionId());
					setWorkfowMetricsSLAStatus(
						WorkfowMetricsSLAStatus.valueOf(
							document.getString("status")));
				}
			}
		).orElseGet(
			() -> null
		);
	}

	protected List<Document> getDocuments(
		long companyId, long instanceId, LocalDateTime lastCheckLocalDateTime) {

		SearchSearchRequest searchSearchRequest = new SearchSearchRequest();

		searchSearchRequest.addSorts(
			_sorts.field(
				Field.getSortableFieldName("createDate_Number"),
				SortOrder.ASC));
		searchSearchRequest.setIndexNames(
			_tokenWorkflowMetricsIndexNameBuilder.getIndexName(companyId));
		searchSearchRequest.setQuery(
			new BooleanQueryImpl() {
				{
					setPreBooleanFilter(
						new BooleanFilter() {
							{
								addRequiredTerm("companyId", companyId);
								addRequiredTerm("deleted", false);
								addRequiredTerm("instanceId", instanceId);

								if (lastCheckLocalDateTime != null) {
									add(
										_createCompletionDateRangeFilter(
											lastCheckLocalDateTime),
										BooleanClauseOccur.SHOULD);
									add(
										new BooleanFilter() {
											{
												add(
													new ExistsFilter(
														"completionDate"),
													BooleanClauseOccur.
														MUST_NOT);
											}
										},
										BooleanClauseOccur.SHOULD);
								}
							}
						});
				}
			});
		searchSearchRequest.setSize(10000);

		return Stream.of(
			_searchRequestExecutor.executeSearchRequest(searchSearchRequest)
		).map(
			SearchSearchResponse::getSearchHits
		).map(
			SearchHits::getSearchHits
		).flatMap(
			List::parallelStream
		).map(
			SearchHit::getDocument
		).collect(
			Collectors.toList()
		);
	}

	protected boolean isBreached(
		Document document, LocalDateTime nowLocalDateTime,
		LocalDateTime overdueLocalDateTime) {

		if (nowLocalDateTime.isBefore(overdueLocalDateTime)) {
			return false;
		}

		LocalDateTime createDateLocalDateTime = LocalDateTime.parse(
			document.getString("createDate"), _dateTimeFormatter);

		if (createDateLocalDateTime.isAfter(overdueLocalDateTime)) {
			return false;
		}

		if (Validator.isNull(document.getValue("completionDate"))) {
			return true;
		}

		LocalDateTime completionDateLocalDateTime = LocalDateTime.parse(
			document.getString("completionDate"), _dateTimeFormatter);

		if (completionDateLocalDateTime.isAfter(overdueLocalDateTime)) {
			return true;
		}

		return false;
	}

	protected boolean isOnTime(
		Document document, LocalDateTime nowLocalDateTime,
		LocalDateTime overdueLocalDateTime) {

		if (nowLocalDateTime.isBefore(overdueLocalDateTime)) {
			return true;
		}

		LocalDateTime createDateLocalDateTime = LocalDateTime.parse(
			document.getString("createDate"), _dateTimeFormatter);

		if (createDateLocalDateTime.isAfter(overdueLocalDateTime)) {
			return false;
		}

		if (Validator.isNull(document.getValue("completionDate"))) {
			return false;
		}

		LocalDateTime completionDateLocalDateTime = LocalDateTime.parse(
			document.getString("completionDate"), _dateTimeFormatter);

		if (completionDateLocalDateTime.isBefore(overdueLocalDateTime) ||
			Objects.equals(completionDateLocalDateTime, overdueLocalDateTime)) {

			return true;
		}

		return false;
	}

	protected static class TaskInterval {

		public LocalDateTime getEndLocalDateTime() {
			return _endLocalDateTime;
		}

		public LocalDateTime getStartLocalDateTime() {
			return _startLocalDateTime;
		}

		public void setEndLocalDateTime(LocalDateTime endLocalDateTime) {
			_endLocalDateTime = endLocalDateTime;
		}

		public void setStartLocalDateTime(LocalDateTime startLocalDateTime) {
			_startLocalDateTime = startLocalDateTime;
		}

		private LocalDateTime _endLocalDateTime;
		private LocalDateTime _startLocalDateTime;

	}

	private long _computeElapsedTime(
		LocalDateTime endLocalDateTime, LocalDateTime starLocalDateTime,
		WorkflowMetricsSLACalendar workflowMetricsSLACalendar,
		WorkflowMetricsSLAStopwatch workflowMetricsSLAStopwatch) {

		long elapsedTime = 0;

		for (TaskInterval taskInterval :
				workflowMetricsSLAStopwatch.getTaskIntervals()) {

			if (endLocalDateTime.isBefore(taskInterval._startLocalDateTime) ||
				starLocalDateTime.isAfter(taskInterval._endLocalDateTime)) {

				continue;
			}

			Duration duration = workflowMetricsSLACalendar.getDuration(
				_getMaxLocalDateTime(
					starLocalDateTime, taskInterval._startLocalDateTime),
				_getMinLocalDateTime(
					endLocalDateTime, taskInterval._endLocalDateTime));

			elapsedTime += duration.toMillis();
		}

		return elapsedTime;
	}

	private DateRangeFilter _createCompletionDateRangeFilter(
		LocalDateTime lastCheckLocalDateTime) {

		DateRangeFilterBuilder completionDateRangeFilterBuilder =
			_filterBuilders.dateRangeFilterBuilder();

		completionDateRangeFilterBuilder.setFieldName("completionDate");
		completionDateRangeFilterBuilder.setFrom(
			_dateTimeFormatter.format(lastCheckLocalDateTime));

		return completionDateRangeFilterBuilder.build();
	}

	private WorkflowMetricsSLAProcessResult
		_createWorkflowMetricsSLAProcessResult(
			long companyId, List<Document> documents, long elapsedTime,
			long instanceId, LocalDateTime nowLocalDateTime,
			WorkflowMetricsSLACalendar workflowMetricsSLACalendar,
			WorkflowMetricsSLADefinitionVersion
				workflowMetricsSLADefinitionVersion,
			WorkfowMetricsSLAStatus workfowMetricsSLAStatus) {

		WorkflowMetricsSLAProcessResult workflowMetricsSLAProcessResult =
			new WorkflowMetricsSLAProcessResult() {
				{
					setCompanyId(companyId);
					setElapsedTime(elapsedTime);
					setInstanceId(instanceId);
					setLastCheckLocalDateTime(nowLocalDateTime);
					setOnTime(
						elapsedTime <=
							workflowMetricsSLADefinitionVersion.getDuration());

					long remainingTime = _getRemainingTime(
						elapsedTime, workflowMetricsSLADefinitionVersion);

					setOverdueLocalDateTime(
						workflowMetricsSLACalendar.getOverdueLocalDateTime(
							nowLocalDateTime,
							Duration.ofMillis(remainingTime)));

					setProcessId(
						workflowMetricsSLADefinitionVersion.getProcessId());
					setRemainingTime(remainingTime);
					setSLADefinitionId(
						workflowMetricsSLADefinitionVersion.
							getWorkflowMetricsSLADefinitionId());
					setWorkfowMetricsSLAStatus(workfowMetricsSLAStatus);
				}
			};

		workflowMetricsSLAProcessResult.setWorkflowMetricsSLATaskResults(
			_createWorkflowMetricsSLATaskResults(
				documents, nowLocalDateTime, workflowMetricsSLAProcessResult));

		return workflowMetricsSLAProcessResult;
	}

	private WorkflowMetricsSLAStopwatch _createWorkflowMetricsSLAStopwatch(
		List<Document> documents, LocalDateTime createDateLocalDateTime,
		LocalDateTime lastCheckLocalDateTime, long startNodeId,
		WorkflowMetricsSLADefinitionVersion workflowMetricsSLADefinitionVersion,
		WorkfowMetricsSLAStatus workfowMetricsSLAStatus) {

		WorkflowMetricsSLAStopwatch workflowMetricsSLAStopwatch =
			new WorkflowMetricsSLAStopwatch(workfowMetricsSLAStatus);

		Map<Long, String> startTimeMarkers = _getTimeMarkers(
			StringUtil.split(
				workflowMetricsSLADefinitionVersion.getStartNodeKeys()));

		if (Objects.equals(
				workfowMetricsSLAStatus, WorkfowMetricsSLAStatus.NEW) &&
			startTimeMarkers.containsKey(startNodeId)) {

			workflowMetricsSLAStopwatch.run(createDateLocalDateTime);
		}
		else if (Objects.equals(
					workfowMetricsSLAStatus, WorkfowMetricsSLAStatus.RUNNING)) {

			workflowMetricsSLAStopwatch.run(lastCheckLocalDateTime);
		}

		Map<Long, String> pauseTimeMarkers = _getTimeMarkers(
			StringUtil.split(
				workflowMetricsSLADefinitionVersion.getPauseNodeKeys()));
		Map<Long, String> stopTimeMarkers = _getTimeMarkers(
			StringUtil.split(
				workflowMetricsSLADefinitionVersion.getStopNodeKeys()));

		Iterator<Document> iterator = documents.iterator();

		while (iterator.hasNext() && !workflowMetricsSLAStopwatch.isStopped()) {
			Document document = iterator.next();

			long taskId = document.getLong("taskId");

			TaskInterval taskInterval = _toTaskInterval(
				document, lastCheckLocalDateTime, null);

			if (pauseTimeMarkers.containsKey(taskId) &&
				!stopTimeMarkers.containsKey(taskId)) {

				workflowMetricsSLAStopwatch.pause(
					taskInterval._startLocalDateTime);

				if (taskInterval._endLocalDateTime != null) {
					workflowMetricsSLAStopwatch.run(
						taskInterval._endLocalDateTime);
				}
			}

			if (startTimeMarkers.containsKey(taskId)) {
				if (Objects.equals(startTimeMarkers.get(taskId), "enter")) {
					workflowMetricsSLAStopwatch.run(
						taskInterval._startLocalDateTime);
				}
				else if (Objects.equals(
							startTimeMarkers.get(taskId), "leave") &&
						 (taskInterval._endLocalDateTime != null)) {

					workflowMetricsSLAStopwatch.run(
						taskInterval._endLocalDateTime);
				}
			}

			if (stopTimeMarkers.containsKey(taskId)) {
				if (Objects.equals(stopTimeMarkers.get(taskId), "enter")) {
					workflowMetricsSLAStopwatch.stop(
						taskInterval._startLocalDateTime);
				}
				else if (Objects.equals(stopTimeMarkers.get(taskId), "leave") &&
						 (taskInterval._endLocalDateTime != null)) {

					workflowMetricsSLAStopwatch.stop(
						taskInterval._endLocalDateTime);
				}
			}
		}

		return workflowMetricsSLAStopwatch;
	}

	private WorkflowMetricsSLATaskResult _createWorkflowMetricsSLATaskResult(
		Document document, LocalDateTime nowLocalDateTime,
		WorkflowMetricsSLAProcessResult workflowMetricsSLAProcessResult) {

		return new WorkflowMetricsSLATaskResult() {
			{
				setBreached(
					WorkflowMetricsSLAProcessor.this.isBreached(
						document, nowLocalDateTime,
						workflowMetricsSLAProcessResult.
							getOverdueLocalDateTime()));
				setCompanyId(workflowMetricsSLAProcessResult.getCompanyId());
				setInstanceId(workflowMetricsSLAProcessResult.getInstanceId());
				setLastCheckLocalDateTime(
					workflowMetricsSLAProcessResult.
						getLastCheckLocalDateTime());
				setOnTime(
					WorkflowMetricsSLAProcessor.this.isOnTime(
						document, nowLocalDateTime,
						workflowMetricsSLAProcessResult.
							getOverdueLocalDateTime()));
				setProcessId(workflowMetricsSLAProcessResult.getProcessId());
				setSLADefinitionId(
					workflowMetricsSLAProcessResult.getSLADefinitionId());
				setTaskId(document.getLong("taskId"));
				setTaskName(document.getString("taskName"));
				setTokenId(document.getLong("tokenId"));
				setWorkfowMetricsSLAStatus(
					_getWorkfowMetricsSLAStatus(
						document, workflowMetricsSLAProcessResult));
			}
		};
	}

	private List<WorkflowMetricsSLATaskResult>
		_createWorkflowMetricsSLATaskResults(
			List<Document> documents, LocalDateTime nowLocalDateTime,
			WorkflowMetricsSLAProcessResult workflowMetricsSLAProcessResult) {

		Stream<Document> stream = documents.stream();

		return stream.map(
			document -> _createWorkflowMetricsSLATaskResult(
				document, nowLocalDateTime, workflowMetricsSLAProcessResult)
		).collect(
			Collectors.toList()
		);
	}

	private LocalDateTime _getMaxLocalDateTime(
		LocalDateTime localDateTime1, LocalDateTime localDateTime2) {

		if (localDateTime1.isAfter(localDateTime2)) {
			return localDateTime1;
		}

		return localDateTime2;
	}

	private LocalDateTime _getMinLocalDateTime(
		LocalDateTime localDateTime1, LocalDateTime localDateTime2) {

		if (localDateTime1.isBefore(localDateTime2)) {
			return localDateTime1;
		}

		return localDateTime2;
	}

	private long _getRemainingTime(
		long elapsedTime,
		WorkflowMetricsSLADefinitionVersion
			workflowMetricsSLADefinitionVersion) {

		return Math.min(
			9999 * Time.DAY,
			workflowMetricsSLADefinitionVersion.getDuration() - elapsedTime);
	}

	private Map<Long, String> _getTimeMarkers(List<String> nodeKeys) {
		Map<Long, String> map = new HashMap<>();

		for (String nodeKey : nodeKeys) {
			List<String> parts = StringUtil.split(nodeKey, CharPool.COLON);

			long nodeId = GetterUtil.getLong(parts.get(0));

			if (parts.size() == 1) {
				map.put(nodeId, StringPool.BLANK);
			}
			else {
				map.put(nodeId, parts.get(1));
			}
		}

		return map;
	}

	private WorkfowMetricsSLAStatus _getWorkfowMetricsSLAStatus(
		Document document,
		WorkflowMetricsSLAProcessResult workflowMetricsSLAProcessResult) {

		if (Objects.equals(
				workflowMetricsSLAProcessResult.getWorkfowMetricsSLAStatus(),
				WorkfowMetricsSLAStatus.NEW)) {

			return WorkfowMetricsSLAStatus.NEW;
		}

		if (GetterUtil.getBoolean(document.getBoolean("completed"))) {
			return WorkfowMetricsSLAStatus.COMPLETED;
		}

		return WorkfowMetricsSLAStatus.RUNNING;
	}

	private TaskInterval _toTaskInterval(
		Document document, LocalDateTime lastCheckLocalDateTime,
		LocalDateTime nowLocalDateTime) {

		TaskInterval taskInterval = new TaskInterval();

		if (Validator.isNull(document.getValue("completionDate"))) {
			taskInterval._endLocalDateTime = nowLocalDateTime;
		}
		else {
			taskInterval._endLocalDateTime = LocalDateTime.parse(
				document.getString("completionDate"), _dateTimeFormatter);
		}

		LocalDateTime createDateLocalDateTime = LocalDateTime.parse(
			document.getString("createDate"), _dateTimeFormatter);

		if ((lastCheckLocalDateTime != null) &&
			lastCheckLocalDateTime.isAfter(createDateLocalDateTime)) {

			taskInterval._startLocalDateTime = lastCheckLocalDateTime;
		}
		else {
			taskInterval._startLocalDateTime = createDateLocalDateTime;
		}

		return taskInterval;
	}

	private List<TaskInterval> _toTaskIntervals(
		List<Document> documents, LocalDateTime lastCheckLocalDateTime,
		LocalDateTime nowLocalDateTime) {

		if (ListUtil.isEmpty(documents)) {
			return Collections.emptyList();
		}

		Stack<TaskInterval> taskIntervals = new Stack<>();

		taskIntervals.push(
			_toTaskInterval(
				documents.get(0), lastCheckLocalDateTime, nowLocalDateTime));

		for (Document document : documents) {
			TaskInterval topTaskInterval = taskIntervals.peek();

			TaskInterval taskInterval = _toTaskInterval(
				document, lastCheckLocalDateTime, nowLocalDateTime);

			LocalDateTime topEndLocalDateTime =
				topTaskInterval.getEndLocalDateTime();

			if (topEndLocalDateTime.isBefore(
					taskInterval.getStartLocalDateTime())) {

				taskIntervals.push(taskInterval);
			}
			else if (topEndLocalDateTime.isBefore(
						taskInterval.getEndLocalDateTime())) {

				topTaskInterval._endLocalDateTime =
					taskInterval.getEndLocalDateTime();
			}
		}

		return taskIntervals;
	}

	private final DateTimeFormatter _dateTimeFormatter =
		DateTimeFormatter.ofPattern(
			PropsUtil.get(PropsKeys.INDEX_DATE_FORMAT_PATTERN));

	@Reference
	private FilterBuilders _filterBuilders;

	@Reference
	private SearchRequestExecutor _searchRequestExecutor;

	@Reference(
		target = "(workflow.metrics.index.entity.name=sla-process-result)"
	)
	private WorkflowMetricsIndexNameBuilder
		_slaProcessResultWorkflowMetricsIndexNameBuilder;

	@Reference
	private Sorts _sorts;

	@Reference(target = "(workflow.metrics.index.entity.name=token)")
	private WorkflowMetricsIndexNameBuilder
		_tokenWorkflowMetricsIndexNameBuilder;

	@Reference
	private WorkflowMetricsSLACalendarTracker
		_workflowMetricsSLACalendarTracker;

}