/**
 * SPDX-FileCopyrightText: (c) 2000 Liferay, Inc. https://liferay.com
 * SPDX-License-Identifier: LGPL-2.1-or-later OR LicenseRef-Liferay-DXP-EULA-2.0.0-2023-06
 */

import {FormSupport} from 'dynamic-data-mapping-form-renderer';
import dom from 'metal-dom';
import {DragDrop} from 'metal-drag-drop';
import Component from 'metal-jsx';

import {
	disableFieldDropTargets,
	disableFieldSetDragSources,
	disableFieldSetDropTargets,
} from '../../util/dragAndDrop.es';
import formBuilderProps from './props.es';

const withMoveableFields = (ChildComponent) => {
	class MoveableFields extends Component {
		createDragAndDrop() {
			this._dragAndDrop = new DragDrop({
				container: this.element,
				sources: '.moveable .ddm-drag:not([data-drag-disabled="true"])',
				targets:
					'.moveable .ddm-target:not([data-drop-disabled="true"])',
				useShim: false,
			});

			this._dragAndDrop.on(
				DragDrop.Events.END,
				this._handleDragAndDropEnd.bind(this)
			);

			this._dragAndDrop.on(
				DragDrop.Events.DRAG,
				this._handleDragStarted.bind(this)
			);
		}

		disposeDragAndDrop() {
			if (this._dragAndDrop) {
				this._dragAndDrop.dispose();
			}
		}

		disposeInternal() {
			super.disposeInternal();

			this.disposeDragAndDrop();
		}

		render() {
			return (
				<div class="moveable">
					<ChildComponent {...this.props} />
				</div>
			);
		}

		rendered() {
			const {allowNestedFields = true, pages} = this.props;

			disableFieldSetDragSources(this.element, pages);
			disableFieldSetDropTargets(this.element, pages);

			if (!allowNestedFields) {
				disableFieldDropTargets(this.element, pages);
			}

			if (!document.querySelector('.dragging')) {
				this._refreshDragAndDrop();
			}
		}

		_getClosestParent(node) {
			return dom.closest(node.parentElement, `.ddm-field-container`);
		}

		_handleDragAndDropEnd({source, target}) {
			const lastParent = document.querySelector('.dragging');

			if (lastParent) {
				lastParent.classList.remove('dragging');
				lastParent.removeAttribute('style');
			}

			if (!target) {
				target = document.querySelector(
					'.ddm-form-builder .ddm-target.targetOver'
				);
			}

			if (target) {
				const sourceFieldNode = dom.closest(
					source,
					'.ddm-field-container'
				);

				const sourceFieldPage = parseInt(
					dom.closest(source, '[data-ddm-page]').dataset.ddmPage,
					10
				);

				let targetFieldName;

				if (target.classList.contains('ddm-field-container')) {
					targetFieldName = target.dataset.fieldName;
				}

				const sourceFieldName = sourceFieldNode.dataset.fieldName;

				if (sourceFieldName === targetFieldName) {
					return;
				}

				let targetParentFieldName;
				const targetParentFieldNode = this._getClosestParent(target);

				if (targetParentFieldNode) {
					targetParentFieldName =
						targetParentFieldNode.dataset.fieldName;
				}

				this._handleFieldMoved({
					sourceFieldName,
					sourceFieldPage,
					targetFieldName,
					targetIndexes: FormSupport.getIndexes(target.parentElement),
					targetParentFieldName,
				});
			}
		}

		_handleDragStarted({source}) {
			const {height} = source.getBoundingClientRect();
			const {parentElement} = source;

			parentElement.setAttribute(
				'style',
				`height: ${height}px !important;`
			);
			parentElement.classList.add('dragging');
		}

		_handleFieldMoved(event) {
			const {dispatch} = this.context;

			dispatch('fieldMoved', event);
		}

		_refreshDragAndDrop() {
			this.disposeDragAndDrop();
			this.createDragAndDrop();
		}
	}

	MoveableFields.PROPS = {
		...formBuilderProps,
	};

	return MoveableFields;
};

export default withMoveableFields;
