package com.kp.md.picker.common;

import android.content.Context;
import android.content.res.Configuration;
import android.content.res.Resources;
import android.content.res.TypedArray;
import android.graphics.Color;
import android.graphics.PorterDuff;
import android.support.annotation.NonNull;
import android.support.v4.content.ContextCompat;
import android.util.TypedValue;
import android.view.View;
import android.widget.Button;
import android.widget.ImageView;

import com.kp.md.picker.MPicker;
import com.kp.md.picker.R;
import com.kp.md.picker.helpers.MOptions;
import com.kp.md.picker.utilities.MUtils;


/**
 * Created by Admin on 15/02/2016.
 */
public class ButtonHandler implements View.OnClickListener {

    private boolean mIsInLandscapeMode;

    private ButtonLayout mPortraitButtonHandler;

    // Can be 'android.widget.Button' or 'android.widget.ImageView'
    View mPositiveButtonDP, mPositiveButtonTP, mNegativeButtonDP, mNegativeButtonTP;
    // 'Button' used for switching between 'MDatePicker'
    // and 'MTimePicker'. Also displays the currently
    // selected date/time depending on the visible picker
    Button mSwitcherButtonDP, mSwitcherButtonTP;

    Callback mCallback;

    int mIconOverlayColor /* color used with the applied 'ColorFilter' */,
            mDisabledAlpha /* android.R.attr.disabledAlpha * 255 */,
            mButtonBarBgColor;

    public ButtonHandler(@NonNull MPicker mPicker) {
        Context context = mPicker.getContext();

        mIsInLandscapeMode = context.getResources().getConfiguration().orientation == Configuration.ORIENTATION_LANDSCAPE;

        if (mIsInLandscapeMode) {
            initializeForLandscape(mPicker);
        } else {
            // Takes care of initialization
            mPortraitButtonHandler = (ButtonLayout) mPicker.findViewById(R.id.button_layout);
        }
    }

    private void initializeForLandscape(MPicker mMaterialPicker) {
        Context context = MUtils.createThemeWrapper(mMaterialPicker.getContext(),
                R.attr.mPickerStyle,
                R.style.MPickerStyleLight,
                R.attr.spButtonLayoutStyle,
                R.style.ButtonLayoutStyle);
        final Resources res = context.getResources();

        final TypedArray a = context.obtainStyledAttributes(R.styleable.ButtonLayout);

        mSwitcherButtonDP = (Button) mMaterialPicker.findViewById(R.id.buttonSwitcherDP);
        mSwitcherButtonTP = (Button) mMaterialPicker.findViewById(R.id.buttonSwitcherTP);

        Button bPositiveDP = (Button) mMaterialPicker.findViewById(R.id.buttonPositiveDP);
        Button bPositiveTP = (Button) mMaterialPicker.findViewById(R.id.buttonPositiveTP);

        Button bNegativeDP = (Button) mMaterialPicker.findViewById(R.id.buttonNegativeDP);
        Button bNegativeTP = (Button) mMaterialPicker.findViewById(R.id.buttonNegativeTP);

        ImageView ivPositiveDP = (ImageView) mMaterialPicker.findViewById(R.id.imageViewPositiveDP);
        ImageView ivPositiveTP = (ImageView) mMaterialPicker.findViewById(R.id.imageViewPositiveTP);

        ImageView ivNegativeDP = (ImageView) mMaterialPicker.findViewById(R.id.imageViewNegativeDP);
        ImageView ivNegativeTP = (ImageView) mMaterialPicker.findViewById(R.id.imageViewNegativeTP);

        try {
            // obtain float value held by android.R.attr.disabledAlpha
            TypedValue typedValueDisabledAlpha = new TypedValue();
            context.getTheme().resolveAttribute(android.R.attr.disabledAlpha,
                    typedValueDisabledAlpha, true);

            // defaults to 0.5 ~ 122/255
            mDisabledAlpha = typedValueDisabledAlpha.type == TypedValue.TYPE_FLOAT ?
                    (int) (typedValueDisabledAlpha.getFloat() * 255)
                    : 122;

            // buttons or icons?
            int presentation = a.getInt(R.styleable.ButtonLayout_spPresentation, 0);

            int bgColor = a.getColor(R.styleable.ButtonLayout_spButtonBgColor,
                    MUtils.COLOR_BUTTON_NORMAL);
            int pressedBgColor = a.getColor(R.styleable.ButtonLayout_spButtonPressedBgColor,
                    MUtils.COLOR_CONTROL_HIGHLIGHT);

            mButtonBarBgColor = a.getColor(R.styleable.ButtonLayout_spButtonBarBgColor,
                    Color.TRANSPARENT);

            int buttonInvertedBgColor =
                    a.getColor(R.styleable.ButtonLayout_spButtonInvertedBgColor,
                            MUtils.COLOR_ACCENT);
            int buttonPressedInvertedBgColor =
                    a.getColor(R.styleable.ButtonLayout_spButtonPressedInvertedBgColor,
                            ContextCompat.getColor(context, R.color.sp_ripple_material_dark));
            MUtils.setViewBackground(mSwitcherButtonDP,
                    MUtils.createButtonBg(context, buttonInvertedBgColor,
                            buttonPressedInvertedBgColor));
            MUtils.setViewBackground(mSwitcherButtonTP,
                    MUtils.createButtonBg(context, buttonInvertedBgColor,
                            buttonPressedInvertedBgColor));

            if (presentation == 0 /* mode: Button */) {
                bPositiveDP.setVisibility(View.VISIBLE);
                bPositiveTP.setVisibility(View.VISIBLE);

                bNegativeDP.setVisibility(View.VISIBLE);
                bNegativeTP.setVisibility(View.VISIBLE);

                bPositiveDP.setText(res.getString(R.string.ok));
                bPositiveTP.setText(res.getString(R.string.ok));

                bNegativeDP.setText(res.getString(R.string.cancel));
                bNegativeTP.setText(res.getString(R.string.cancel));

                MUtils.setViewBackground(bPositiveDP,
                        MUtils.createButtonBg(context, bgColor,
                                pressedBgColor));
                MUtils.setViewBackground(bPositiveTP,
                        MUtils.createButtonBg(context, bgColor,
                                pressedBgColor));

                MUtils.setViewBackground(bNegativeDP,
                        MUtils.createButtonBg(context, bgColor,
                                pressedBgColor));
                MUtils.setViewBackground(bNegativeTP,
                        MUtils.createButtonBg(context, bgColor,
                                pressedBgColor));

                mPositiveButtonDP = bPositiveDP;
                mPositiveButtonTP = bPositiveTP;

                mNegativeButtonDP = bNegativeDP;
                mNegativeButtonTP = bNegativeTP;
            } else /* mode: ImageView */ {
                ivPositiveDP.setVisibility(View.VISIBLE);
                ivPositiveTP.setVisibility(View.VISIBLE);

                ivNegativeDP.setVisibility(View.VISIBLE);
                ivNegativeTP.setVisibility(View.VISIBLE);

                mIconOverlayColor = a.getColor(R.styleable.ButtonLayout_spIconColor,
                        MUtils.COLOR_ACCENT);

                ivPositiveDP.setColorFilter(mIconOverlayColor, PorterDuff.Mode.MULTIPLY);
                ivPositiveTP.setColorFilter(mIconOverlayColor, PorterDuff.Mode.MULTIPLY);

                ivNegativeDP.setColorFilter(mIconOverlayColor, PorterDuff.Mode.MULTIPLY);
                ivNegativeTP.setColorFilter(mIconOverlayColor, PorterDuff.Mode.MULTIPLY);

                MUtils.setViewBackground(ivPositiveDP,
                        MUtils.createImageViewBg(bgColor,
                                pressedBgColor));
                MUtils.setViewBackground(ivPositiveTP,
                        MUtils.createImageViewBg(bgColor,
                                pressedBgColor));

                MUtils.setViewBackground(ivNegativeDP,
                        MUtils.createImageViewBg(bgColor,
                                pressedBgColor));
                MUtils.setViewBackground(ivNegativeTP,
                        MUtils.createImageViewBg(bgColor,
                                pressedBgColor));

                mPositiveButtonDP = ivPositiveDP;
                mPositiveButtonTP = ivPositiveTP;

                mNegativeButtonDP = ivNegativeDP;
                mNegativeButtonTP = ivNegativeTP;
            }
        } finally {
            a.recycle();
        }

        // set OnClickListeners
        mPositiveButtonDP.setOnClickListener(this);
        mPositiveButtonTP.setOnClickListener(this);

        mNegativeButtonDP.setOnClickListener(this);
        mNegativeButtonTP.setOnClickListener(this);

        mSwitcherButtonDP.setOnClickListener(this);
        mSwitcherButtonTP.setOnClickListener(this);
    }

    /**
     * Initializes state for this layout
     *
     * @param switcherRequired Whether the switcher button needs
     *                         to be shown.
     * @param callback         Callback to 'MPicker'
     */
    public void applyOptions(boolean switcherRequired, @NonNull Callback callback) {
        mCallback = callback;

        if (mIsInLandscapeMode) {
            mSwitcherButtonDP.setVisibility(switcherRequired ? View.VISIBLE : View.GONE);
            mSwitcherButtonTP.setVisibility(switcherRequired ? View.VISIBLE : View.GONE);
        } else {
            // Let ButtonLayout handle callbacks
            mPortraitButtonHandler.applyOptions(switcherRequired, callback);
        }
    }

    // Returns whether switcher button is being used in this layout
    public boolean isSwitcherButtonEnabled() {
        return mIsInLandscapeMode ?
                (mSwitcherButtonDP.getVisibility() == View.VISIBLE || mSwitcherButtonTP.getVisibility() == View.VISIBLE)
                : (mPortraitButtonHandler.isSwitcherButtonEnabled());
    }

    // Used when the pickers are switched
    public void updateSwitcherText(@NonNull MOptions.Picker displayedPicker, CharSequence text) {
        if (mIsInLandscapeMode) {
            if (displayedPicker == MOptions.Picker.DATE_PICKER) {
                mSwitcherButtonDP.setText(text);
            } else if (displayedPicker == MOptions.Picker.TIME_PICKER) {
                mSwitcherButtonTP.setText(text);
            }
        } else {
            mPortraitButtonHandler.updateSwitcherText(text);
        }
    }

    // Disables the positive button as and when the user selected options
    // become invalid.
    public void updateValidity(boolean valid) {
        if (mIsInLandscapeMode) {
            mPositiveButtonDP.setEnabled(valid);
            mPositiveButtonTP.setEnabled(valid);

            // TODO: Find a better way to do this
            // Disabled state for Icon presentation (only for the positive check-mark icon)
            if (mPositiveButtonDP instanceof ImageView) {
                int color = mIconOverlayColor;

                if (!valid) {
                    color = (mDisabledAlpha << 24) | (mIconOverlayColor & 0x00FFFFFF);
                }

                ((ImageView) mPositiveButtonDP).setColorFilter(color,
                        PorterDuff.Mode.MULTIPLY);
                ((ImageView) mPositiveButtonTP).setColorFilter(color,
                        PorterDuff.Mode.MULTIPLY);
            }
        } else {
            mPortraitButtonHandler.updateValidity(valid);
        }
    }

    @Override
    public void onClick(View v) {
        if (v == mPositiveButtonDP || v == mPositiveButtonTP) {
            mCallback.onOkay();
        } else if (v == mNegativeButtonDP || v == mNegativeButtonTP) {
            mCallback.onCancel();
        } else if (v == mSwitcherButtonDP || v == mSwitcherButtonTP) {
            mCallback.onSwitch();
        }
    }

    public interface Callback {
        void onOkay();
        void onCancel();
        void onSwitch();
    }
}
