package com.instabug.survey.utils;

import static com.instabug.survey.settings.SurveysSettings.setIsLiveAppRequestedTrue;

import android.app.Activity;
import android.content.ActivityNotFoundException;
import android.content.Context;
import android.content.Intent;
import android.net.Uri;

import com.google.android.gms.tasks.Task;
import com.google.android.play.core.review.ReviewInfo;
import com.google.android.play.core.review.ReviewManager;
import com.google.android.play.core.review.ReviewManagerFactory;
import com.instabug.library.IBGNetworkWorker;
import com.instabug.library.internal.device.InstabugDeviceProperties;
import com.instabug.library.networkv2.NetworkManager;
import com.instabug.library.networkv2.RequestResponse;
import com.instabug.library.networkv2.request.Request;
import com.instabug.library.networkv2.request.RequestMethod;
import com.instabug.library.networkv2.request.RequestType;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.survey.Constants;
import com.instabug.survey.InAppRatingCallback;
import com.instabug.survey.InAppRatingRequestCallback;
import com.instabug.survey.settings.SurveysSettings;

import java.io.FileNotFoundException;

/**
 * @author hossam.
 */

public class PlayStoreUtils {

    private static final String MARKET_URI = "market://details?id=";
    private static final String PLAY_STORE_URL = "https://play.google.com/store/apps/details?id=";

    // TODO: 8/13/20 Use the New Google Play API to rate apps on the store
    public static void rateNow(Context context) {
        Uri uri = Uri.parse(MARKET_URI + InstabugDeviceProperties.getPackageName(context));
        Intent goToMarket = new Intent(Intent.ACTION_VIEW, uri);
        // To count with Play market backstack, After pressing back button,
        // to taken back to our application, we need to add following flags to intent.
        goToMarket.addFlags(Intent.FLAG_ACTIVITY_NO_HISTORY |
                Intent.FLAG_ACTIVITY_NEW_DOCUMENT |
                Intent.FLAG_ACTIVITY_MULTIPLE_TASK);
        goToMarket.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
        try {
            context.startActivity(goToMarket);
        } catch (ActivityNotFoundException e) {
            try {
                Intent intent = new Intent(Intent.ACTION_VIEW,
                        Uri.parse(PLAY_STORE_URL + InstabugDeviceProperties.getPackageName(context)));
                intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
                context.startActivity(intent);
            } catch (Exception exception) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "Error: " + exception.getMessage() + " while rating app");
            }
        }
    }

    public static void isLiveApp(Context context) {
        if (SurveysSettings.isAppStoreRatingEnabled() != null) {
            return;
        }

        String url = (PLAY_STORE_URL + InstabugDeviceProperties.getPackageName(context));
        NetworkManager networkManager = new NetworkManager();
        Request checkLiveAppRequest = new Request.Builder()
                .method(RequestMethod.GET)
                .url(url)
                .disableDefaultParameters(true)
                .build();
        networkManager.doRequest(IBGNetworkWorker.SURVEYS, RequestType.NORMAL, checkLiveAppRequest, new Request.Callbacks<RequestResponse, Throwable>() {
            @Override
            public void onSucceeded(RequestResponse requestResponse) {
                InstabugSDKLogger
                        .v(PlayStoreUtils.class.getSimpleName(), "checkingIsLiveApp succeeded, Response code: " + requestResponse.getResponseCode());

                setIsLiveAppRequestedTrue();
                SurveysSettings.setIsAppLive(requestResponse.getResponseCode() == RequestResponse.HttpStatusCode._2xx.OK);
            }

            @Override
            public void onFailed(Throwable error) {
                if (error instanceof FileNotFoundException) {
                    InstabugSDKLogger.w(Constants.LOG_TAG, "Couldn't find this app on playstore");
                } else {
                    InstabugSDKLogger.w(Constants.LOG_TAG, "checkingIsLiveApp got error: " + error.toString());
                }
                SurveysSettings.setIsAppLive(false);
                setIsLiveAppRequestedTrue();
            }
        });
    }

    public static void requestInAppRating(final Activity activity, ReviewInfo reviewInfo, final InAppRatingRequestCallback callback) {
        try {
            final ReviewManager manager = ReviewManagerFactory.create(activity);
            Task<Void> flow = manager.launchReviewFlow(activity, reviewInfo);
            flow.addOnCompleteListener(callback::onComplete);
            flow.addOnFailureListener(e -> callback.onFailure(new Exception("GooglePlay in-app review flow request failed", e)));
        } catch (Exception e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "GooglePlay in-app review request failed due to error: " + e.getMessage());
        }
    }

    public static void prepareInAppRating(final Activity activity, final InAppRatingCallback callback) {
        try {
            final ReviewManager manager = ReviewManagerFactory.create(activity);
            Task<ReviewInfo> request = manager.requestReviewFlow();
            request.addOnCompleteListener(task -> {
                try {
                    if (task.isSuccessful()) {
                        // `getResult()` throws RuntimeExecutionException
                        ReviewInfo reviewInfo = task.getResult();
                        callback.onSuccess(reviewInfo);
                    } else {
                        callback.onFailure(new Exception("GooglePlay in-app review task did not succeed, result: " + task.getResult()));
                    }
                } catch (Exception e) {
                    InstabugSDKLogger.e(Constants.LOG_TAG, "Couldn't get GooglePlay in-app review request result" + e.getMessage());
                }
            }).addOnFailureListener(e -> callback.onFailure(new Exception("GooglePlay in-app review request failed", e)));
        } catch (Exception e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "GooglePlay in-app review request failed due to error: " + e.getMessage());
        }
    }

}
