package com.instabug.survey.ui.survey.rateus;

import android.graphics.drawable.Drawable;
import android.os.Bundle;
import android.view.View;
import android.view.animation.Animation;
import android.view.animation.AnimationUtils;
import android.widget.ImageView;
import android.widget.TextView;

import androidx.annotation.CallSuper;
import androidx.annotation.Nullable;
import androidx.core.content.ContextCompat;

import com.instabug.library.core.InstabugCore;
import com.instabug.library.util.AccessibilityUtils;
import com.instabug.library.util.Colorizer;
import com.instabug.survey.R;
import com.instabug.survey.models.Question;
import com.instabug.survey.models.Survey;
import com.instabug.survey.ui.survey.QuestionFragment;
import com.instabug.survey.ui.survey.SurveyFragmentCallbacks;

import org.jetbrains.annotations.NotNull;

public abstract class RateUsAbstractFragment extends QuestionFragment {

    @Nullable
    private TextView questionTitleTextView;
    @Nullable
    private ImageView brandingImage;

    public static RateUsFragment newInstance(Survey survey, SurveyFragmentCallbacks surveyFragmentCallbacks) {
        RateUsFragment questionFragment = new RateUsFragment();
        Bundle bundle = new Bundle();
        bundle.putSerializable(KEY_QUESTION_ARGUMENT, survey.getQuestions().get(0));
        questionFragment.setArguments(bundle);
        questionFragment.setSurveyFragmentCallbacks(surveyFragmentCallbacks);
        return questionFragment;
    }

    @Override
    public void onCreate(@Nullable Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setRetainInstance(true);
        if (getArguments() != null) {
            question = (Question) getArguments().getSerializable(KEY_QUESTION_ARGUMENT);
        }
    }

    @Override
    @CallSuper
    protected void initViews(View rootView, @Nullable Bundle savedInstanceState) {
        super.initViews(rootView, savedInstanceState);
        questionTitleTextView = rootView.findViewById(R.id.txt_rate_us_title);
        questionTextView = rootView.findViewById(R.id.txt_rate_us_question);
        brandingImage = rootView.findViewById(R.id.instabug_img_thanks);
        ImageView imageView = brandingImage;
        if (imageView != null) {
            imageView.setColorFilter(getBrandingImageColorFilter());
            if (getContext() != null) {
                Drawable drawable = ContextCompat.getDrawable(getContext(), R.drawable.ibg_survey_ic_thanks_background);
                if (drawable != null)
                    imageView.setBackgroundDrawable(getBrandingImageBackgroundDrawable(drawable));
            }
        }
        if (questionTitleTextView != null) {
            questionTitleTextView.setTextColor(getQuestionTitleTextColor());
        }
        startIntoAnimation();
    }

    protected int getBrandingImageColorFilter() {
        // This is the default behavior
        return InstabugCore.getPrimaryColor();
    }

    protected Drawable getBrandingImageBackgroundDrawable(Drawable drawable) {
        return Colorizer.getPrimaryColorTintedDrawable(drawable);
    }

    protected int getQuestionTitleTextColor() {
        return InstabugCore.getPrimaryColor();
    }

    @Override
    @Nullable
    public String getAnswer() {
        if (question == null) return null;
        return question.getAnswer();
    }

    public void showRateUsQuestion() {
        if (survey == null || questionTextView == null || question == null) return;
        if (survey.getThankYouTitle() != null) {
            if (questionTitleTextView != null)
                questionTitleTextView.setText(survey.getThankYouTitle());
        } else {
            if (questionTitleTextView != null)
                questionTitleTextView.setText(R.string.instabug_custom_survey_thanks_title);
        }
        if (survey.getThankYouMessage() != null) {
            questionTextView.setText(survey.getThankYouMessage());
        } else {
            if (question.getTitle() != null) {
                questionTextView.setText(question.getTitle());
            }
        }
    }

    private void startIntoAnimation() {
        if (getContext() == null) return;

        final Animation a1 = AnimationUtils.loadAnimation(getContext(), R.anim.ib_srv_anim_fade_in_scale);
        final Animation a2 = AnimationUtils.loadAnimation(getContext(), R.anim.ib_srv_anim_fly_in);
        final Animation a3 = AnimationUtils.loadAnimation(getContext(), R.anim.ib_srv_anim_fly_in);
        a2.setStartOffset(300);
        a3.setStartOffset(400);
        if (questionTitleTextView != null) {
            questionTitleTextView.getViewTreeObserver().addOnGlobalLayoutListener(() -> {
                if (brandingImage != null) {
                    brandingImage.startAnimation(a1);
                }
                if (questionTitleTextView != null) {
                    questionTitleTextView.startAnimation(a2);
                }
                if (questionTextView != null) {
                    questionTextView.startAnimation(a3);
                }
            });
        }
    }

    @Override
    public void onViewCreated(@NotNull View view, @Nullable Bundle savedInstanceState) {
        super.onViewCreated(view, savedInstanceState);
        if (AccessibilityUtils.isTalkbackEnabled()) {
            View thanksContainer = findViewById(R.id.thanks_container_layout);
            if (thanksContainer != null)
                thanksContainer.requestFocus();
        }
    }

    @Override
    protected int getLayout() {
        return R.layout.survey_rate_us_fragment;
    }

    @Nullable
    public Question getQuestion() {
        return question;
    }
}
