package com.instabug.survey.ui.survey.mcq;


import static androidx.core.view.accessibility.AccessibilityNodeInfoCompat.ACTION_CLICK;

import android.annotation.SuppressLint;
import android.app.Activity;
import android.content.Context;
import android.content.res.Resources;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.BaseAdapter;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.TextView;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.core.graphics.ColorUtils;
import androidx.core.view.AccessibilityDelegateCompat;
import androidx.core.view.ViewCompat;
import androidx.core.view.accessibility.AccessibilityNodeInfoCompat;

import com.instabug.library.InstabugColorTheme;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.util.AccessibilityUtils;
import com.instabug.library.util.AttrResolver;
import com.instabug.library.util.DrawableUtils;
import com.instabug.library.util.ResourcesUtils;
import com.instabug.survey.R;
import com.instabug.survey.models.Question;

/**
 * Created by mohamedzakaria on 7/18/17.
 */

public abstract class SurveyMCQGridAbstractAdapter extends BaseAdapter {
    private final LayoutInflater inflater;
    private AnswerOnClickListener onClickListener;
    private Question question;
    private int selectedPos = -1;
    protected Context context;

    public SurveyMCQGridAbstractAdapter(Activity activity, Question question,
                                        AnswerOnClickListener onClickListener) {
        context = activity;
        inflater = LayoutInflater.from(activity);
        this.question = question;
        setDefaultSelected(question);
        this.onClickListener = onClickListener;
    }

    private void setDefaultSelected(Question question) {
        if (question.getOptionalAnswers() == null) return;

        for (int i = 0; i < question.getOptionalAnswers().size(); i++) {
            if (question.getAnswer() != null && question.getAnswer().equals(question.getOptionalAnswers().get(i))) {
                selectedPos = i;
                break;
            }
        }
    }

    @Override
    public int getCount() {
        return question != null && question.getOptionalAnswers() != null ? question.getOptionalAnswers().size() : 0;
    }

    @Override
    public String getItem(int position) {
        if (question.getOptionalAnswers() == null) return "null";

        return question.getOptionalAnswers().get(position);
    }

    @Override
    public long getItemId(int position) {
        return position;
    }

    @SuppressLint("InflateParams")
    @Override
    public View getView(int position, View convertView, ViewGroup parent) {
        final MCQViewHolder holder;
        if (convertView == null) {
            holder = new MCQViewHolder();
            convertView = inflater.inflate(
                    R.layout.instabug_survey_mcq_item, null);
            holder.item = convertView.findViewById(R.id.mcq_item);
            holder.optionalAnswer = convertView.findViewById(R.id.survey_optional_answer_textview);
            holder.mcqSelectorIcon = convertView.findViewById(R.id.selector_img);
            convertView.setTag(holder);
        } else {
            holder = (MCQViewHolder) convertView.getTag();
        }

        if (question.getOptionalAnswers() != null && holder.optionalAnswer != null)
            holder.optionalAnswer.setText(question.getOptionalAnswers().get(position));

        boolean isSelected = position == selectedPos;
        if (isSelected) {
            handleSelectedHolderColors(holder);
        } else {
            handleUnselectedHolderColors(holder);
        }
        if (onClickListener != null && question.getOptionalAnswers() != null) {
            if (holder.item != null) {
                holder.item.setOnClickListener(registerOnClickListener(question.getOptionalAnswers().get(position), position));
                setAccessibilityDelegate(position, holder.item, isSelected);
            }

            if (holder.optionalAnswer != null) {
                holder.optionalAnswer.setOnClickListener(registerOnClickListener(question.getOptionalAnswers().get(position), position));
                holder.optionalAnswer.setImportantForAccessibility( View.IMPORTANT_FOR_ACCESSIBILITY_NO);
            }
            if (holder.mcqSelectorIcon != null) {
                holder.mcqSelectorIcon.setOnClickListener(registerOnClickListener(question.getOptionalAnswers().get(position), position));
                holder.mcqSelectorIcon.setImportantForAccessibility(View.IMPORTANT_FOR_ACCESSIBILITY_NO);

            }
        }
        return convertView;
    }

    private void setAccessibilityDelegate(int position, @NonNull LinearLayout item, boolean isSelected) {
        ViewCompat.setAccessibilityDelegate(item, new AccessibilityDelegateCompat() {
            @Override
            public void onInitializeAccessibilityNodeInfo(View host, AccessibilityNodeInfoCompat info) {
                super.onInitializeAccessibilityNodeInfo(host, info);
                Resources res = host.getResources();
                int currentOption = position + 1;
                String selectedText = ResourcesUtils.getSelectedText(res, isSelected);
                String itemDescription = getItemDescription(res, currentOption, selectedText, position);
                info.setImportantForAccessibility(true);
                info.setText(itemDescription);
                info.addAction(new AccessibilityNodeInfoCompat.AccessibilityActionCompat(ACTION_CLICK, res.getString(R.string.ib_action_select)));

            }
        });
    }

    private String getItemDescription(Resources res, int currentOption, String selectedText, int position) {
        return res.getString(R.string.ib_msq_item_description,
                currentOption,
                getCount(),
                getItem(position),
                selectedText);
    }


    private void handleSelectedHolderColors(MCQViewHolder holder) {
        if (InstabugCore.getTheme() == InstabugColorTheme.InstabugColorThemeLight) {
            if (holder.item != null)
                DrawableUtils.setColor(holder.item, ColorUtils.setAlphaComponent(getSelectedItemBgColor(holder), 25));
        } else {
            if (holder.item != null)
                DrawableUtils.setColor(holder.item, ColorUtils.setAlphaComponent(getSelectedItemBgColor(holder), 50));
        }
        if (holder.optionalAnswer != null) {
            holder.optionalAnswer.setTextColor(getSelectedTextColor(holder));
        }
        handleSelectedIconImage(holder);
    }

    protected abstract int getSelectedItemBgColor(MCQViewHolder holder);

    protected abstract int getSelectedTextColor(MCQViewHolder holder);

    protected abstract void handleSelectedIconImage(MCQViewHolder holder);

    private void handleUnselectedHolderColors(MCQViewHolder holder) {
        if (holder.item != null) {
            DrawableUtils.setColor(holder.item, getUnselectedItemBgColor(holder));
        }
        if (context != null && holder.optionalAnswer != null) {
            holder.optionalAnswer.setTextColor(AttrResolver.resolveAttributeColor(context, R.attr.instabug_survey_mcq_text_color));
        }
        handleUnselectedIconImage(holder);
    }

    protected abstract int getUnselectedItemBgColor(MCQViewHolder holder);

    protected abstract void handleUnselectedIconImage(MCQViewHolder holder);

    private View.OnClickListener registerOnClickListener(final String answer, final int position) {
        return view -> {
            setSelected(position);
            AccessibilityUtils.sendTextEvent(answer + " " + view.getResources().getString(R.string.ib_selected));
            onClickListener.onAnswerClicked(view, answer);
        };
    }

    private void setSelected(int position) {
        selectedPos = position;
        notifyDataSetChanged();
    }

    @Nullable
    public String getCurrentAnswer() {
        if (selectedPos == -1) {
            return null;
        }
        return getItem(selectedPos);
    }

    public void setAnswer(@Nullable String answer) {
        if (answer != null && !answer.isEmpty()) {
            for (int i = 0; i < getCount(); i++) {
                String preDefinedAnswers = getItem(i);
                if (answer.equalsIgnoreCase(preDefinedAnswers)) {
                    selectedPos = i;
                    break;
                }
            }
        }
    }

    protected static class MCQViewHolder {
        @Nullable
        LinearLayout item;
        @Nullable
        TextView optionalAnswer;
        @Nullable
        ImageView mcqSelectorIcon;
    }

    public interface AnswerOnClickListener {
        /**
         * used with MCQ questions
         *
         * @param v
         * @param answer
         */
        void onAnswerClicked(View v, String answer);
    }
}