package com.instabug.survey.network;

import android.content.Context;

import androidx.annotation.Nullable;

import com.instabug.library.networkv2.request.Request;
import com.instabug.library.util.TimeUtils;
import com.instabug.survey.models.Survey;
import com.instabug.survey.network.service.SurveysService;
import com.instabug.survey.settings.SurveysSettings;
import com.instabug.survey.utils.SurveysUtils;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.List;

/**
 * @author mesbah
 */

public class SurveysFetcher {

    private Callback callback;

    public SurveysFetcher(Callback callback) {
        this.callback = callback;
    }

    public void fetch(Context context, String locale) throws JSONException {
        if (SurveysUtils.isSurveysFeatureEnabled() && shouldFetchAgain()) {
            SurveysService.getInstance().fetchSurveys(locale, new Request.Callbacks<JSONObject, Throwable>() {
                @Override
                public void onSucceeded(@Nullable JSONObject response) {
                    try {
                        SurveysSettings.setLastFetchedAt(TimeUtils.currentTimeMillis());
                        if (response != null) {
                            List<Survey> remoteSurveys = Survey.fromJson(response);
                            List<Survey> pausedSurveys = Survey.getPausedSurveysFromJson(response);
                            remoteSurveys.addAll(pausedSurveys);
                            callback.onFetchingSucceeded(remoteSurveys);
                        } else {
                            callback.onFetchingFailed(new NullPointerException("Json response is null"));
                        }
                    } catch (JSONException jSONException) {
                        callback.onFetchingFailed(jSONException);
                    }
                }

                @Override
                public void onFailed(Throwable error) {
                    callback.onFetchingFailed(error);
                }
            });
        }
    }

    private boolean shouldFetchAgain() {
        return (TimeUtils.currentTimeMillis() - SurveysSettings.getLastFetchedAt()) > 10000;
    }

    public interface Callback {

        void onFetchingSucceeded(List<Survey> remoteSurveys);

        void onFetchingFailed(Throwable error);
    }
}
