package com.instabug.survey.common.models;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.instabug.library.internal.storage.cache.Cacheable;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.Serializable;
import java.util.ArrayList;

/**
 * Created by zak on 1/31/18.
 */

public class ActionEvent implements Cacheable, Serializable {
    public static final String KEY_EVENT_TYPE = "event_type";
    public static final String KEY_TIMESTAMP = "timestamp";
    public static final String KEY_INDEX = "index";

    private static final String KEY_SUBMIT = "submit";
    private static final String KEY_SHOW = "show";
    private static final String KEY_DISMISS = "dismiss";
    private static final String KEY_RATE = "rate";
    private static final String KEY_SYNCED = "synced";
    @Nullable
    private EventType eventType;
    long timestamp;
    int index;
    private boolean isSynced = false;

    private ActionEvent() {
    }

    public ActionEvent(@NonNull EventType eventType, long timestamp, int index) {
        this.eventType = eventType;
        this.timestamp = timestamp;
        this.index = index;
    }

    @Nullable
    public EventType getEventType() {
        return eventType;
    }

    public void setEventType(@NonNull EventType eventType) {
        this.eventType = eventType;
    }

    public long getTimestamp() {
        return timestamp;
    }

    public void setTimestamp(long timestamp) {
        this.timestamp = timestamp;
    }

    public int getIndex() {
        return index;
    }

    public void setIndex(int index) {
        this.index = index;
    }

    public static ArrayList<ActionEvent> fromJson(JSONArray surveyEventsJsonArray) throws
            JSONException {
        ArrayList<ActionEvent> actionEvents = new ArrayList<>();
        for (int i = 0; i < surveyEventsJsonArray.length(); i++) {
            ActionEvent actionEvent = new ActionEvent();
            actionEvent.fromJson(surveyEventsJsonArray.getJSONObject(i).toString());
            actionEvents.add(actionEvent);
        }
        return actionEvents;
    }

    public static JSONArray toJson(ArrayList<ActionEvent> actionEvents) throws JSONException {
        JSONArray surveyEventsArray = new JSONArray();
        if (actionEvents != null) {
            for (int i = 0; i < actionEvents.size(); i++) {
                if (actionEvents.get(i) != null) {
                    surveyEventsArray.put(new JSONObject(actionEvents.get(i).toJson()));
                }
            }
        }
        return surveyEventsArray;
    }

    @Override
    public String toJson() throws JSONException {
        JSONObject surveyEventJsonObject = new JSONObject();
        surveyEventJsonObject.put(KEY_EVENT_TYPE,
                getEventType() == null ? "" : getEventType().toString());
        surveyEventJsonObject.put(KEY_INDEX, getIndex());
        surveyEventJsonObject.put(KEY_TIMESTAMP, timestamp);
        surveyEventJsonObject.put(KEY_SYNCED, isSynced);
        return surveyEventJsonObject.toString();
    }

    @Override
    public void fromJson(String modelAsJson) throws JSONException {
        JSONObject surveyEventJsonObject = new JSONObject(modelAsJson);
        if (surveyEventJsonObject.has(KEY_EVENT_TYPE)) {
            String type = surveyEventJsonObject.getString(KEY_EVENT_TYPE);
            switch (type) {
                case KEY_SUBMIT:
                    setEventType(EventType.SUBMIT);
                    break;
                case KEY_DISMISS:
                    setEventType(EventType.DISMISS);
                    break;
                case KEY_RATE:
                    setEventType(EventType.RATE);
                    break;
                case KEY_SHOW:
                    setEventType(EventType.SHOW);
                    break;
                default:
                    setEventType(EventType.UNDEFINED);
                    break;
            }
        }
        if (surveyEventJsonObject.has(KEY_INDEX))
            setIndex(surveyEventJsonObject.getInt(KEY_INDEX));
        if (surveyEventJsonObject.has(KEY_TIMESTAMP))
            setTimestamp(surveyEventJsonObject.getLong(KEY_TIMESTAMP));
        if (surveyEventJsonObject.has(KEY_SYNCED)) {
            setSynced(surveyEventJsonObject.getBoolean(KEY_SYNCED));
        }
    }

    public boolean isSynced() {
        return isSynced;
    }

    public void setSynced(boolean synced) {
        isSynced = synced;
    }


    public enum EventType {
        SHOW("show"), DISMISS("dismiss"), SUBMIT("submit"), RATE("rate"), UNDEFINED("undefined");
        private final String event;

        EventType(String event) {
            this.event = event;
        }

        @Override
        public String toString() {
            return event;
        }

    }

}
