package com.instabug.survey;

import android.annotation.SuppressLint;
import android.content.Context;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;
import androidx.annotation.WorkerThread;

import com.instabug.library.Instabug;
import com.instabug.library.InstabugState;
import com.instabug.library.InstabugStateProvider;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.core.eventbus.UserEventsEventBus;
import com.instabug.library.internal.device.InstabugDeviceProperties;
import com.instabug.library.internal.storage.cache.db.InstabugDBInsertionListener;
import com.instabug.library.user.UserEvent;
import com.instabug.library.user.UserManagerWrapper;
import com.instabug.library.util.DeviceStateProvider;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.LocaleUtils;
import com.instabug.library.util.TaskDebouncer;
import com.instabug.library.util.TimeUtils;
import com.instabug.library.util.threading.PoolProvider;
import com.instabug.survey.announcements.settings.AnnouncementsSettings;
import com.instabug.survey.cache.SurveysCacheManager;
import com.instabug.survey.common.AutoShowingManager;
import com.instabug.survey.common.models.Frequency;
import com.instabug.survey.common.models.UserInteraction;
import com.instabug.survey.common.userInteractions.UserInteractionCacheManager;
import com.instabug.survey.configuration.SurveysConfigurationsProvider;
import com.instabug.survey.di.ServiceLocator;
import com.instabug.survey.models.CountryInfo;
import com.instabug.survey.models.Survey;
import com.instabug.survey.models.UserInteractionOnTypes;
import com.instabug.survey.network.CountryInfoResolver;
import com.instabug.survey.network.SurveysFetcher;
import com.instabug.survey.settings.SurveysSettings;
import com.instabug.survey.utils.SurveysUtils;
import com.instabug.survey.utils.SurveysValidator;

import org.json.JSONException;

import java.lang.ref.WeakReference;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.TimeUnit;

import io.reactivexport.disposables.Disposable;
import io.reactivexport.functions.Consumer;

/**
 * @author mesbah
 */

public class SurveysManager implements SurveysFetcher.Callback, SurveysValidator.Callback, CountryInfoResolver.CountryInfoResolverCallback {

    @Nullable
    private static SurveysManager singleton;
    private final WeakReference<Context> context;
    private final SurveysFetcher surveysFetcher;
    private SurveysValidator surveysValidator;
    @Nullable
    private Disposable triggerDisposable;
    private final CountryInfoResolver countryInfoResolver;
    private final TaskDebouncer networkTaskDebouncer;
    private final SurveysConfigurationsProvider configurationsProvider = ServiceLocator.getConfigurationsProvider();

    @VisibleForTesting
    boolean hasTokenChanged = false;

    private SurveysManager(@NonNull Context context) {
        this.context = new WeakReference<>(context);
        surveysFetcher = new SurveysFetcher(this);
        surveysValidator = new SurveysValidator(this,
                InstabugDeviceProperties.getAppVersionName(context), // eg. 1.0.0-beta
                DeviceStateProvider.getAppVersion(context)); // eg. 1.0.0-beta (3)
        countryInfoResolver = new CountryInfoResolver(this);
        networkTaskDebouncer = new TaskDebouncer(TimeUnit.MINUTES.toMillis(1));
        registerSurveysTriggerEvents();
    }

    public synchronized static void init() {
        if (Instabug.getApplicationContext() == null) return;

        singleton = new SurveysManager(Instabug.getApplicationContext());
    }

    /**
     * Returns the current singleton instance of this class.
     *
     * @return a {@link SurveysManager} instance
     */
    @SuppressLint("ERADICATE_RETURN_NOT_NULLABLE")
    public synchronized static SurveysManager getInstance() {
        if (singleton == null) {
            init();
        }
        return singleton;
    }

    public void startFetching(final String locale) {
        networkTaskDebouncer.debounce(new Runnable() {
            @Override
            public void run() {
                fetchSurveys(locale);
            }
        });
    }

    public void setHasTokenChanged(boolean hasTokenChanged) {
        this.hasTokenChanged = hasTokenChanged;
    }

    public void fetchImmediately(final String locale) {
        SurveysSettings.setLastFetchedAt(0);
        fetchSurveys(locale);
    }

    private void fetchSurveys(String locale) {
        if (locale != null) {
            try {
                if (context.get() != null) {
                    surveysFetcher.fetch(context.get(), locale);
                }
            } catch (JSONException e) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "Couldn't fetch surveys due to: " + e.getMessage(), e);
            }
        }
    }


    public void registerSurveysTriggerEvents() {
        if (triggerDisposable == null || triggerDisposable.isDisposed()) {
            triggerDisposable = UserEventsEventBus.getInstance().subscribe(new Consumer<UserEvent>() {
                @Override
                public void accept(UserEvent userEvent) throws Exception {
                    if (!canShowSurveys())
                        return;
                    if (userEvent instanceof SurveyTimerEvent) {
                        InstabugSDKLogger.v(Constants.LOG_TAG, "Surveys auto showing is triggered");
                        surveysValidator.showSurveysByTimeTriggerIfAvailable();
                    } else {
                        if (SurveysSettings.isSurveysAutoShowing() && userEvent.getEventIdentifier() != null) {
                            InstabugSDKLogger.v(Constants.LOG_TAG, "Survey with event: {" + userEvent.getEventIdentifier() + "} is triggered");
                            surveysValidator.showSurveysByEventTriggerIfAvailable(userEvent.getEventIdentifier());
                        }
                    }
                }
            });
        }
    }

    public void unregisterSurveysTriggerEvents() {
        if (triggerDisposable != null && !triggerDisposable.isDisposed()) {
            triggerDisposable.dispose();
        }
    }

    @SuppressLint("THREAD_SAFETY_VIOLATION")
    boolean showValidSurvey() {
        if (!Instabug.isEnabled()) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Couldn't show survey because Instabug SDK is disabled.");
            return false;
        }
        try {
            if (canShowSurveys()) {
                Survey firstValidSurvey = getFirstValidSurvey();
                if (firstValidSurvey != null) {
                    showSurvey(firstValidSurvey);
                    return true;
                } else {
                    return false;
                }
            } else {
                return false;
            }
        } catch (ParseException e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Something went wrong while getting first valid survey", e);
            return false;
        }
    }

    @VisibleForTesting
    boolean canShowSurveys() {
        return InstabugStateProvider.getInstance().getState().equals(InstabugState.ENABLED)
                && SurveysUtils.isSurveysFeatureEnabled()
                && Instabug.isAppOnForeground() &&
                !InstabugCore.isForegroundBusy() &&
                configurationsProvider.isSurveysAvailableAndUsageNotExceeded()
                && !hasTokenChanged;
    }

    boolean showSurvey(String token) {
        if (canShowSurveys()) {
            Survey optinSurvey = getSurvey(token);
            if (optinSurvey != null && !optinSurvey.isPaused()) {
                showSurvey(optinSurvey);
                return true;
            }
        }
        return false;
    }

    @Nullable
    @VisibleForTesting
    Survey getSurvey(String token) {
        List<Survey> surveyList = SurveysCacheManager.getSurveys();
        for (Survey survey : surveyList) {
            if (survey.getToken() != null && survey.getToken().equals(token)) {
                InstabugSDKLogger.d(Constants.LOG_TAG, "Showing survey With token " + token);
                return survey;
            }
        }
        InstabugSDKLogger.d(Constants.LOG_TAG, "No Survey With token " + token);
        return null;
    }

    @Override
    public void onFetchingSucceeded(List<Survey> remoteSurveys) {
        cacheLastRetrievedLocale();
        invalidateSurveyUserInteractions(remoteSurveys);
        invalidateCachedSurveys(remoteSurveys);
        migrateSurveys(remoteSurveys);
        if (!Instabug.isEnabled()) {
            return;
        }
        startAutomaticSurveyTrigger();
        hasTokenChanged = false;
    }

    @VisibleForTesting
    void cacheLastRetrievedLocale() {
        if (context.get() != null) {
            String locale = LocaleUtils.getCurrentLocaleResolved(context.get());
            SurveysSettings.setLastFetchedLocaleCode(locale);
        }
    }

    @VisibleForTesting
    void invalidateCachedSurveys(List<Survey> remoteSurveys) {
        List<Survey> allSurveys = SurveysCacheManager.getSurveys();
        for (Survey survey : allSurveys) {
            if (!remoteSurveys.contains(survey)) {
                SurveysCacheManager.delete(survey.getId());
            }
        }
    }

    /**
     * This method to invalidate user interactions if survey is removed then any related user interactions
     * will be removed as well.
     * <p>
     * Order of calling this method must be considered, as this method will be triggered before invalidating surveys
     *
     * @param remoteSurveys to check it aganist the cached version.
     */
    @VisibleForTesting
    void invalidateSurveyUserInteractions(List<Survey> remoteSurveys) {
        List<Survey> allSurveys = SurveysCacheManager.getSurveys();
        String uuid = UserManagerWrapper.getUserUUID();
        List<UserInteraction> userInteractionsToBeRemoved = new ArrayList<>();
        for (Survey survey : allSurveys) {
            if (!remoteSurveys.contains(survey)) {
                UserInteraction userInteraction = UserInteractionCacheManager.retrieveUserInteraction(
                        survey.getId(),
                        uuid,
                        UserInteractionOnTypes.SURVEY
                );
                if (userInteraction != null) {
                    userInteractionsToBeRemoved.add(userInteraction);
                }
            }
        }
        if (!userInteractionsToBeRemoved.isEmpty())
            UserInteractionCacheManager.deleteBulkOfUserInteractions(userInteractionsToBeRemoved);


    }


    @Override
    public void onFetchingFailed(Throwable error) {
        if (error.getMessage() != null) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Couldn't fetch surveys due to: " + error.getMessage(), error);
        }
        startAutomaticSurveyTrigger();
    }

    private void startAutomaticSurveyTrigger() {
        try {
            Thread.sleep(SurveysSettings.DELAY_BEFORE_SHOWING);
            if (SurveysSettings.isSurveysAutoShowing() && Instabug.isAppOnForeground()) {
                UserEventsEventBus.getInstance().post(new SurveyTimerEvent());
            }
        } catch (InterruptedException e) {
            if (e.getMessage() != null) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "Couldn't show survey because thread was interrupted");
            }
        }
    }


    /**
     * Persist surveys if they are not already persisted,
     * otherwise update them if they are paused or the locale has changed
     */
    @WorkerThread
    @VisibleForTesting
    void migrateSurveys(final List<Survey> remoteSurveys) {
        for (Survey survey : remoteSurveys) {
            if (SurveysCacheManager.isSurveyExisting(survey.getId())) {
                Survey cachedSurvey = SurveysCacheManager.getSurveyById(survey.getId());
                if (cachedSurvey != null) {
                    boolean publishStatusChanged = isPublishStatusChanged(survey, cachedSurvey);
                    boolean localeChanged = false;
                    if (!survey.isPaused()) {
                        localeChanged = isLocaleChanged(survey, cachedSurvey);
                    }
                    if (publishStatusChanged || localeChanged) {
                        SurveysCacheManager.insertOrUpdatePausedOrLocale(survey,
                                publishStatusChanged, localeChanged);
                    }
                    if (isFrequencyChanged(survey, cachedSurvey)) {
                        cachedSurvey.getTarget().setFrequency(survey.getTarget().getFrequency());
                        SurveysCacheManager.updateSurveyTarget(cachedSurvey);
                    }
                }
            } else if (!survey.isPaused()) {
                SurveysCacheManager.addSurvey(survey);
            }
        }
    }

    /**
     * This method to merge user interaction, check cached user interaction for the remote survey
     * if the survey already has user interaction then update this survey, otherwise treat it as a new one.
     *
     * @param remoteSurveys .
     */
    @VisibleForTesting
    void mergeUserInteraction(List<Survey> remoteSurveys) {
        final String uuid = UserManagerWrapper.getUserUUID();
        List<Survey> surveysToBeUpdated = new ArrayList<>();
        for (Survey remoteSurvey : remoteSurveys) {
            UserInteraction userInteraction = UserInteractionCacheManager.retrieveUserInteraction(
                    remoteSurvey.getId(),
                    uuid,
                    UserInteractionOnTypes.SURVEY);
            if (userInteraction != null) {
                remoteSurvey.setUserInteraction(userInteraction);
                surveysToBeUpdated.add(remoteSurvey);
            }
        }

        if (!remoteSurveys.isEmpty()) {
            SurveysCacheManager.updateBulk(surveysToBeUpdated);
        }
    }

    @VisibleForTesting
    boolean isLocaleChanged(Survey survey, Survey cachedSurvey) {
        return survey.getLocalization().getCurrentLocale() != null &&
                !survey.getLocalization().getCurrentLocale()
                        .equals(cachedSurvey.getLocalization().getCurrentLocale());
    }

    @VisibleForTesting
    boolean isFrequencyChanged(Survey survey, Survey cachedSurvey) {
        Frequency remoteFrequency = survey.getTarget().getFrequency();
        Frequency cachedFrequency = cachedSurvey.getTarget().getFrequency();
        return remoteFrequency.getType() != cachedFrequency.getType() ||
                remoteFrequency.getDismissedReshowInterval() != cachedFrequency.getDismissedReshowInterval() ||
                remoteFrequency.getShowingInterval() != cachedFrequency.getShowingInterval();
    }

    @VisibleForTesting
    boolean isPublishStatusChanged(Survey survey, Survey cachedSurvey) {
        return cachedSurvey.isPaused() != survey.isPaused();
    }

    @Nullable
    private Survey getFirstValidSurvey() throws ParseException {
        return surveysValidator.getFirstValidSurvey();
    }

    private void showSurvey(@NonNull final Survey survey) {
        if (canShowSurveys()) {
            startSurveyActivity(survey);
        }
    }

    private void startSurveyActivity(@NonNull final Survey survey) {
        AutoShowingManager.getInstance().showSurvey(survey);
    }

    boolean hasRespondToSurvey(String token) {
        Survey survey = getSurvey(token);
        if (survey != null) {
            return survey.isAnswered();
        } else {
            InstabugSDKLogger.e(Constants.LOG_TAG, "No survey with token:" + token + " was found.");
            return false;
        }
    }

    public void updateDismissedSurveysSessionCount() {
        List<Survey> validSurveys = SurveysCacheManager.getSurveys();
        for (Survey survey : validSurveys) {
            if (survey.isCancelled() && survey.shouldShowAgain()) {
                survey.incrementSessionCount();
                SurveysCacheManager.updateSessions(survey);
            }
        }
    }

    void resolveCountryInfo(CountryInfo countryInfo, boolean forceResolve) {
        try {
            String countryInfoString = SurveysSettings.getCountryInfo();
            long ttl = SurveysSettings.DEFAULT_COUNTRY_RESOLVER_INTERVAL;
            if (countryInfoString != null && !countryInfoString.trim().isEmpty()) {
                countryInfo.fromJson(countryInfoString);
                ttl = countryInfo.getTtl();
            }
            if (TimeUtils.currentTimeMillis() - SurveysSettings.getCountyInfoLastFetch() > TimeUnit.DAYS.toMillis(ttl) || forceResolve) {
                // Fetch and update last fetch time
                if (context != null && context.get() != null) {
                    countryInfoResolver.resolveCountryCode(context.get());
                }
            } else {
                // Use the persisted info
                onSuccess(countryInfo);
            }
        } catch (JSONException e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Can't resolve country info due to: " + e.getMessage());
        }

    }

    synchronized void showSurveyWithId(long surveyId) {
        if (SurveysCacheManager.getSurveyById(surveyId) != null)
            showSurvey(SurveysCacheManager.getSurveyById(surveyId));
    }

    List<com.instabug.survey.Survey> getAvailableSurveys() {
        return surveysValidator.getValidSurveys();
    }

    @Override
    public synchronized void onTimeEventTrigger(@NonNull Survey survey) {
        showSurvey(survey);
    }

    @Override
    public synchronized void onUserEventTrigger(@NonNull Survey survey) {
        showSurvey(survey);
    }

    synchronized void release() {
        unregisterSurveysTriggerEvents();
        //Clear showing flag with every session end
        AutoShowingManager.getInstance().setAnnouncementShown(false);
        AutoShowingManager.getInstance().setSurveyShown(false);
        AutoShowingManager.getInstance().release();
        if (singleton != null) {
            singleton = null;
        }
    }

    @Override
    public void onSuccess(CountryInfo countryInfo) {
        try {
            SurveysSettings.setCountryInfo(countryInfo.toJson());
            AnnouncementsSettings.setCountryInfo(countryInfo.toJson());
        } catch (JSONException e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Can't update country info due to: " + e.getMessage());
        }
    }

    @Override
    public void onError(Throwable error) {
        InstabugSDKLogger.e(Constants.LOG_TAG, "Can't resolve country info due to: " + error.getMessage());
    }

    @VisibleForTesting
    static synchronized void releaseInstance() {
        singleton = null;
    }

    void notifyLogout() {
        UserManagerWrapper.getUUIDAsync(new InstabugDBInsertionListener<String>() {
            @Override
            public void onDataInserted(String uuid) {
                // Called from a BG thread
                List<Survey> cachedSurveys = SurveysCacheManager.getSurveys();
                if (cachedSurveys != null && !cachedSurveys.isEmpty()) {
                    UserInteractionCacheManager.insertUserInteractions(cachedSurveys, uuid);
                    SurveysCacheManager.resetSurveyUserInteraction(cachedSurveys);
                }
            }
        });
    }

    /**
     * This method to when user logged in, trigger merge user interactions into survey
     */
    void notifyUserLoggedIn() {
        PoolProvider.postIOTask(new Runnable() {
            @Override
            public void run() {
                List<Survey> cachedSurveys = SurveysCacheManager.getSurveys();
                if (cachedSurveys != null && !cachedSurveys.isEmpty()) {
                    mergeUserInteraction(cachedSurveys);
                }
            }
        });
    }

    public void notifyAppVersionChanged() {
        PoolProvider.postIOTask(() -> {
            Context ctx = context.get();
            if (ctx == null) return;
            surveysValidator = new SurveysValidator(this,
                    InstabugDeviceProperties.getAppVersionName(ctx), // eg. 1.0.0-beta
                    DeviceStateProvider.getAppVersion(ctx)); // eg. 1.0.0-beta (3)
            startAutomaticSurveyTrigger();
        });
    }
}