package com.instabug.survey.ui.survey.text;

import android.os.Bundle;
import android.text.Editable;
import android.text.TextWatcher;
import android.view.View;
import android.widget.EditText;
import android.widget.RelativeLayout;
import android.widget.TextView;

import androidx.annotation.CallSuper;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;
import androidx.core.view.ViewCompat;

import com.instabug.survey.R;
import com.instabug.survey.models.Question;
import com.instabug.survey.ui.survey.QuestionFragment;
import com.instabug.survey.ui.survey.SurveyFragmentCallbacks;
import com.instabug.survey.utils.KeyboardUtils;

import org.jetbrains.annotations.NotNull;

/**
 * Created by mohamedzakaria on 7/19/17.
 */

public abstract class TextQuestionAbstractFragment extends QuestionFragment implements TextWatcher {
    @Nullable
    protected EditText answerEditText;
    @Nullable
    private Runnable startTextWatcherRunnable;

    @Override
    protected int getLayout() {
        return R.layout.instabug_dialog_text_survey;
    }

    public static TextQuestionAbstractFragment newInstance(boolean shouldChangeContainerHeight, Question question, SurveyFragmentCallbacks surveyFragmentCallbacks) {
        Bundle args = new Bundle();
        args.putSerializable(KEY_QUESTION_ARGUMENT, question);
        args.putBoolean(KEY_SHOULD_CHANGE_CONTAINER_HEIGHT_ARGUMENT, shouldChangeContainerHeight);
        TextQuestionAbstractFragment fragment = new TextQuestionFragment();
        fragment.setArguments(args);
        fragment.setSurveyFragmentCallbacks(surveyFragmentCallbacks);
        return fragment;
    }

    @Override
    public void onCreate(@Nullable Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setRetainInstance(true);
        if (getArguments() != null) {
            question = (Question) getArguments().getSerializable(KEY_QUESTION_ARGUMENT);
        }
    }

    @Override
    @CallSuper
    protected void initViews(View rootView, @Nullable Bundle savedInstanceState) {
        super.initViews(rootView, savedInstanceState);
        questionTextView = (TextView) rootView.findViewById(R.id.instabug_text_view_question);
        answerEditText = (EditText) rootView.findViewById(R.id.instabug_edit_text_answer);
        EditText editText = answerEditText;
        if (editText != null) {
            editText.getLayoutParams().height = getResources().getDimensionPixelSize(R.dimen.question_answer_text_height);
            ((RelativeLayout.LayoutParams) editText.getLayoutParams()).bottomMargin = 10;
            handleTextViewMaxLines();
        }
    }

    @Override
    public void onViewCreated(@NotNull View view, @Nullable Bundle savedInstanceState) {
        super.onViewCreated(view, savedInstanceState);
        view.setFocusableInTouchMode(true);
        if (question != null) showTextQuestion(question);
    }

    public void showTextQuestion(Question question) {
        EditText editText = answerEditText;
        TextView textView = questionTextView;
        if (textView == null || editText == null) return;
        String questionTitle = null;
        if (question.getTitle() != null) {
            questionTitle = getQuestionTitle(question.getTitle());
        }
        if (questionTitle != null) {
            textView.setText(questionTitle);
            ViewCompat.setLabelFor(textView, editText.getId());
        }
        editText.setHint(getLocalizedString(R.string.instabug_str_hint_enter_your_answer));
        startTextWatcherRunnable = new Runnable() {
            @Override
            public void run() {
                editText.addTextChangedListener(TextQuestionAbstractFragment.this);
            }
        };
        editText.postDelayed(startTextWatcherRunnable, 200);
        handleSelectedAnswer(question);
    }

    protected String getQuestionTitle(String questionTitle) {
        return questionTitle;
    }

    private void handleSelectedAnswer(Question question) {
        if (question.getAnswer() != null && !question.getAnswer().isEmpty() && answerEditText != null) {
            answerEditText.setText(question.getAnswer());
        }
    }

    @Nullable
    public String getAnswer() {
        if (answerEditText != null && !answerEditText.getText().toString().trim().equals("")) {
            return answerEditText.getText().toString();
        } else {
            return null;
        }
    }

    public void showKeyboard() {
        if (getActivity() != null && answerEditText != null) {
            answerEditText.requestFocus();
            KeyboardUtils.showKeyboard(getActivity(), answerEditText);
        }
    }

    @Override
    public void onDestroyView() {
        super.onDestroyView();
        EditText editText = answerEditText;
        if (editText != null) {
            editText.removeTextChangedListener(this);
            if (startTextWatcherRunnable != null) {
                editText.removeCallbacks(startTextWatcherRunnable);
                startTextWatcherRunnable = null;
                answerEditText = null;
            }
        }
    }

    @Override
    public void beforeTextChanged(CharSequence s, int start, int count, int after) {

    }

    @Override
    public void onTextChanged(CharSequence s, int start, int before, int count) {
    }

    @Override
    public void afterTextChanged(Editable s) {
        if (question == null) return;
        question.setAnswer(s.toString());
        if (surveyFragmentCallbacks != null)
            surveyFragmentCallbacks.onAnswerTextQuestion(question);
    }

    @Override
    public void onDestroy() {
        surveyFragmentCallbacks = null;
        super.onDestroy();
    }

    @VisibleForTesting
    @Nullable
    Question getQuestion() {
        return question;
    }
}