package com.instabug.survey.ui.survey;

import static android.view.View.GONE;
import static android.view.View.INVISIBLE;
import static com.instabug.survey.models.Question.Type.TEXT;

import android.content.Context;
import android.graphics.Color;
import android.os.Bundle;
import android.os.SystemClock;
import android.text.TextUtils;
import android.view.KeyEvent;
import android.view.View;
import android.widget.Button;
import android.widget.LinearLayout;

import androidx.annotation.CallSuper;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;
import androidx.core.content.ContextCompat;
import androidx.fragment.app.Fragment;
import androidx.viewpager.widget.ViewPager;

import com.instabug.library.Feature;
import com.instabug.library.IBGFeature;
import com.instabug.library.Instabug;
import com.instabug.library.InstabugColorTheme;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.core.ui.InstabugBaseFragment;
import com.instabug.library.settings.SettingsManager;
import com.instabug.library.ui.custom.InstabugViewPager;
import com.instabug.library.util.AccessibilityUtils;
import com.instabug.library.util.DisplayUtils;
import com.instabug.library.util.DrawableUtils;
import com.instabug.library.util.LocaleHelper;
import com.instabug.survey.R;
import com.instabug.survey.models.Question;
import com.instabug.survey.models.Survey;
import com.instabug.survey.models.SurveyTypes;
import com.instabug.survey.settings.SurveysSettings;
import com.instabug.survey.ui.SurveyActivity;
import com.instabug.survey.ui.SurveyActivityCallback;
import com.instabug.survey.ui.survey.adapter.QuestionsPagerAdapter;
import com.instabug.survey.ui.survey.mcq.MCQQuestionAbstractFragment;
import com.instabug.survey.ui.survey.nps.NPSQuestionFragment;
import com.instabug.survey.ui.survey.rateus.RateUsAbstractFragment;
import com.instabug.survey.ui.survey.rateus.RateUsFragment;
import com.instabug.survey.ui.survey.starrating.StarRatingQuestionAbstractFragment;
import com.instabug.survey.ui.survey.text.TextQuestionAbstractFragment;
import com.instabug.survey.ui.survey.text.customized.CustomizedTextQuestionFragment;
import com.instabug.survey.utils.KeyboardUtils;
import com.instabug.survey.utils.PlayStoreUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * @author mesbah.
 */

public abstract class SurveyAbstractFragment extends InstabugBaseFragment<SurveyFragmentPresenter> implements
        SurveyFragmentContract.View, View.OnClickListener, SurveyFragmentCallbacks {

    protected static final String KEY_SURVEY_ARGUMENT = "survey";
    protected static final String KEY_SHOULD_SHOW_KEYBOARD = "should_show_keyboard";
    @Nullable
    Survey survey;
    @Nullable
    protected Button submitButton;
    @Nullable
    protected InstabugViewPager questionsPager;
    @Nullable
    private QuestionsPagerAdapter questionsPagerAdapter;

    protected int currentQuestionPosition = -1;
    private String CURRENT_QUESTION_POSITION = "CURRENT_QUESTION_POSITION";
    @Nullable
    private SurveyActivityCallback surveyActivityCallback;
    private boolean shouldShowKeyboard = false;

    private long mLastClickTime;
    protected List<QuestionAbstractFragment> preparedFragments = new ArrayList<>();

    @Override
    public void onAttach(Context context) {
        super.onAttach(context);
        if (getActivity() instanceof SurveyActivity) {
            try {
                surveyActivityCallback = (SurveyActivityCallback) getActivity();
            } catch (Exception e) {
                throw new RuntimeException("Must implement SurveyActivityCallback ");
            }
        }
    }

    @Override
    public void onDetach() {
        surveyActivityCallback = null;
        super.onDetach();
    }

    @Override
    protected int getLayout() {
        return R.layout.instabug_dialog_survey;
    }

    @Override
    public void onCreate(@Nullable Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setRetainInstance(true);
        if (getArguments() != null) {
            survey = (Survey) getArguments().getSerializable(KEY_SURVEY_ARGUMENT);
            shouldShowKeyboard = getArguments().getBoolean(KEY_SHOULD_SHOW_KEYBOARD);
        }
        if (survey != null) presenter = new SurveyFragmentPresenter(this, survey);
    }

    protected static Bundle getFragmentArgs(Survey survey, boolean shouldShowKeyboard) {
        Bundle args = new Bundle();
        args.putSerializable(KEY_SURVEY_ARGUMENT, survey);
        args.putBoolean(KEY_SHOULD_SHOW_KEYBOARD, shouldShowKeyboard);
        return args;
    }

    @Override
    @CallSuper
    protected void initViews(View rootView, @Nullable Bundle savedInstanceState) {
        rootView.setOnKeyListener(new View.OnKeyListener() {
            @Override
            public boolean onKey(View v, int keyCode, KeyEvent event) {
                return keyCode == KeyEvent.KEYCODE_BACK;
            }
        });
        // submit button
        submitButton = rootView.findViewById(R.id.instabug_btn_submit);
        questionsPager = (InstabugViewPager) findViewById(R.id.instabug_survey_pager);
        if (submitButton != null) {
            submitButton.setOnClickListener(this);
        }

        if (survey == null || survey.getQuestions() == null) return;
        InstabugViewPager viewPager = questionsPager;
        if (viewPager != null) {
            viewPager.setSwipeable(false);
            viewPager.setOffscreenPageLimit(survey.getQuestions().size());
            // back button

            if (getActivity() == null) return;

            // handle RTL
            if (LocaleHelper.isRTL(getActivity())) {
                viewPager.setRotation(180);
            }
        }
    }

    @Override
    public void onViewCreated(@NonNull View view, @Nullable Bundle savedInstanceState) {
        super.onViewCreated(view, savedInstanceState);
        view.setFocusableInTouchMode(true);
        SurveyFragmentPresenter currentPresenter = presenter;
        if (currentPresenter != null) {
            if (shouldHandlePowerByFooter()) {
                currentPresenter.handlePowerByFooter();
            }
            currentPresenter.showSurvey();
        }
        handleSubmitButtonState(savedInstanceState);
    }

    protected boolean shouldHandlePowerByFooter() {
        return true;
    }

    private void handleSubmitButtonState(@Nullable Bundle savedInstanceState) {
        if (survey == null || presenter == null || questionsPager == null) return;

        if (savedInstanceState != null) {
            if (savedInstanceState.getInt(CURRENT_QUESTION_POSITION) != -1) {
                currentQuestionPosition = savedInstanceState.getInt(CURRENT_QUESTION_POSITION);
                boolean hasValidAnswer = presenter.hasValidAnswer(survey, currentQuestionPosition);
                setSubmitButtonEnabled(hasValidAnswer);
            }
        } else {
            currentQuestionPosition = questionsPager.getCurrentItem();
            boolean hasValidAnswer = presenter.hasValidAnswer(survey, currentQuestionPosition);
            setSubmitButtonEnabled(hasValidAnswer);
        }
    }

    @Override
    public void showSurvey(final Survey survey) {
        Button button = submitButton;
        InstabugViewPager viewPager = questionsPager;
        if (button == null || viewPager == null) return;
        preparedFragments = getPreparedFragments(survey);
        questionsPagerAdapter = new QuestionsPagerAdapter(getChildFragmentManager(),
                preparedFragments);
        viewPager.addOnPageChangeListener(new ViewPager.SimpleOnPageChangeListener() {
            @Override
            public void onPageSelected(int position) {
                Fragment selectedFragment = preparedFragments.get(position);
                if (selectedFragment instanceof RateUsAbstractFragment) {
                    ((RateUsAbstractFragment) selectedFragment).showRateUsQuestion();
                }
                super.onPageSelected(position);
            }
        });
        viewPager.setOffscreenPageLimit(0);
        viewPager.setAdapter(questionsPagerAdapter);
        currentQuestionPosition = 0;

        if (questionsPagerAdapter.getCount() > 1 && survey.getType() != SurveyTypes.STORE_RATING) {

            if (isLastQuestionInNPSSurvey()) {
                button.setText(R.string.instabug_str_action_submit);
            } else {
                button.setText(R.string.instabug_str_survey_next);
            }

            setQuestionsIndicator(0, survey.getQuestions().size());

            viewPager.addOnPageChangeListener(new ViewPager.OnPageChangeListener() {
                @Override
                public void onPageScrolled(int position, float positionOffset,
                                           int positionOffsetPixels) {

                }

                @Override
                public void onPageSelected(final int position) {

                    currentQuestionPosition = position;
                    if (getActivity() != null && getActivity() instanceof SurveyActivityCallback) {
                        ((SurveyActivityCallback) getActivity()).onPageSelected(position);
                    }
                    handleOnPageSelected(position, survey);

                    handleKeyboardVisibility(position);

                    handlePbiViewVisibility();

                    handleFragmentContainerHeight(position);

                    focusAccessibilityOnSelectedPage(position);
                }

                @Override
                public void onPageScrollStateChanged(int state) {

                }
            });

        } else {
            hideProgressbar(GONE);
        }

        if (survey.getType() == SurveyTypes.STORE_RATING ||
                (survey.getQuestions().get(0).getAnswer() != null &&
                        !survey.getQuestions().get(0).getAnswer().isEmpty())) {
            setSubmitButtonEnabled(true);
        } else {
            setSubmitButtonEnabled(false);
        }
    }

    protected void handleFragmentContainerHeight(int position) {
    }

    protected void handlePbiViewVisibility() {
    }

    protected void focusAccessibilityOnSelectedPage(int position) {
        if (AccessibilityUtils.isTalkbackEnabled() && questionsPagerAdapter != null) {
            QuestionAbstractFragment questionAbstractFragment = questionsPagerAdapter.getItem(position);
            questionAbstractFragment.focusAccessibilityOnView(questionAbstractFragment.questionTextView);
        }
    }

    @CallSuper
    public void handleOnPageSelected(int position, Survey survey) {
        Button button = submitButton;
        if (button != null) {
            setQuestionsIndicator(position, survey.getQuestions().size());
            if (!survey.isNPSSurvey()) {
                if (isFirstQuestion()) {
                    // First question fragment
                    button.setText(R.string.instabug_str_survey_next);
                } else if (isLastQuestion()) {
                    button.setText(R.string.instabug_str_action_submit);
                } else {
                    button.setText(R.string.instabug_str_survey_next);
                }

                final String currentQuestionAnswer = survey.getQuestions().get(position).getAnswer();
                final boolean isAnswerNullOrEmpty = currentQuestionAnswer == null || currentQuestionAnswer.trim().isEmpty();
                setSubmitButtonEnabled(!isAnswerNullOrEmpty);

            } else if (survey.isNPSSurvey()) {
                if (isLastQuestion()) {
                    // Last question fragment (Optional App Store Rating)
                    handleRateUsButtonState();
                } else if (isFirstQuestion()) {
                    // NPS Question
                    button.setText(R.string.instabug_str_survey_next);
                } else {
                    button.setVisibility(View.VISIBLE);
                    button.setText(R.string.instabug_str_action_submit);
                    setSubmitButtonEnabled(true);
                }
            }
        }
    }

    private void handleKeyboardVisibility(final int position) {
        if (questionsPager == null) return;

        questionsPager.postDelayed(new Runnable() {
            @Override
            public void run() {
                if (questionsPagerAdapter == null || survey == null) return;

                if (questionsPagerAdapter.getCount() > position) {
                    boolean isAccessibilityEnabled = AccessibilityUtils.isTalkbackEnabled();

                    QuestionAbstractFragment item = questionsPagerAdapter.getItem(position);
                    if (item instanceof TextQuestionAbstractFragment && !isAccessibilityEnabled) {
                        ((TextQuestionAbstractFragment) item).showKeyboard();
                        return;
                    }

                    if (survey.isStoreRatingSurvey() &&
                            survey.getQuestions().size() > position &&
                            survey.getQuestions().get(position).getType() == TEXT &&
                            shouldShowKeyboard &&
                            !isAccessibilityEnabled) {
                        ((TextQuestionAbstractFragment) questionsPagerAdapter.getItem(position)).showKeyboard();
                        shouldShowKeyboard = false;
                    } else {
                        if (getActivity() != null)
                            KeyboardUtils.hide(getActivity());
                    }
                }
            }
        }, 100);
    }

    @Override
    public void onResume() {
        super.onResume();
        if (questionsPager == null) return;
        if (shouldHandleKeyboardVisibility()) {
            handleKeyboardVisibility(questionsPager.getCurrentItem());
        }
    }

    protected abstract boolean shouldHandleKeyboardVisibility();

    @Override
    public void onStart() {
        super.onStart();
        checkViewsVisibility();
    }

    private void checkViewsVisibility() {
        if (submitButton != null && submitButton.getVisibility() == INVISIBLE) {
            submitButton.setVisibility(View.VISIBLE);
        }
        if (questionsPager != null && questionsPager.getVisibility() == INVISIBLE) {
            questionsPager.setVisibility(View.VISIBLE);
        }
    }

    @Override
    public void onSaveInstanceState(Bundle outState) {
        outState.putInt(CURRENT_QUESTION_POSITION, currentQuestionPosition);
        super.onSaveInstanceState(outState);
    }

    /**
     * Preparing the question fragments of each survey
     * NPS surveys had a implicit question type that not retrieved from the dashboard for showing thanks msg with app rating
     */
    @VisibleForTesting
    List<QuestionAbstractFragment> getPreparedFragments(Survey survey) {
        List<QuestionAbstractFragment> fragmentList = new ArrayList<QuestionAbstractFragment>();
        for (int index = 0; index < survey.getQuestions().size(); index++) {
            Question question = survey.getQuestions().get(index);
            if (survey.isNPSSurvey() && !question.isEnabled()) {
                continue;
            }
            boolean shouldChangeContainerHeight = index == 0;
            if (question.getType() == Question.Type.MCQ) {
                fragmentList.add(MCQQuestionAbstractFragment.newInstance(shouldChangeContainerHeight, question, this));
            } else if (question.getType() == TEXT) {
                shouldChangeContainerHeight = survey.getType() == SurveyTypes.STORE_RATING || shouldChangeContainerHeight;
                if (SurveysSettings.isCustomizationEnabled()) {
                    fragmentList.add(CustomizedTextQuestionFragment.newInstance(shouldChangeContainerHeight, question, this));
                } else {
                    fragmentList.add(TextQuestionAbstractFragment.newInstance(shouldChangeContainerHeight, question, this));
                }
            } else if (question.getType() == Question.Type.STAR_RATE) {
                fragmentList.add(StarRatingQuestionAbstractFragment.newInstance(shouldChangeContainerHeight, question, this));
            } else if (question.getType() == Question.Type.NPS) {
                hideProgressbar(GONE);
                fragmentList.add(NPSQuestionFragment.newInstance(shouldChangeContainerHeight, question, this));
            }
        }
        if (survey.isNPSSurvey()) {
            fragmentList.add(RateUsAbstractFragment.newInstance(survey, this));
        }
        return fragmentList;
    }

    @Override
    public void setSubmitButtonEnabled(boolean submitButtonEnabled) {
        Button button = submitButton;
        if (button == null) return;
        button.setEnabled(submitButtonEnabled);
        if (getActivity() == null) return;

        if (submitButtonEnabled) {
            // apply Verizon theme IBGSRV-7100
            if (SurveysSettings.isCustomizationEnabled() && survey != null && survey.getType() == SurveyTypes.STORE_RATING) {
                // customization enabled
                if (InstabugCore.getTheme() == InstabugColorTheme.InstabugColorThemeLight) {
                    DrawableUtils.setColor(button, Color.BLACK);
                    button.setTextColor(ContextCompat.getColor(getActivity(), android.R.color.white));
                } else {
                    DrawableUtils.setColor(button, Color.WHITE);
                    button.setTextColor(ContextCompat.getColor(getActivity(), android.R.color.black));
                }
            } else {
                DrawableUtils.setColor(button, getSubmitButtonColor());
                button.setTextColor(ContextCompat.getColor(getActivity(), android.R.color.white));
            }
        } else {
            if (InstabugCore.getTheme() == InstabugColorTheme.InstabugColorThemeLight) {
                DrawableUtils.setColor(button, ContextCompat.getColor(getActivity(),
                        R.color.survey_btn_disabled_color_light));
            } else {
                if (SurveysSettings.isCustomizationEnabled() && survey != null && survey.getType() == SurveyTypes.STORE_RATING) {
                    button.setTextColor(Color.WHITE);
                    DrawableUtils.setColor(button, Color.parseColor("#d9d9d9"));
                } else {
                    button.setTextColor(ContextCompat.getColor(getActivity(), R.color
                            .survey_btn_txt_color_dark));
                    DrawableUtils.setColor(button, ContextCompat.getColor(getActivity(),
                            R.color.survey_btn_disabled_color_dark));
                }
            }
        }
    }

    protected abstract int getSubmitButtonColor();

    @VisibleForTesting
    void setQuestionsIndicator(int pos, int totalNumberOfQuetions) {
    }

    @Override
    public void onClick(View v) {
        int id = v.getId();
        if (id == R.id.instabug_btn_submit) {
            handleOnSubmitButtonClicked(v);
        } else if (id == R.id.instabug_ic_survey_close) {
            if (SystemClock.elapsedRealtime() - mLastClickTime < 1000) {
                return;
            }
            mLastClickTime = SystemClock.elapsedRealtime();
            handleCloseButtonClicked();
        }
    }

    private void handleOnSubmitButtonClicked(View v) {
        if (survey == null || questionsPagerAdapter == null)
            return;

        InstabugViewPager pager = questionsPager;
        if (pager != null) {
            int currentPagerPosition = pager.getCurrentItem();
            Fragment questionFragment
                    = getChildFragmentManager().findFragmentByTag("android:switcher:" + R.id
                    .instabug_survey_pager + ":" + currentPagerPosition);
            String currentQuestionAnswer = null;

            if (survey.isNPSSurvey()) {
                handleNpsSurvey(currentPagerPosition);
            } else {
                if (questionFragment != null) {
                    currentQuestionAnswer = ((QuestionAbstractFragment) questionFragment).getAnswer();
                }
                if (currentQuestionAnswer == null) {
                    if (isEmptyAnswers() && !survey.isStoreRatingSurvey()) {
                        return;
                    }
                } else {
                    handleKeyboardVisibility(currentPagerPosition + 1);
                    pager.postDelayed(new Runnable() {
                        @Override
                        public void run() {
                            pager.scrollForward(true);
                        }
                    }, 300);
                }

                // set current question answer
                if (survey == null || survey.getQuestions() == null) return;

                if (!survey.isStoreRatingSurvey() && survey.getQuestions().size() > currentPagerPosition) {
                    survey.getQuestions().get(currentPagerPosition).setAnswer(currentQuestionAnswer);
                }
            }

            // Final Question Fragment
            if (currentQuestionAnswer != null &&
                    currentPagerPosition >= questionsPagerAdapter.getCount() - 1) {// hide keyboard if visible
                handleSurveySubmit();
            }
        }
    }

    private void handleSurveyNext(int i) {
        handleKeyboardVisibility(i);
        if (questionsPager != null) {
            questionsPager.postDelayed(new Runnable() {
                @Override
                public void run() {
                    if (questionsPager != null)
                        questionsPager.scrollForward(true);
                }
            }, 300);
        }
    }

    private void handleNpsSurvey(int currentPagerPosition) {
        if (survey == null || surveyActivityCallback == null) return;

        if (isLastQuestion()) {
            if (survey.isAppStoreRatingEnabled()) {
                survey.addRateEvent();
                if (Instabug.getApplicationContext() == null) return;
                Instabug.willRedirectToStore();
                PlayStoreUtils.rateNow(Instabug.getApplicationContext());
            }
            surveyActivityCallback.submitSurvey(survey);
        } else {
            handleNpsQuestionSubmit(currentPagerPosition);
        }
    }

    private void handleSurveySubmit() {
        if (getActivity() == null || survey == null) return;

        SurveyActivityCallback callback = surveyActivityCallback;
        if (callback != null) {
            KeyboardUtils.hide(getActivity());
            hideProgressbar(INVISIBLE);
            hideBackButton();
            callback.submitSurvey(survey);
        }
    }

    private boolean isEmptyAnswers() {
        if (survey == null) return true;
        SurveyActivityCallback callback = surveyActivityCallback;
        if (callback != null) {
            if (!survey.isNPSSurvey()) {
                return true;
            } else {
                hideProgressbar(INVISIBLE);
                hideBackButton();
                callback.submitSurvey(survey);
            }
            return false;
        }
        return true;
    }

    private void handleNpsQuestionSubmit(int currentPagerPosition) {
        handleSurveyNext(currentPagerPosition);
    }

    protected void hideProgressbar(int visibility) {
    }

    protected abstract void hideBackButton();

    private void handleRateUsButtonState() {
        if (survey == null || submitButton == null || surveyActivityCallback == null)
            return;
        hideBackButton();
        Button button = submitButton;
        if (button != null) {
            if (survey.isAppStoreRatingEnabled() && SurveysSettings.isAppLive()) {
                if (survey.getRatingCTATitle() != null) {
                    button.setText(survey.getRatingCTATitle());
                } else {
                    button.setText(R.string.surveys_nps_btn_rate_us);
                }
            } else {
                button.setVisibility(GONE);
                SurveyActivityCallback callback = surveyActivityCallback;
                if (callback != null) {
                    // Auto dismiss the thank you fragment
                    surveyActivityCallback.submitSurvey(survey);
                }
            }
        }
    }

    private void handleCloseButtonClicked() {
        if (survey == null || questionsPager == null || surveyActivityCallback == null) return;
        if (isFirstQuestion()) {
            surveyActivityCallback.dismissSurvey(survey);
        } else {
            if (survey.isNPSSurvey() && survey.hasPositiveNpsAnswer()) {
                if (questionsPager.getAdapter() != null) {
                    questionsPager.setCurrentItem(questionsPager.getAdapter().getCount() > 2 ?
                            questionsPager.getCurrentItem() - 2 : questionsPager.getCurrentItem() - 1);
                }
            } else {
                questionsPager.scrollBackward(true);
            }
        }
    }

    protected boolean isFirstQuestion() {
        if (questionsPager == null) return false;
        return questionsPager.getCurrentItem() == 0;
    }

    protected boolean isLastQuestion() {
        if (questionsPager == null || questionsPagerAdapter == null) return false;
        return questionsPager.getCurrentItem() == questionsPagerAdapter.getCount() - 1;
    }

    @Override
    public void onAnswerMCQQuestion(Question answeredQuestion) {
        if (survey == null) return;
        if (survey.getQuestions() == null) return;

        survey.getQuestions().get(getQuestionPosition(answeredQuestion.getId())).setAnswer
                (answeredQuestion
                        .getAnswer());
        setSubmitButtonEnabled(true);
    }

    @Override
    public void onAnswerTextQuestion(Question answeredQuestion) {
        if (survey == null) return;
        if (survey.getQuestions() == null) return;

        survey.getQuestions().get(getQuestionPosition(answeredQuestion.getId())).setAnswer
                (answeredQuestion
                        .getAnswer());

        final String answer = answeredQuestion.getAnswer();
        final boolean isAnswerNullOrEmpty = answer == null || answer.trim().isEmpty();

        if (!survey.isNPSSurvey()) {
            setSubmitButtonEnabled(!isAnswerNullOrEmpty);
        }
    }

    private int getQuestionPosition(long id) {
        if (survey != null && survey.getQuestions() != null && survey.getQuestions().size() > 0) {
            for (int i = 0; i < survey.getQuestions().size(); i++) {
                if (survey.getQuestions().get(i).getId() == id)
                    return i;
            }
        }
        return 0;
    }

    @Override
    public void onAnswerRateQuestion(Question answeredQuestion) {
        if (survey == null) return;
        if (answeredQuestion.getAnswer() != null) {
            int stars = Integer.parseInt(answeredQuestion.getAnswer());
            if (stars >= 1) {
                setSubmitButtonEnabled(true);
                if (survey.getQuestions() == null) return;

                survey.getQuestions().get(getQuestionPosition(answeredQuestion.getId())).setAnswer
                        (answeredQuestion
                                .getAnswer());
            } else {
                setSubmitButtonEnabled(false);
            }
        } else {
            setSubmitButtonEnabled(false);
        }
    }

    public void onBackPressed() {
        if (questionsPager == null) return;
        Fragment fragment = preparedFragments.get(currentQuestionPosition);
        if (fragment instanceof RateUsFragment) {
            return;
        }
        questionsPager.scrollBackward(true);
    }

    @Override
    public void onAnswerNPSQuestion(Question answeredQuestion) {
        if (survey == null) return;
        if (survey.getQuestions() == null) return;

        survey.getQuestions().get(getQuestionPosition(answeredQuestion.getId())).setAnswer
                (answeredQuestion
                        .getAnswer());
        setSubmitButtonEnabled(true);
    }

    private boolean isLastQuestionInNPSSurvey() {
        if (survey == null) return false;
        if (questionsPagerAdapter == null) return false;

        if (!survey.isNPSSurvey()) {
            return false;
        }

        // Rate us page index
        int lastPageIndex = questionsPagerAdapter.getCount() - 1;

        // Check if the current position is right before the rate us page
        // Which is the last page in nps surveys
        return currentQuestionPosition == lastPageIndex - 1;
    }

    @Override
    public void handlePoweredByFooter() {
        if (InstabugCore.getFeatureState(IBGFeature.WHITE_LABELING) == Feature.State.ENABLED
                && !SettingsManager.getInstance().isCustomBrandingEnabled()) {
            //Hide the footer anyway.
            InstabugCore.handlePbiFooter(getView());
        } else if (submitButton != null) {
            InstabugCore.handlePbiFooter(getView());
            InstabugCore.handlePbiFooterThemeColor(
                    getView(),
                    R.color.pbi_footer_color_dark,
                    R.color.pbi_footer_color_light
            );

            ((LinearLayout.LayoutParams) submitButton.getLayoutParams())
                    .bottomMargin = DisplayUtils.dpToPxIntRounded(getResources(), 8);

            submitButton.requestLayout();
        }
    }

    @Override
    public void onSwipeRight() {
        InstabugViewPager viewPager = questionsPager;
        if (getContext() == null || survey == null
                || submitButton == null || viewPager == null)
            return;

        if (survey.isNPSSurvey()) {
            if (LocaleHelper.isRTL(getContext())) {
                scrollNpsForward();
            } else {
                if (viewPager.getCurrentItem() != 2) {
                    viewPager.setCurrentItem(viewPager.getCurrentItem() - 1, true);
                    hideBackButton();
                }
            }
        } else {
            viewPager.postDelayed(new Runnable() {
                @Override
                public void run() {
                    if (survey == null || getContext() == null || viewPager == null) return;

                    if (LocaleHelper.isRTL(getContext())) {
                        if (survey.getQuestions().get(currentQuestionPosition).getAnswer() != null
                                && !TextUtils.isEmpty(survey.getQuestions().get(currentQuestionPosition).getAnswer())) {
                            viewPager.scrollForward(true);
                        }
                    } else {
                        viewPager.scrollBackward(true);
                    }
                }
            }, 300);
        }
    }

    private void scrollNpsForward() {
        if (survey == null || submitButton == null || questionsPager == null)
            return;
        if (currentQuestionPosition == 0 && survey.getQuestions().get(0).getAnswer() != null) {
            questionsPager.setCurrentItem(questionsPager.getCurrentItem() + 1, true);
            submitButton.setText(R.string.instabug_str_action_submit);
        } else if (questionsPager.getCurrentItem() >= 1 || survey.getQuestions().get(0).getAnswer() == null) {
            // do nothing
        } else {
            questionsPager.setCurrentItem(1, true);
            showBackButton();
        }
    }

    protected abstract void showBackButton();

    @Override
    public void onSwipeLeft() {
        if (getContext() == null || survey == null)
            return;

        InstabugViewPager viewPager = questionsPager;
        if (viewPager != null)
            if (survey.isNPSSurvey()) {
                if (LocaleHelper.isRTL(getContext())) {
                    if (currentQuestionPosition == 1) {
                        // do nothing
                        viewPager.setCurrentItem(0, true);
                    }
                } else {
                    scrollNpsForward();
                }
            } else {
                viewPager.postDelayed(new Runnable() {
                    @Override
                    public void run() {
                        if (survey == null || getContext() == null) return;

                        if (LocaleHelper.isRTL(getContext())) {
                            viewPager.scrollBackward(true);
                        } else {
                            if (survey.getQuestions().get(currentQuestionPosition).getAnswer() != null
                                    && !TextUtils.isEmpty(survey.getQuestions().get(currentQuestionPosition).getAnswer())) {
                                viewPager.scrollForward(true);
                            }
                        }
                    }
                }, 200);
            }
    }
}