package com.instabug.survey.ui.custom;

import android.annotation.SuppressLint;
import android.annotation.TargetApi;
import android.content.Context;
import android.content.res.TypedArray;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.CornerPathEffect;
import android.graphics.Paint;
import android.graphics.Path;
import android.graphics.Rect;
import android.graphics.Typeface;
import android.os.Build;
import android.text.TextPaint;
import android.util.AttributeSet;
import android.view.MotionEvent;
import android.view.View;

import androidx.annotation.ColorInt;
import androidx.annotation.IntRange;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.core.view.ViewCompat;
import androidx.customview.widget.ExploreByTouchHelper;

import com.instabug.library.core.InstabugCore;
import com.instabug.library.util.AttrResolver;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.survey.Constants;
import com.instabug.survey.R;

import java.util.ArrayList;

/**
 * @author hossam.
 */
@SuppressLint("ERADICATE_FIELD_NOT_INITIALIZED")
public abstract class NpsAbstractView extends View implements ChildViewsProvider {

    private static final int SELECTION_VIEW_EDGE = 30;
    private static final int
            DESIRED_WIDTH = 500,
            DESIRED_HEIGHT = 280,
            PADDING_BOTTOM = 2;
    protected int width, height;
    protected int count = 11;
    protected int selected = -1;
    protected int selectedNumber = -1;
    private boolean onTouchUp = false;
    protected boolean isRtl = false;
    @Nullable
    private OnNpsSelectionListener onNpsSelectionListener;
    protected ArrayList<Rect> npsRects = new ArrayList<>();
    @ColorInt
    protected int
            circlesRectColor,
            borderColor,
            indicatorViewBackgroundColor,
            numbersColor,
            indicatorViewTextColor,
            indicatorViewCircleColor,
            selectedViewEdgeSize,
            rectLine,
            heightWithPaddingBottom,
            selectedRect;
    @NonNull
    protected Paint
            paintText,
            paintBorder,
            paintSelectionShape,
            paintSelectionShapeCircle,
            paintSelectionShapeText;
    @NonNull
    protected Path
            borderPath,
            selectionPath,
            scoreIndicatorPath;
    @NonNull
    protected CornerPathEffect
            cornerPathEffect,
            rectCornerPathEffect;

    protected float
            numbersTextSize,
            selectedNumbersTextSize,
            rectCornerPathEffectRadius;

    public NpsAbstractView(Context context) {
        super(context);
    }

    public NpsAbstractView(Context context, @Nullable AttributeSet attrs) {
        super(context, attrs);
        init(attrs);
    }

    public NpsAbstractView(Context context, @Nullable AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        init(attrs);
    }

    @TargetApi(Build.VERSION_CODES.LOLLIPOP)
    public NpsAbstractView(
            Context context, @Nullable AttributeSet attrs, int defStyleAttr, int defStyleRes) {
        super(context, attrs, defStyleAttr, defStyleRes);
        init(attrs);
    }

    public int getCirclesRectColor() {
        return circlesRectColor;
    }

    public void setCirclesRectColor(int circlesRectColor) {
        this.circlesRectColor = circlesRectColor;
    }

    public int getBorderColor() {
        return borderColor;
    }

    public void setBorderColor(@ColorInt int borderColor) {
        this.borderColor = borderColor;
    }

    public int getIndicatorViewBackgroundColor() {
        return indicatorViewBackgroundColor;
    }

    public void setIndicatorViewBackgroundColor(@ColorInt int indicatorViewBackgroundColor) {
        this.indicatorViewBackgroundColor = indicatorViewBackgroundColor;
    }

    public int getNumbersColor() {
        return numbersColor;
    }

    public void setNumbersColor(@ColorInt int numbersColor) {
        this.numbersColor = numbersColor;
    }

    public int getIndicatorViewTextColor() {
        return indicatorViewTextColor;
    }

    public void setIndicatorViewTextColor(int indicatorViewTextColor) {
        this.indicatorViewTextColor = indicatorViewTextColor;
    }

    public int getIndicatorViewCircleColor() {
        return indicatorViewCircleColor;
    }

    public void setIndicatorViewCircleColor(int indicatorViewCircleColor) {
        this.indicatorViewCircleColor = indicatorViewCircleColor;
    }

    protected void init(@Nullable AttributeSet attrs) {
        if (attrs == null) return;




        TypedArray typedArray = getContext().obtainStyledAttributes(attrs, R.styleable.NpsAbstractView);

        count = typedArray.getInt(
                R.styleable.NpsAbstractView_nps_count, 11);

        numbersTextSize = typedArray.getDimensionPixelSize(
                R.styleable.NpsAbstractView_nps_num_text_size, (int) pixelToDp(this.getContext(), 40));

        selectedViewEdgeSize = typedArray.getDimensionPixelSize(
                R.styleable.NpsAbstractView_nps_selected_view_edge_size, (int) pixelToDp(this.getContext(),
                        SELECTION_VIEW_EDGE));

        selectedNumbersTextSize = typedArray.getDimensionPixelSize(
                R.styleable.NpsAbstractView_nps_selected_num_text_size, (int) pixelToDp(this.getContext()
                        , 20));

        selectedRect = typedArray.getDimensionPixelSize(
                R.styleable.NpsAbstractView_nps_selected_rect_size, (int) pixelToDp(this.getContext()
                        , 100));

        rectCornerPathEffectRadius = typedArray.getDimensionPixelSize(
                R.styleable.NpsAbstractView_nps_rect_corners_radius, 2);

        if (count == 0) {
            count = 1;
        }

        setCirclesRectColor(AttrResolver.resolveAttributeColor(
                this.getContext(), R.attr.survey_nps_circles_container_background));
        setBorderColor(AttrResolver.resolveAttributeColor(
                this.getContext(), R.attr.survey_nps_circles_container_border_background));
        setNumbersColor(AttrResolver.resolveAttributeColor(
                this.getContext(), R.attr.survey_nps_numbers_color));
        if (!isInEditMode()) {
            setIndicatorViewBackgroundColor(InstabugCore.getPrimaryColor());
        }
        setIndicatorViewTextColor(Color.WHITE);
        setIndicatorViewCircleColor(Color.WHITE);
        typedArray.recycle();

        selectionPath = new Path();
        scoreIndicatorPath = new Path();
        borderPath = new Path();
        paintBorder = new Paint(Paint.ANTI_ALIAS_FLAG);
        paintText = new TextPaint(Paint.ANTI_ALIAS_FLAG);
        paintSelectionShape = new Paint(Paint.ANTI_ALIAS_FLAG);
        paintSelectionShapeCircle = new Paint(Paint.ANTI_ALIAS_FLAG);
        paintSelectionShapeText = new TextPaint(Paint.ANTI_ALIAS_FLAG);
        cornerPathEffect = new CornerPathEffect(dpToPixel(this.getContext(), 4f));
        rectCornerPathEffect = new CornerPathEffect(rectCornerPathEffectRadius);
        ExploreByTouchHelper exploreByTouchHelper = new CustomExploreByTouchHelper(new NPSVirtualViewsProvider(this));
        ViewCompat.setAccessibilityDelegate(this, exploreByTouchHelper);
        setOnHoverListener((view, event) -> exploreByTouchHelper.dispatchHoverEvent(event));
    }

    @Override
    protected void onMeasure(int widthMeasureSpec, int heightMeasureSpec) {
        super.onMeasure(widthMeasureSpec, heightMeasureSpec);
        int widthMode = MeasureSpec.getMode(widthMeasureSpec);
        int widthSize = MeasureSpec.getSize(widthMeasureSpec);
        int heightMode = MeasureSpec.getMode(heightMeasureSpec);
        int heightSize = MeasureSpec.getSize(heightMeasureSpec);

        if (widthMode == MeasureSpec.EXACTLY) {
            width = widthSize;
        } else if (widthMode == MeasureSpec.AT_MOST) {
            width = Math.min(DESIRED_WIDTH, widthSize);
        } else {
            width = DESIRED_WIDTH;
        }

        if (heightMode == MeasureSpec.EXACTLY) {
            height = heightSize;
        } else if (widthMode == MeasureSpec.AT_MOST) {
            height = Math.min(DESIRED_HEIGHT, heightSize);
        } else {
            height = DESIRED_HEIGHT;
        }

        float fontScale = getContext().getResources().getConfiguration().fontScale;
        int scaledHeight = Math.round(height * fontScale);

        width = Math.abs(width);
        height = Math.abs(scaledHeight);
        heightWithPaddingBottom = height - PADDING_BOTTOM;
        setMeasuredDimension(width, height);
    }

    //drawRect(left, top, right, bottom, paint)
    @Override
    protected void onDraw(Canvas canvas) {
        super.onDraw(canvas);


        if (android.os.Build.VERSION.SDK_INT >= android.os.Build.VERSION_CODES.JELLY_BEAN_MR1) {
            isRtl = getLayoutDirection() == View.LAYOUT_DIRECTION_RTL;
        }
        calculateNpsRects();

        drawRectOfTheCircles(canvas);
        if (shouldHaveBorder()) {
            drawBorder(canvas);
        }
        drawNumbers(canvas);
        if (onTouchUp) {
            drawScoreIndicator(canvas);
        } else {
            if (shouldDrawSelectionShape()) {
                drawSelectionShape(canvas);
            }
        }
    }

    protected abstract void calculateNpsRects();

    protected abstract boolean shouldHaveBorder();

    protected abstract boolean shouldDrawSelectionShape();

    protected abstract void drawRectOfTheCircles(Canvas canvas);

    protected abstract void drawNumbers(Canvas canvas);

    protected abstract void drawScoreIndicator(Canvas canvas);

    public static float dpToPixel(Context c, float dp) {
        float density = c.getResources().getDisplayMetrics().density;
        float pixel = dp * density;
        return pixel;
    }

    public static float pixelToDp(Context c, float px) {
        float density = c.getResources().getDisplayMetrics().density;
        float pixel = px / density;
        return pixel;
    }

    private void drawBorder(Canvas canvas) {
        paintBorder.setStrokeWidth(dpToPixel(this.getContext(), 2f));
        paintBorder.setStyle(Paint.Style.STROKE);
        paintBorder.setColor(getBorderColor());
        paintBorder.setPathEffect(rectCornerPathEffect);
        canvas.drawRect(selectedViewEdgeSize, (float) (Math.floor(heightWithPaddingBottom / 1.7))
                , getWidth() -
                        selectedViewEdgeSize, heightWithPaddingBottom, paintBorder);
    }

    private void drawSelectionShape(Canvas canvas) {
        if (selected != -1) {
            selectionPath.reset();
            paintSelectionShape.setColor(getIndicatorViewBackgroundColor());
            paintSelectionShape.setPathEffect(cornerPathEffect);
            float xl, xr, y, ignored;
            xl = npsRects.get(selected).left;
            xr = npsRects.get(selected).right;
            y = npsRects.get(selected).top;

            if (rectLine > selectedRect) {
                ignored = rectLine - selectedRect;
                xl += ignored / 2;
                xr -= ignored / 2;
            }

            float x2 = xl - selectedViewEdgeSize;
            float x3 = xr + selectedViewEdgeSize;

            selectionPath.moveTo(x2, y); //| 1
            selectionPath.lineTo(x2, ((float) heightWithPaddingBottom / 1.7f));//|2
            selectionPath.lineTo(xl, (((float) heightWithPaddingBottom / 1.7f) + (float)
                    selectedViewEdgeSize));//\
            selectionPath.lineTo(xl, (float) heightWithPaddingBottom);//__1
            selectionPath.lineTo(xr, (float) heightWithPaddingBottom);//__2
            selectionPath.lineTo(xr, (((float) heightWithPaddingBottom / 1.7f)
                    + selectedViewEdgeSize));//|
            selectionPath.lineTo(x3, ((float) heightWithPaddingBottom / 1.7f));///1
            selectionPath.lineTo(x3, 0f);//|2
            selectionPath.close();

            canvas.drawPath(selectionPath, paintSelectionShape);

            paintSelectionShapeText.setColor(getIndicatorViewTextColor());
            paintSelectionShapeText.setTextAlign(Paint.Align.CENTER);
            paintSelectionShapeText.setTextSize(selectedNumbersTextSize);
            paintSelectionShapeText.setTypeface(Typeface.defaultFromStyle(Typeface.BOLD));

            canvas.drawText(String.valueOf(selectedNumber), xl + (xr - xl) / 2f,
                    (((float) heightWithPaddingBottom / 1.7f / 1.5f))
                    , paintSelectionShapeText);
        }
    }

    @Override
    public boolean onTouchEvent(MotionEvent event) {
        float x, y;
        x = event.getX();
        y = event.getY();

        // get masked (not specific to a pointer) action
        int maskedAction = event.getActionMasked();

        switch (maskedAction) {

            case MotionEvent.ACTION_DOWN:
            case MotionEvent.ACTION_POINTER_DOWN: {
                onMoveEvent(x, y);
                InstabugSDKLogger.d(Constants.LOG_TAG, "onTouchEvent: NPS view touch rect : " + selected);
                break;
            }
            case MotionEvent.ACTION_MOVE: { // a pointer was moved
                onMoveEvent(x, y);
                break;
            }
            case MotionEvent.ACTION_UP:
            case MotionEvent.ACTION_POINTER_UP:
            case MotionEvent.ACTION_CANCEL: {
                onActionUp();
                onSelection();
                break;
            }
            default:
                break;
        }

        invalidate();

        return true;
    }

    private void onMoveEvent(float x, float y) {
        onTouchUp = false;
        if (selected != -1 && npsRects.get(selected).contains((int) x, (int) y)) {
            return;
        }

        for (int i = 0; i < count; i++) {
            if (npsRects.size() > i && npsRects.get(i).contains((int) x, (int) y)) {
                if (selected != i) {
                    if (isRtl) {
                        selectedNumber = 10 - i;
                    } else {
                        selectedNumber = i;
                    }
                    selected = i;
                    invalidate();
                }
                return;
            }
        }
    }

    private void onActionUp() {
        onTouchUp = true;
        invalidate();
    }

    private void onSelection() {
        if (onNpsSelectionListener != null) {
            onNpsSelectionListener.onScoreSelected(selectedNumber);
        }
    }

    public void setOnSelectionListener(@NonNull OnNpsSelectionListener onNpsSelectionListener) {
        this.onNpsSelectionListener = onNpsSelectionListener;
    }

    /**
     * gets the selected score value from 0 to 10 and -1 if there no selected value
     *
     * @return score as integer form scale 0 to 10
     */
    public int getScore() {
        return selected;
    }

    /**
     * gets the selected score value from 0 to 10 and -1 if there no selected value
     *
     * @return score as integer form scale 0 to 10
     */
    public void setScore(@IntRange(from = 0, to = 11) int score) {
        this.selected = score;
        this.selectedNumber = score;
        onTouchUp = true;
        postInvalidate();
    }

    public void setScore(@IntRange(from = 0, to = 11) int score, boolean notifyListener) {
        setScore(score);
        if (notifyListener) {
            onSelection();
        }
    }

    /**
     * Interface definition for a callback to be invoked
     * when a value from scale 0 to 10 has been selected
     */
    public interface OnNpsSelectionListener {
        /**
         * Called once the user selected a value from scale 0 to 10.
         *
         * @param score score value as integer form scale 0 to 10
         */
        void onScoreSelected(int score);
    }

    @Override
    public int childPositionAt(float x, float y) {
        for (int i = 0; i < npsRects.size(); i++) {
            Rect rect = npsRects.get(i);

            if (rect != null && rect.contains((int) x, (int) y))
                return i;
        }
        return ExploreByTouchHelper.INVALID_ID;
    }

    @Nullable
    @Override
    public Rect childBoundsAtPosition(int position) {
        int numberY = (int) Math.floor(heightWithPaddingBottom / 1.7);
        Rect numberRect = new Rect(npsRects.get(position));
        numberRect.top = numberY;
        return numberRect;
    }
}
