package com.instabug.survey.ui;

import android.animation.ValueAnimator;
import android.os.Build;
import android.os.Bundle;
import android.os.Handler;
import android.view.GestureDetector;
import android.view.MotionEvent;
import android.view.ViewGroup;
import android.widget.FrameLayout;
import android.widget.RelativeLayout;

import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;
import androidx.core.content.ContextCompat;
import androidx.fragment.app.Fragment;

import com.instabug.library._InstabugActivity;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.core.plugin.Plugin;
import com.instabug.library.core.ui.BaseFragmentActivity;
import com.instabug.library.util.AccessibilityUtils;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.StatusBarUtils;
import com.instabug.survey.Constants;
import com.instabug.survey.R;
import com.instabug.survey.SurveyPlugin;
import com.instabug.survey.SurveysManager;
import com.instabug.survey.common.AutoShowingManager;
import com.instabug.survey.models.Survey;
import com.instabug.survey.models.SurveyTypes;
import com.instabug.survey.settings.SurveysSettings;
import com.instabug.survey.ui.gestures.GestureListener;
import com.instabug.survey.ui.survey.QuestionAbstractFragment;
import com.instabug.survey.ui.survey.SurveyAbstractFragment;
import com.instabug.survey.ui.survey.rateus.RateUsFragment;
import com.instabug.survey.ui.survey.thankspage.ThanksAbstractFragment;
import com.instabug.survey.ui.survey.welcomepage.WelcomeFragment;
import com.instabug.survey.utils.SurveysUtils;
import com.instabug.survey.utils.ThemeResolver;
import com.instabug.survey.utils.ViewUtilsKt;

import java.util.List;

/**
 * Created by tarek on 3/26/17.
 */

public abstract class SurveyAbstractActivity extends BaseFragmentActivity<SurveyPresenter> implements
        SurveyActivityContract.View,
        _InstabugActivity, SurveyActivityCallback {

    protected int currentQuestionPosition = -1;
    public static final String KEY_SURVEY = "survey";
    private static final String TAG_THANKS_FRAGMENT = "THANKS_FRAGMENT";
    protected static final String VIEW_TYPE = "viewType";
    boolean isResumed = false;
    @Nullable
    private Handler handler;

    protected FrameLayout fragmentContainer;
    protected RelativeLayout activityContainer;
    @Nullable
    protected Survey survey;
    @Nullable
    private GestureDetector gestureDetector;
    @Nullable
    private Handler finishDelayHandler;
    @Nullable
    private Runnable finishDelayRunnable;

    @Override
    protected void onSaveInstanceState(Bundle outState) {
        super.onSaveInstanceState(outState);
        try {
            if (presenter != null && presenter.getState() != null)
                outState.putInt(VIEW_TYPE, presenter.getState().getValue());
        } catch (IllegalStateException e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Something went wrong while saving survey state", e);
        }
    }

    @Override
    public void onCreate(@Nullable final Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);

        setTheme(ThemeResolver.resolveTheme(InstabugCore.getTheme()));
        StatusBarUtils.setStatusBarForDialog(this);

        fragmentContainer = findViewById(R.id.instabug_fragment_container);
        activityContainer = findViewById(R.id.survey_activity_container);
        if(Build.VERSION.SDK_INT > Build.VERSION_CODES.UPSIDE_DOWN_CAKE) {
            ViewUtilsKt.addSystemWindowInsetToPadding(activityContainer, true, true, true, true);
        }

        presenter = new SurveyPresenter(this);
        if (getIntent() != null) {
            survey = (Survey) getIntent().getSerializableExtra(KEY_SURVEY);
        }
        if (survey != null) {
            handleFrameLayoutHeight(savedInstanceState);
        } else {
            InstabugSDKLogger.w(Constants.LOG_TAG, "survey activity will be finished the survey is null");
            finish();
            return;
        }
        // this fixes https://instabug.atlassian.net/browse/IBG-3232
        fragmentContainer.postDelayed(new Runnable() {
            @Override
            public void run() {
                if (InstabugCore.getStartedActivitiesCount() <= 1) {
                    InstabugSDKLogger.d(Constants.LOG_TAG, "Survey Error: StartedActivitiesCount <= 1");
                    finish();
                    return;
                }
                try {
                    if (!isFinishing()) {
                        if (isResumed) {
                            if (savedInstanceState == null) {
                                if (presenter != null
                                        && presenter.shouldShowWelcomeScreen()
                                        && survey != null && survey.getType() != SurveyTypes.STORE_RATING) {
                                    showWelcomeMsgFragment(survey);
                                } else {
                                    Survey passedSurvey = survey;
                                    if (passedSurvey != null) {
                                        SurveyNavigator.navigateToSurvey(getSupportFragmentManager(),
                                                passedSurvey);
                                    }
                                }
                            }
                        } else {
                            finish();
                        }
                    }
                } catch (Exception e) {
                    InstabugSDKLogger.e(Constants.LOG_TAG,
                            "Survey has not been shown due to this error: "
                                    + e.getMessage());
                    finish();
                }

            }

        }, 500);
    }

    protected abstract void handleFrameLayoutHeight(@Nullable Bundle savedInstanceState);

    private void showWelcomeMsgFragment(Survey survey) {
        navigateToFragment(WelcomeFragment.newInstance(survey));
    }

    protected void navigateToFragment(Fragment fragment) {
        navigateToFragment(fragment, R.anim.instabug_anim_flyin_from_bottom, R.anim
                .instabug_anim_flyout_to_bottom);
    }

    protected void navigateToFragment(Fragment fragment, int interAnimation, int exitAnimation) {
        getSupportFragmentManager().beginTransaction()
                .setCustomAnimations(interAnimation, exitAnimation)
                .replace(R.id.instabug_fragment_container, fragment)
                .commitAllowingStateLoss();
    }

    @Override
    protected void onResume() {
        super.onResume();
        InstabugCore.setPluginState(SurveyPlugin.class, Plugin.STATE_FOREGROUND);
        isResumed = true;
        //Hide if It exists => if the app had been moved to background
        //while thanks screen/ rate screen were showing, So we remove it with animation after returning to FG
        handledAutoDismissingIfResumed();

        AutoShowingManager.getInstance().setSurveyShown(true);
    }

    private void handledAutoDismissingIfResumed() {
        Fragment fragmentById = getSupportFragmentManager().findFragmentById(R.id.instabug_fragment_container);
        if (fragmentById instanceof SurveyAbstractFragment) {
            for (Fragment questionFragment : fragmentById.getChildFragmentManager().getFragments()) {
                if (questionFragment instanceof RateUsFragment && questionFragment.isVisible()) {
                    if (survey == null) {
                        hideFragmentImmediately(fragmentById);
                    } else if (!SurveysSettings.isAppLive() || !survey.isAppStoreRatingEnabled()) {
                        hideFragmentAfterDelay(fragmentById);
                    }
                    break;
                }
            }
        }

        if (getSupportFragmentManager() != null
                && getSupportFragmentManager().findFragmentByTag(TAG_THANKS_FRAGMENT) != null) {
            hideFragmentImmediately(getSupportFragmentManager().findFragmentByTag(TAG_THANKS_FRAGMENT));
        }
    }

    @Override
    protected void onPause() {
        isResumed = false;
        super.onPause();
        overridePendingTransition(0, 0);
    }


    @Override
    protected int getLayout() {
        return R.layout.instabug_survey_activity;
    }

    @Override
    protected void initViews() {
        //We don't need it
    }

    @Override
    public void finishSurvey(boolean showThanksFragment) {
        if (getSupportFragmentManager() == null) return;

        Fragment fragment = getSupportFragmentManager().findFragmentById(R.id
                .instabug_fragment_container);

        if (fragment != null) {
            getSupportFragmentManager().beginTransaction()
                    .setCustomAnimations(0, R.anim.instabug_anim_flyout_to_bottom)
                    .remove(fragment).commitAllowingStateLoss();
        }
        Handler handler = new Handler();
        if (showThanksFragment) {
            getSupportFragmentManager().beginTransaction()
                    .setCustomAnimations(0, 0)
                    .replace(R.id.instabug_fragment_container, ThanksAbstractFragment.newInstance(survey),
                            TAG_THANKS_FRAGMENT)
                    .commitAllowingStateLoss();
            if (!AccessibilityUtils.isTalkbackEnabled()) {
                finishDelayRunnable = () -> {
                    Fragment thanksFragment = getSupportFragmentManager().findFragmentByTag(TAG_THANKS_FRAGMENT);
                    if (thanksFragment != null)
                        hideFragmentAfterDelay(thanksFragment);
                };
                handler.postDelayed(finishDelayRunnable, 600);
            }
        } else {
            finishDelayRunnable = new Runnable() {
                @Override
                public void run() {
                    finish();
                }
            };
            handler.postDelayed(finishDelayRunnable, 300);
        }
        finishDelayHandler = handler;
        SurveysUtils.executeRunnableAfterShowingSurvey();
    }


    @Override
    public void finishNPSSurvey(boolean withoutDelay) {
        final Fragment fragment = getSupportFragmentManager().getFragments().get(getSupportFragmentManager().getFragments().size() - 1);
        if (withoutDelay) {
            hideFragmentImmediately(fragment);
        } else if (!AccessibilityUtils.isTalkbackEnabled()) {

            hideFragmentAfterDelay(fragment);
        }

    }

    @VisibleForTesting
    public void setSurveyPresenter(SurveyPresenter surveyPresenter) {
        presenter = surveyPresenter;
    }

    private void hideFragmentImmediately(Fragment fragment) {
        if (fragment != null) {
            getSupportFragmentManager().beginTransaction()
                    .setCustomAnimations(0, R.anim.instabug_anim_flyout_to_bottom)
                    .remove(fragment)
                    .commitAllowingStateLoss();
            new Handler().postDelayed(new Runnable() {
                @Override
                public void run() {
                    finish();
                    SurveysUtils.executeRunnableAfterShowingSurvey();
                }
            }, 400);
        }
    }

    private void hideFragmentAfterDelay(final Fragment fragment) {
        handler = new Handler();
        handler.postDelayed(new Runnable() {
            @Override
            public void run() {
                try {
                    hideFragmentImmediately(fragment);
                } catch (Exception e) {
                    getSupportFragmentManager().popBackStack();
                    finish();
                    InstabugSDKLogger.e(Constants.LOG_TAG,
                            "Fragment couldn't save it's state due to: " + e.getMessage());
                }
            }
        }, 3000);
    }

    @Override
    public void onBackPressed() {
        if (presenter != null) {
            presenter.handleOnBackPressedLogic();
        }
    }

    @Override
    protected void onDestroy() {
        if (finishDelayHandler != null) {
            if (finishDelayRunnable != null)
                finishDelayHandler.removeCallbacks(finishDelayRunnable);
            finishDelayHandler = null;
            finishDelayRunnable = null;
        }
        super.onDestroy();
        InstabugCore.setPluginState(SurveyPlugin.class, Plugin.STATE_BACKGROUND);
        if (SurveysManager.getInstance() != null) {
            SurveysManager.getInstance().registerSurveysTriggerEvents();
        }
        AutoShowingManager.getInstance().setSurveyShown(false);
    }

    @Override
    public void setFrameLayoutHeightWithAnimation(int heightInPx) {
        ValueAnimator anim = ValueAnimator.ofInt(fragmentContainer.getMeasuredHeight(), heightInPx);
        anim.addUpdateListener(new ValueAnimator.AnimatorUpdateListener() {
            @Override
            public void onAnimationUpdate(ValueAnimator valueAnimator) {
                int val = (Integer) valueAnimator.getAnimatedValue();
                ViewGroup.LayoutParams layoutParams = fragmentContainer.getLayoutParams();
                layoutParams.height = val;
                fragmentContainer.setLayoutParams(layoutParams);
            }
        });
        anim.setDuration(300);
        anim.start();
    }

    @Override
    public void setFrameLayoutHeightTo(int height) {
        RelativeLayout.LayoutParams params = (RelativeLayout.LayoutParams)
                fragmentContainer.getLayoutParams();
        params.height = height;
        fragmentContainer.setLayoutParams(params);
    }

    public void setLayoutHeightSecondary(ViewType type, boolean withAnimation) {
        if (presenter != null) {
            presenter.setLayoutHeightSecondary(type, withAnimation);
        }
    }

    public void setBackgroundTransparent(boolean shouldBeTransparent) {
        getWindow().getDecorView().setBackgroundColor(ContextCompat.getColor(this,
                shouldBeTransparent ? R.color.instabug_transparent_color
                        : R.color.instabug_dialog_bg_color));
    }

    public void handleCloseClicked(Survey survey) {
        if (presenter != null) {
            presenter.dismissSurvey(survey);
        }
    }

    @Nullable
    public ViewType getCurrentViewType() {
        if (presenter != null) {
            return presenter.getState();
        }
        return ViewType.PRIMARY;
    }

    @Override
    public boolean dispatchTouchEvent(MotionEvent event) {
        if (gestureDetector == null) {
            gestureDetector = new GestureDetector(this, new GestureListener(new GestureListener
                    .GesturesCallback() {
                @Override
                public void onSwipeRight() {
                    List<Fragment> list = getSupportFragmentManager().getFragments();
                    for (Fragment fragment : list) {
                        if (fragment instanceof SurveyAbstractFragment) {
                            ((SurveyAbstractFragment) fragment).onSwipeRight();
                            break;
                        }
                    }
                }

                @Override
                public void onSwipeLeft() {
                    List<Fragment> list = getSupportFragmentManager().getFragments();
                    for (Fragment fragment : list) {
                        if (fragment instanceof SurveyAbstractFragment) {
                            if (presenter != null) {
                                presenter.setLayoutHeightSecondary(ViewType.PRIMARY, true);
                            }
                            ((SurveyAbstractFragment) fragment).onSwipeLeft();
                            break;
                        }
                    }
                }

                @Override
                public void onHardSwipeUp() {

                }

                @Override
                public void onHardSwipeDown() {

                }

                @Override
                public void onClick() {
                    List<Fragment> list = getSupportFragmentManager().getFragments();
                    for (Fragment fragment : list) {
                        if (fragment instanceof QuestionAbstractFragment) {
                            if (((QuestionAbstractFragment) fragment).isPartial()) {
                                ((QuestionAbstractFragment) fragment).onClick();
                            }
                            break;
                        }
                    }
                }
            }));
        }
        gestureDetector.onTouchEvent(event);
        return super.dispatchTouchEvent(event);
    }

    @Override
    public void submitSurvey(Survey survey) {
        if (presenter != null) {
            presenter.submitSurvey(survey);
        }
    }

    @Override
    public void dismissSurvey(Survey survey) {
        if (presenter != null) {
            presenter.dismissSurvey(survey);
        }
    }

    @Override
    public void onPageSelected(int position) {
        currentQuestionPosition = position;
    }

    @Nullable
    public Survey getSurvey() {
        return survey;
    }

    @Nullable
    public ViewType getViewState() {
        if (presenter != null) {
            return presenter.getState();
        }
        return ViewType.PRIMARY;
    }
}

