package com.instabug.survey.configuration

import com.instabug.library.core.InstabugCore
import com.instabug.survey.di.ServiceLocator
import org.json.JSONObject

interface SurveysConfigurationHandler {
    /**
     * Handles feature request response part related to Surveys/Announcements
     *
     * @param configuration response of feature request
     * @return true if configuration string parsed correctly.
     * else otherwise
     */
    fun handleConfiguration(configuration: String?): Boolean
}

object SurveysConfigurationHandlerImpl : SurveysConfigurationHandler {
    private val configurationsProvider by lazy { ServiceLocator.configurationsProvider }
    override fun handleConfiguration(configuration: String?): Boolean {
        return try {
            configuration
                ?.let { JSONObject(configuration) }
                ?.updateSurveysAvailability()
                ?.updateAnnouncementsAvailability()
                ?.updateSurveysUsageExceeded()
                ?.let { true }
                ?: false
        } catch (exception: Exception) {
            InstabugCore.reportError(exception, "couldn't parse surveys feature flags ")
            false
        }

    }

    private fun JSONObject.updateSurveysAvailability(): JSONObject {
        val isSurveyAvailable = optBoolean("surveys", false)
        configurationsProvider.isSurveysAvailable = isSurveyAvailable
        return this
    }

    private fun JSONObject.updateAnnouncementsAvailability(): JSONObject {
        val isAnnouncementsAvailable = optBoolean("announcements", false)
        configurationsProvider.isAnnouncementsAvailable = isAnnouncementsAvailable

        return this
    }

    private fun JSONObject.updateSurveysUsageExceeded(): JSONObject {
        val isUsageExceeded = optJSONObject("product_usage_exceeded")
            ?.optBoolean("surveys", false) ?: false
        configurationsProvider.isSurveysUsageExceeded = isUsageExceeded
        return this
    }
}