package com.instabug.survey.ui.popup;

import android.app.Activity;
import android.graphics.Color;
import android.os.Build;
import android.view.LayoutInflater;
import android.view.View;
import android.widget.Button;
import android.widget.ImageView;
import android.widget.TextView;

import androidx.annotation.LayoutRes;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.appcompat.app.AlertDialog;
import androidx.appcompat.widget.AppCompatImageView;

import com.instabug.library.Feature;
import com.instabug.library.IBGFeature;
import com.instabug.library.InstabugColorTheme;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.util.AccessibilityUtils;
import com.instabug.survey.R;

import java.lang.ref.WeakReference;

public class CustomQuestionDialog {
    @Nullable
    private static void show(@NonNull final Activity activity,
                             @LayoutRes int viewRes,
                             @Nullable String message,
                             @Nullable String positiveCTA,
                             @Nullable String negativeCTA,
                             @Nullable final CustomDialogListener dialogListener) {

        if (activity.isFinishing()) return;

        final AlertDialog confirmationDialog = new AlertDialog.Builder(activity, android.R.style.Theme_Translucent_NoTitleBar).create();
        LayoutInflater inflater = activity.getLayoutInflater();
        View dialogView = inflater.inflate(viewRes, null);
        confirmationDialog.setView(dialogView);

        confirmationDialog.setCancelable(false);

        TextView question = dialogView.findViewById(R.id.instabug_text_view_question);
        Button positiveCTABtn = (Button) dialogView.findViewById(R.id.ib_survey_btn_yes);
        Button negativeCTABtn = (Button) dialogView.findViewById(R.id.ib_survey_btn_no);
        AppCompatImageView illustration = (AppCompatImageView) dialogView.findViewById(R.id.ib_survey_custom_dialog_illustration);
        ImageView close = (ImageView) dialogView.findViewById(R.id.survey_partial_close_btn);
        if (positiveCTABtn != null && negativeCTABtn != null && illustration != null) {
            applyTheme(positiveCTABtn, negativeCTABtn, illustration);
        }

        // set texts
        setTexts(question, message, positiveCTABtn, positiveCTA, negativeCTABtn, negativeCTA);

        handleOptionalCloseButtonAction(activity, dialogView);

        // set click Listeners
        setListeners(positiveCTABtn, negativeCTABtn, close, confirmationDialog, dialogListener);

        // handle instabug footer
        InstabugCore.handlePbiFooter(dialogView);
        if (InstabugCore.getFeatureState(IBGFeature.WHITE_LABELING) == Feature.State.ENABLED)
            InstabugCore.handlePbiFooterThemeColor(
                    dialogView,
                    R.color.pbi_footer_color_dark,
                    R.color.pbi_footer_color_light
            );

        confirmationDialog.show();
    }

    private static void setListeners(@Nullable Button positiveCTABtn,
                                     @Nullable Button negativeCTABtn,
                                     @Nullable ImageView close,
                                     final AlertDialog confirmationDialog,
                                     @Nullable final CustomDialogListener dialogListener) {
        // set listeners
        setClickListener(positiveCTABtn, new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                confirmationDialog.dismiss();
                if (dialogListener != null)
                    dialogListener.onPositiveActionCalled();
            }
        });

        setClickListener(negativeCTABtn, new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                confirmationDialog.dismiss();
                if (dialogListener != null)
                    dialogListener.onNegativeActionCalled();
            }
        });

        setClickListener(close, new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                if (dialogListener != null)
                    dialogListener.onDismissActionCalled();
            }
        });
    }

    private static void setClickListener(@Nullable View view, View.OnClickListener listener) {
        if (view != null) {
            view.setOnClickListener(listener);
        }
    }

    private static void setTexts(@Nullable TextView question, @Nullable String message,
                                 @Nullable Button positiveCTABtn, @Nullable String positiveCTA,
                                 @Nullable Button negativeCTABtn, @Nullable String negativeCTA) {
        if (question != null && message != null)
            question.setText(message);
        if (positiveCTABtn != null && positiveCTA != null)
            positiveCTABtn.setText(positiveCTA);
        if (negativeCTABtn != null && negativeCTA != null)
            negativeCTABtn.setText(negativeCTA);
    }

    public static class Builder {
        private final WeakReference<Activity> activity;
        @Nullable
        private String message;
        @LayoutRes
        private int layoutRes;
        @Nullable
        private String negativeCTA;
        @Nullable
        private String positiveCTA;
        @Nullable
        private CustomDialogListener dialogListener;

        public Builder(Activity activity) {
            this.activity = new WeakReference<Activity>(activity);
        }

        public Builder setMessage(@Nullable String message) {
            this.message = message;
            return this;
        }

        public Builder setPositiveCTA(String positiveCTA) {
            this.positiveCTA = positiveCTA;
            return this;
        }

        public Builder setNegativeCTA(String negativeCTA) {
            this.negativeCTA = negativeCTA;
            return this;
        }

        public Builder setLayoutRes(@LayoutRes int layoutRes) {
            this.layoutRes = layoutRes;
            return this;
        }

        public Builder setDialogListener(CustomDialogListener dialogListener) {
            this.dialogListener = dialogListener;
            return this;
        }

        public void build() {
            Activity activityRef = activity.get();
            if (activityRef != null) {
                show(activityRef, layoutRes, message, positiveCTA, negativeCTA, dialogListener);
            }
        }

    }

    private static void applyTheme(Button positiveCTABtn, Button negativeCTABtn, AppCompatImageView illustration) {

        if (InstabugCore.getTheme() == InstabugColorTheme.InstabugColorThemeLight) {
            positiveCTABtn.setTextColor(Color.WHITE);
            negativeCTABtn.setTextColor(Color.BLACK);

            positiveCTABtn.setBackgroundColor(Color.BLACK);
            negativeCTABtn.setBackgroundResource(R.drawable.ib_survey_bg_apprating_negative);
            if (illustration != null) {
                illustration.setImageResource(R.drawable.ib_ic_survey_apprating_light);
            }
        } else {
            positiveCTABtn.setTextColor(Color.BLACK);
            negativeCTABtn.setTextColor(Color.WHITE);

            positiveCTABtn.setBackgroundColor(Color.WHITE);
            negativeCTABtn.setBackgroundResource(R.drawable.ib_survey_bg_apprating_negative_dark);
            if (illustration != null) {
                illustration.setImageResource(R.drawable.ib_ic_survey_apprating_dark);
            }
        }
    }

    private static void handleOptionalCloseButtonAction(final Activity activity, View dialogView) {
        ImageView closeBtn = (ImageView) dialogView.findViewById(R.id.survey_partial_close_btn);
        if (closeBtn != null) {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP_MR1
                    && AccessibilityUtils.isTalkbackEnabled()) {
                closeBtn.setAccessibilityTraversalAfter(closeBtn.getRootView().getId());
            }
            closeBtn.setOnClickListener(new View.OnClickListener() {
                @Override
                public void onClick(View v) {
                    activity.finish();
                }
            });
        }
    }

    interface CustomDialogListener {
        void onPositiveActionCalled();

        void onNegativeActionCalled();

        void onDismissActionCalled();
    }
}
