package com.instabug.survey.cache;

import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.SurveyEntry.COLUMN_ANSWERED;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.SurveyEntry.COLUMN_ATTEMPT_COUNT;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.SurveyEntry.COLUMN_CONDITIONS_OPERATOR;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.SurveyEntry.COLUMN_CUSTOM_ATTRIBUTES;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.SurveyEntry.COLUMN_DISMISSED_AT;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.SurveyEntry.COLUMN_EVENT_INDEX;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.SurveyEntry.COLUMN_GOOGLE_PLAY_RATING;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.SurveyEntry.COLUMN_ID;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.SurveyEntry.COLUMN_IS_CANCELLED;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.SurveyEntry.COLUMN_PAUSED;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.SurveyEntry.COLUMN_QUESTIONS;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.SurveyEntry.COLUMN_SESSIONS_COUNT;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.SurveyEntry.COLUMN_SESSION_ID;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.SurveyEntry.COLUMN_SHOULD_SHOW_AGAIN;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.SurveyEntry.COLUMN_SHOWN_AT;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.SurveyEntry.COLUMN_SURVEY_CURRENT_LOCALE;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.SurveyEntry.COLUMN_SURVEY_IS_DISMISSIBLE;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.SurveyEntry.COLUMN_SURVEY_IS_LOCALIZED;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.SurveyEntry.COLUMN_SURVEY_LOCALES;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.SurveyEntry.COLUMN_SURVEY_STATE;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.SurveyEntry.COLUMN_SURVEY_TARGET;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.SurveyEntry.COLUMN_SURVEY_TRIGGER_EVENT;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.SurveyEntry.COLUMN_TARGET_AUDIENCES;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.SurveyEntry.COLUMN_THANKS_LIST;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.SurveyEntry.COLUMN_TITLE;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.SurveyEntry.COLUMN_TOKEN;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.SurveyEntry.COLUMN_TYPE;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.SurveyEntry.COLUMN_USER_EVENTS;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.SurveyEntry.TABLE_NAME;

import android.content.ContentValues;
import android.database.Cursor;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;
import androidx.annotation.WorkerThread;

import com.instabug.library.diagnostics.IBGDiagnostics;
import com.instabug.library.internal.storage.cache.db.DatabaseManager;
import com.instabug.library.internal.storage.cache.db.SQLiteDatabaseWrapper;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.survey.Constants;
import com.instabug.survey.common.models.Condition;
import com.instabug.survey.common.models.SyncingStatus;
import com.instabug.survey.common.models.Target;
import com.instabug.survey.models.Question;
import com.instabug.survey.models.Survey;
import com.instabug.survey.models.ThankYouItem;

import org.json.JSONArray;
import org.json.JSONException;

import java.util.ArrayList;
import java.util.List;

/**
 * @author hossam.
 */
public class SurveysDbHelper {
    /**
     * Inserting a survey in the database
     *
     * @param survey is the Survey object
     * @return the row ID of the newly inserted row, or -1 if an error occurred
     */
    @WorkerThread
    public static synchronized long insert(Survey survey) {
        // Gets the data repository in write mode
        SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
        try {
            db.beginTransaction();

            // Create a new map of values, where column names are the keys
            ContentValues values = new ContentValues();
            values.put(COLUMN_ID, survey.getId());
            values.put(COLUMN_TYPE, survey.getType());
            values.put(COLUMN_GOOGLE_PLAY_RATING, survey.isGooglePlayAppRating());
            values.put(COLUMN_TITLE, survey.getTitle());
            if (survey.getToken() != null)
                values.put(COLUMN_TOKEN, survey.getToken());
            values.put(COLUMN_CONDITIONS_OPERATOR, survey.getConditionsOperator());
            values.put(COLUMN_ANSWERED, survey.isAnswered() ? 1 : 0);
            values.put(COLUMN_DISMISSED_AT, survey.getDismissedAt());
            values.put(COLUMN_SHOWN_AT, survey.getShownAt());
            values.put(COLUMN_IS_CANCELLED, survey.isCancelled() ? 1 : 0);
            values.put(COLUMN_ATTEMPT_COUNT, survey.getAttemptCount());
            values.put(COLUMN_EVENT_INDEX, survey.getEventIndex());
            values.put(COLUMN_SHOULD_SHOW_AGAIN, survey.shouldShowAgain() ? 1 : 0);
            values.put(COLUMN_PAUSED, survey.isPaused() ? 1 : 0);
            values.put(COLUMN_SESSIONS_COUNT, survey.getSessionCounter());
            values.put(COLUMN_QUESTIONS, Question.toJson(survey.getQuestions()).toString());
            values.put(COLUMN_THANKS_LIST, ThankYouItem.toJson(survey.getThankYouItems()).toString());
            values.put(COLUMN_TARGET_AUDIENCES, Condition.toJson(survey.getTargetAudiences())
                    .toString());
            values.put(COLUMN_CUSTOM_ATTRIBUTES, Condition.toJson(survey.getCustomAttributes())
                    .toString());
            values.put(COLUMN_USER_EVENTS, Condition.toJson(survey.getUserEvents()).toString());
            values.put(COLUMN_SURVEY_STATE, survey.getSurveyState().toString());
            values.put(COLUMN_SURVEY_TARGET, survey.getTarget().toJson());
            values.put(COLUMN_SURVEY_TRIGGER_EVENT, survey.getTarget().getTrigger().getUserEvent());
            values.put(COLUMN_SURVEY_IS_LOCALIZED, survey.getLocalization().isLocalized());
            values.put(COLUMN_SURVEY_LOCALES, new JSONArray(survey.getLocalization().getLocales()).toString());
            if (survey.getLocalization() != null && survey.getLocalization().getCurrentLocale() != null)
                values.put(COLUMN_SURVEY_CURRENT_LOCALE, survey.getLocalization().getCurrentLocale());
            values.put(COLUMN_SURVEY_IS_DISMISSIBLE, survey.isDismissible() ? 1 : 0);
            putSessionId(values, survey.getSessionID());
            // Insert the new row, returning the primary key value of the new row
            long rowId = db.insertWithOnConflict(TABLE_NAME, null, values);
            if (rowId == -1) {
                update(survey);
            }
            db.setTransactionSuccessful();
            InstabugSDKLogger.d(Constants.LOG_TAG,
                    "survey id: " + survey.getId() + " has been added to DB");
            return rowId;
        } catch (JSONException e) {
            IBGDiagnostics.reportNonFatalAndLog(e, "survey insertion failed due to " + e.getMessage(), Constants.LOG_TAG);
            return -1;
        } finally {
            db.endTransaction();
            db.close();
        }
    }

    private static void putSessionId(ContentValues values, @Nullable String sessionId) {
        if (sessionId == null)
            values.putNull(COLUMN_SESSION_ID);
        else
            values.put(COLUMN_SESSION_ID, sessionId);
    }

    /**
     * Inserting a survey in the database
     *
     * @param survey               is the Survey object
     * @param publishStatusChanged
     * @param localeChanged
     * @return the row ID of the newly inserted row, or -1 if an error occurred
     */
    @WorkerThread
    public static synchronized long insertOrUpdatePausedOrLocales(Survey survey, boolean publishStatusChanged, boolean localeChanged) {
        // Gets the data repository in write mode
        SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
        try {
            db.beginTransaction();
            // Create a new map of values, where column names are the keys
            ContentValues values = new ContentValues();
            values.put(COLUMN_ID, survey.getId());
            values.put(COLUMN_TYPE, survey.getType());
            values.put(COLUMN_GOOGLE_PLAY_RATING, survey.isGooglePlayAppRating());
            values.put(COLUMN_TITLE, survey.getTitle());
            if (survey.getToken() != null)
                values.put(COLUMN_TOKEN, survey.getToken());
            values.put(COLUMN_CONDITIONS_OPERATOR, survey.getConditionsOperator());
            values.put(COLUMN_ANSWERED, survey.isAnswered() ? 1 : 0);
            values.put(COLUMN_DISMISSED_AT, survey.getDismissedAt());
            values.put(COLUMN_SHOWN_AT, survey.getShownAt());
            values.put(COLUMN_IS_CANCELLED, survey.isCancelled() ? 1 : 0);
            values.put(COLUMN_ATTEMPT_COUNT, survey.getAttemptCount());
            values.put(COLUMN_EVENT_INDEX, survey.getEventIndex());
            values.put(COLUMN_SHOULD_SHOW_AGAIN, survey.shouldShowAgain() ? 1 : 0);
            values.put(COLUMN_PAUSED, survey.isPaused() ? 1 : 0);
            values.put(COLUMN_SESSIONS_COUNT, survey.getSessionCounter());
            values.put(COLUMN_QUESTIONS, Question.toJson(survey.getQuestions()).toString());
            values.put(COLUMN_THANKS_LIST, ThankYouItem.toJson(survey.getThankYouItems()).toString());
            values.put(COLUMN_TARGET_AUDIENCES, Condition.toJson(survey.getTargetAudiences())
                    .toString());
            values.put(COLUMN_CUSTOM_ATTRIBUTES, Condition.toJson(survey.getCustomAttributes())
                    .toString());
            values.put(COLUMN_USER_EVENTS, Condition.toJson(survey.getUserEvents()).toString());
            values.put(COLUMN_SURVEY_STATE, survey.getSurveyState().toString());
            values.put(COLUMN_SURVEY_TARGET, survey.getTarget().toJson());
            values.put(COLUMN_SURVEY_TRIGGER_EVENT, survey.getTarget().getTrigger().getUserEvent());
            values.put(COLUMN_SURVEY_IS_LOCALIZED, survey.getLocalization().isLocalized());
            values.put(COLUMN_SURVEY_LOCALES, new JSONArray(survey.getLocalization().getLocales()).toString());
            if (survey.getLocalization() != null && survey.getLocalization().getCurrentLocale() != null)
                values.put(COLUMN_SURVEY_CURRENT_LOCALE, survey.getLocalization().getCurrentLocale());

            // Insert the new row, returning the primary key value of the new row
            long rowId = db.insertWithOnConflict(TABLE_NAME, null, values);
            if (rowId == -1) {
                if (publishStatusChanged) {
                    updateSurveyPausedField(db, survey);
                }
                if (localeChanged) {
                    updateLocales(db, survey);
                }
            }
            db.setTransactionSuccessful();
            InstabugSDKLogger.d(Constants.LOG_TAG,
                    "survey id: " + survey.getId() + " has been updated");
            return rowId;

        } catch (Exception e) {
            IBGDiagnostics.reportNonFatalAndLog(e, "survey insertion failed due to " + e.getMessage(), Constants.LOG_TAG);
            return -1;
        } finally {
            db.endTransaction();
            db.close();
        }
    }

    public static synchronized void updateSurveyTarget(@NonNull Survey survey) {
        SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
        try {
            updateSurveyTargetField(db, survey);
        } catch (Exception exception) {
            IBGDiagnostics.reportNonFatalAndLog(exception, "survey insertion failed due to " + exception.getMessage(), Constants.LOG_TAG);
        }
    }

    @WorkerThread
    private static void updateSurveyPausedField(@NonNull SQLiteDatabaseWrapper db, Survey survey) {
        ContentValues publishedStatusValues = new ContentValues();
        publishedStatusValues.put(COLUMN_PAUSED, survey.isPaused());
        updateField(db, survey.getId(), publishedStatusValues);
    }

    @WorkerThread
    private static void updateSurveyTargetField(@NonNull SQLiteDatabaseWrapper db, Survey survey) throws JSONException {
        ContentValues targetValues = new ContentValues();
        targetValues.put(COLUMN_SURVEY_TARGET, survey.getTarget().toJson());
        updateField(db, survey.getId(), targetValues);
    }

    @WorkerThread
    @VisibleForTesting
    private static void updateLocales(@NonNull SQLiteDatabaseWrapper db, Survey survey) throws JSONException {
        ContentValues localeContentValues = new ContentValues();
        localeContentValues.put(COLUMN_QUESTIONS, Question.toJson(survey.getQuestions()).toString());
        localeContentValues.put(COLUMN_THANKS_LIST, ThankYouItem.toJson(survey.getThankYouItems()).toString());
        if (survey.getLocalization() != null && survey.getLocalization().getCurrentLocale() != null)
            localeContentValues.put(COLUMN_SURVEY_CURRENT_LOCALE, survey.getLocalization().getCurrentLocale());
        updateField(db, survey.getId(), localeContentValues);
    }


    /**
     * Retrieve ReadyToBeSendSurveys
     *
     * @return a list of Surveys {@link com.instabug.survey.models.Survey}
     */
    @WorkerThread
    public static List<Survey> retrieveReadyToBeSend() {


        // When reading data one should always just get a readable database.
        SQLiteDatabaseWrapper database = DatabaseManager.getInstance().openDatabase();
        Cursor cursor = null;
        try {
            cursor = database.query(
                    // Name of the table to read from
                    TABLE_NAME,
                    // String array of the columns which are supposed to be read
                    null,
                    // The selection argument which specifies which row is read. // ? symbols are
                    // parameters.
                    COLUMN_SURVEY_STATE + "=? ",
                    // The actual parameters values for the selection as a String array. // ? above
                    // take the value
                    // from here
                    new String[]{SyncingStatus.READY_TO_SEND.toString()},
                    // GroupBy clause. Specify a column name to group similar values // in that
                    // column together.
                    null,
                    // Having clause. When using the GroupBy clause this allows you to // specify
                    // which groups to
                    // include.
                    null,
                    // OrderBy clause. Specify a column name here to order the results
                    // according to that column. Optionally append ASC or DESC to specify // an
                    // ascending or
                    // descending order.
                    null);
            // To increase performance first get the index of each column in the cursor
            if (cursor == null) return new ArrayList<>();

            // If moveToFirst() returns false then cursor is empty
            if (!cursor.moveToFirst() && !cursor.isClosed()) {
                cursor.close();
                return new ArrayList<>();
            }

            List<Survey> surveyList = new ArrayList<>();
            do {
                Survey survey = getSurveyFromCursor(cursor);
                surveyList.add(survey);
            } while (cursor.moveToNext());
            // Read the values of a row in the table using the indexes acquired above
            InstabugSDKLogger.d(Constants.LOG_TAG,
                    surveyList.size() + " surveys are ready to be sent");
            return surveyList;
        } catch (Exception e) {
            IBGDiagnostics.reportNonFatalAndLog(e, " retrieve ready to be send surveys failed: " + e.getMessage(), Constants.LOG_TAG);
            return new ArrayList<>();
        } finally {
            // Don't forget to close the Cursor once you are done to avoid memory leaks.
            // Using a try/finally like in this example is usually the best way to handle this
            if (cursor != null) {
                cursor.close();
            }
            // close the database
            database.close();
        }
    }

    /**
     * Retrieve survey by id
     *
     * @return the requested survey {@link com.instabug.survey.models.Survey}
     */
    @Nullable
    public static Survey retrieveById(long surveyId) {


        // When reading data one should always just get a readable database.
        SQLiteDatabaseWrapper database = DatabaseManager.getInstance().openDatabase();
        Cursor cursor = null;
        try {
            cursor = database.query(
                    // Name of the table to read from
                    TABLE_NAME,
                    // String array of the columns which are supposed to be read
                    null,
                    // The selection argument which specifies which row is read. // ? symbols are
                    // parameters.
                    COLUMN_ID + "=? ",
                    // The actual parameters values for the selection as a String array. // ? above
                    // take the value
                    // from here
                    new String[]{String.valueOf(surveyId)},
                    // GroupBy clause. Specify a column name to group similar values // in that
                    // column together.
                    null,
                    // Having clause. When using the GroupBy clause this allows you to // specify
                    // which groups to
                    // include.
                    null,
                    // OrderBy clause. Specify a column name here to order the results
                    // according to that column. Optionally append ASC or DESC to specify // an
                    // ascending or
                    // descending order.
                    null);
            if (cursor == null) return null;

            // If moveToFirst() returns false then cursor is empty
            if (!cursor.moveToFirst() && !cursor.isClosed()) {
                cursor.close();
                return null;
            }
            Survey survey = getSurveyFromCursor(cursor);

            // Read the values of a row in the table using the indexes acquired above
            InstabugSDKLogger.d(Constants.LOG_TAG,
                    "survey with id: " + survey.getId() + " has been retrieved from DB");
            return survey;
        } catch (JSONException e) {
            IBGDiagnostics.reportNonFatalAndLog(e, "survey conversion failed due to " + e.getMessage(), Constants.LOG_TAG);
            return null;
        } catch (Exception e) {
            IBGDiagnostics.reportNonFatalAndLog(e, "retrieve survey by id failed: " + e.getMessage(), Constants.LOG_TAG);
            return null;
        } finally {
            // Don't forget to close the Cursor once you are done to avoid memory leaks.
            // Using a try/finally like in this example is usually the best way to handle this
            if (cursor != null) {
                cursor.close();
            }
            // close the database
            database.close();
        }
    }


    /**
     * Retrieve NotAnsweredSurveys
     *
     * @return a list of Surveys {@link com.instabug.survey.models.Survey}
     */
    public static List<Survey> retrieveNotAnswered() {
        InstabugSDKLogger.d(Constants.LOG_TAG, "Getting unanswered surveys");
        // When reading data one should always just get a readable database.
        SQLiteDatabaseWrapper database = DatabaseManager.getInstance().openDatabase();
        Cursor cursor = null;
        try {
            cursor = database.query(
                    // Name of the table to read from
                    TABLE_NAME,
                    // String array of the columns which are supposed to be read
                    null,
                    // The selection argument which specifies which row is read. // ? symbols are
                    // parameters.
                    COLUMN_ANSWERED + "=?",
                    // The actual parameters values for the selection as a String array. // ? above
                    // take the value
                    // from here
                    new String[]{String.valueOf(0)},
                    // GroupBy clause. Specify a column name to group similar values // in that
                    // column together.
                    null,
                    // Having clause. When using the GroupBy clause this allows you to // specify
                    // which groups to
                    // include.
                    null,
                    // OrderBy clause. Specify a column name here to order the results
                    // according to that column. Optionally append ASC or DESC to specify // an
                    // ascending or
                    // descending order.
                    null);
            if (cursor == null) return new ArrayList<>();

            // If moveToFirst() returns false then cursor is empty
            if (!cursor.moveToFirst() && !cursor.isClosed()) {
                cursor.close();
                return new ArrayList<>();
            }

            List<Survey> surveyList = new ArrayList<>();
            do {
                Survey survey = getSurveyFromCursor(cursor);
                surveyList.add(survey);
            } while (cursor.moveToNext());
            // Read the values of a row in the table using the indexes acquired above
            InstabugSDKLogger.d(Constants.LOG_TAG,
                    "Retrieved " + surveyList.size() + " unanswered surveys from DB");
            return surveyList;
        } catch (Exception e) {
            IBGDiagnostics.reportNonFatalAndLog(e, "survey conversion failed due to " + e.getMessage(), Constants.LOG_TAG);
            return new ArrayList<>();
        } finally {
            // Don't forget to close the Cursor once you are done to avoid memory leaks.
            // Using a try/finally like in this example is usually the best way to handle this
            if (cursor != null) {
                cursor.close();
            }
            // close the database
            database.close();
        }
    }

    /**
     * Retrieve NotAnsweredSurveys
     *
     * @return a list of Surveys {@link com.instabug.survey.models.Survey}
     */
    public static List<Survey> retrieveByTriggerEvent(String triggerEvent) {
        InstabugSDKLogger.d(Constants.LOG_TAG, "Getting surveys by event: " + triggerEvent);
        // When reading data one should always just get a readable database.
        SQLiteDatabaseWrapper database = DatabaseManager.getInstance().openDatabase();
        Cursor cursor = null;
        try {
            cursor = database.query(
                    // Name of the table to read from
                    TABLE_NAME,
                    // String array of the columns which are supposed to be read
                    null,
                    // The selection argument which specifies which row is read. // ? symbols are
                    // parameters.
                    COLUMN_SURVEY_TRIGGER_EVENT + "=?",
                    // The actual parameters values for the selection as a String array. // ? above
                    // take the value
                    // from here
                    new String[]{triggerEvent},
                    // GroupBy clause. Specify a column name to group similar values // in that
                    // column together.
                    null,
                    // Having clause. When using the GroupBy clause this allows you to // specify
                    // which groups to
                    // include.
                    null,
                    // OrderBy clause. Specify a column name here to order the results
                    // according to that column. Optionally append ASC or DESC to specify // an
                    // ascending or
                    // descending order.
                    null);
            if (cursor == null) return new ArrayList<>();
            // If moveToFirst() returns false then cursor is empty
            if (!cursor.moveToFirst() && !cursor.isClosed()) {
                cursor.close();
                return new ArrayList<>();
            }

            List<Survey> surveyList = new ArrayList<>();
            do {
                Survey survey = getSurveyFromCursor(cursor);
                surveyList.add(survey);
            } while (cursor.moveToNext());
            // Read the values of a row in the table using the indexes acquired above
            InstabugSDKLogger.d(Constants.LOG_TAG,
                    "Retrieved " + surveyList.size() + " unanswered surveys from DB");
            return surveyList;
        } catch (Exception e) {
            IBGDiagnostics.reportNonFatalAndLog(e, " retrieve surveys by trigger event failed: " + e.getMessage(), Constants.LOG_TAG);
            return new ArrayList<>();
        } finally {
            // Don't forget to close the Cursor once you are done to avoid memory leaks.
            // Using a try/finally like in this example is usually the best way to handle this
            if (cursor != null) {
                cursor.close();
            }
            // close the database
            database.close();
        }
    }

    /**
     * Retrieve TimeTriggeredSurveys
     *
     * @return a list of Surveys {@link com.instabug.survey.models.Survey}
     */
    public static List<Survey> retrieveTimeTriggeredSurveys() {
        InstabugSDKLogger.d(Constants.LOG_TAG, "Getting time triggered surveys");
        // When reading data one should always just get a readable database.
        SQLiteDatabaseWrapper database = DatabaseManager.getInstance().openDatabase();
        Cursor cursor = null;
        try {
            cursor = database.query(
                    // Name of the table to read from
                    TABLE_NAME,
                    // String array of the columns which are supposed to be read
                    null,
                    // The selection argument which specifies which row is read. // ? symbols are
                    // parameters.
                    COLUMN_SURVEY_TRIGGER_EVENT + "=?",
                    // The actual parameters values for the selection as a String array. // ? above
                    // take the value
                    // from here
                    new String[]{""},
                    // GroupBy clause. Specify a column name to group similar values // in that
                    // column together.
                    null,
                    // Having clause. When using the GroupBy clause this allows you to // specify
                    // which groups to
                    // include.
                    null,
                    // OrderBy clause. Specify a column name here to order the results
                    // according to that column. Optionally append ASC or DESC to specify // an
                    // ascending or
                    // descending order.
                    null);
            if (cursor == null) return new ArrayList<>();
            // If moveToFirst() returns false then cursor is empty
            if (!cursor.moveToFirst() && !cursor.isClosed()) {
                cursor.close();
                return new ArrayList<>();
            }

            List<Survey> surveyList = new ArrayList<>();
            do {
                Survey survey = getSurveyFromCursor(cursor);
                surveyList.add(survey);
            } while (cursor.moveToNext());
            // Read the values of a row in the table using the indexes acquired above
            InstabugSDKLogger.d(Constants.LOG_TAG,
                    "Retrieved " + surveyList.size() + " unanswered surveys from DB");
            return surveyList;
        } catch (Exception e) {
            IBGDiagnostics.reportNonFatalAndLog(e, " retrieve time triggered surveys failed: " + e.getMessage(), Constants.LOG_TAG);
            return new ArrayList<>();
        } finally {
            // Don't forget to close the Cursor once you are done to avoid memory leaks.
            // Using a try/finally like in this example is usually the best way to handle this
            if (cursor != null) {
                cursor.close();
            }
            // close the database
            database.close();
        }
    }

    /**
     * Retrieve EventTriggeredSurveys without considering the event value
     *
     * @return a list of Surveys {@link com.instabug.survey.models.Survey}
     */
    public static List<Survey> retrieveEventTriggeredSurveys() {
        InstabugSDKLogger.d(Constants.LOG_TAG, "Getting event triggerred surveys");
        // When reading data one should always just get a readable database.
        SQLiteDatabaseWrapper database = DatabaseManager.getInstance().openDatabase();
        Cursor cursor = null;
        try {
            cursor = database.query(
                    // Name of the table to read from
                    TABLE_NAME,
                    // String array of the columns which are supposed to be read
                    null,
                    // The selection argument which specifies which row is read. // ? symbols are
                    // parameters.
                    COLUMN_SURVEY_TRIGGER_EVENT + " != ?",
                    // The actual parameters values for the selection as a String array. // ? above
                    // take the value
                    // from here
                    new String[]{""},
                    // GroupBy clause. Specify a column name to group similar values // in that
                    // column together.
                    null,
                    // Having clause. When using the GroupBy clause this allows you to // specify
                    // which groups to
                    // include.
                    null,
                    // OrderBy clause. Specify a column name here to order the results
                    // according to that column. Optionally append ASC or DESC to specify // an
                    // ascending or
                    // descending order.
                    null);
            if (cursor == null) return new ArrayList<>();

            // If moveToFirst() returns false then cursor is empty
            if (!cursor.moveToFirst() && !cursor.isClosed()) {
                cursor.close();
                return new ArrayList<>();
            }

            List<Survey> surveyList = new ArrayList<>();
            do {
                Survey survey = getSurveyFromCursor(cursor);
                surveyList.add(survey);
            } while (cursor.moveToNext());
            // Read the values of a row in the table using the indexes acquired above
            InstabugSDKLogger.d(Constants.LOG_TAG,
                    "Retrieved " + surveyList.size() + " unanswered surveys from DB");
            return surveyList;
        } catch (Exception e) {
            IBGDiagnostics.reportNonFatalAndLog(e, " retrieve event triggered surveys failed: " + e.getMessage(), Constants.LOG_TAG);
            return new ArrayList<>();
        } finally {
            // Don't forget to close the Cursor once you are done to avoid memory leaks.
            // Using a try/finally like in this example is usually the best way to handle this
            if (cursor != null) {
                cursor.close();
            }
            // close the database
            database.close();
        }
    }

    /**
     * Retrieve all userAttributes
     *
     * @return a list of Surveys {@link com.instabug.survey.models.Survey}
     */
    public static List<Survey> retrieve() {

        // When reading data one should always just get a readable database.
        SQLiteDatabaseWrapper database = DatabaseManager.getInstance().openDatabase();
        Cursor cursor = null;
        try {
            cursor = database.query(
                    // Name of the table to read from
                    TABLE_NAME,
                    // String array of the columns which are supposed to be read
                    null,
                    // The selection argument which specifies which row is read. // ? symbols are
                    // parameters.
                    null,
                    // The actual parameters values for the selection as a String array. // ? above
                    // take the value
                    // from here
                    null,
                    // GroupBy clause. Specify a column name to group similar values // in that
                    // column together.
                    null,
                    // Having clause. When using the GroupBy clause this allows you to // specify
                    // which groups to
                    // include.
                    null,
                    // OrderBy clause. Specify a column name here to order the results
                    // according to that column. Optionally append ASC or DESC to specify // an
                    // ascending or
                    // descending order.
                    null);
            if (cursor == null) return new ArrayList<>();

            // If moveToFirst() returns false then cursor is empty
            if (!cursor.moveToFirst() && !cursor.isClosed()) {
                cursor.close();
                return new ArrayList<>();
            }

            List<Survey> surveyList = new ArrayList<>();
            do {
                Survey survey = getSurveyFromCursor(cursor);
                surveyList.add(survey);
            } while (cursor.moveToNext());
            // Read the values of a row in the table using the indexes acquired above
            InstabugSDKLogger.d(Constants.LOG_TAG,
                    surveyList.size() + " surveys" +
                            " have been retrieved from DB");
            return surveyList;
        } catch (Exception e) {
            IBGDiagnostics.reportNonFatalAndLog(e, " retrieve surveys failed: " + e.getMessage(), Constants.LOG_TAG);
            return new ArrayList<>();
        } finally {
            // Don't forget to close the Cursor once you are done to avoid memory leaks.
            // Using a try/finally like in this example is usually the best way to handle this
            if (cursor != null) {
                cursor.close();
            }
            // close the database
            database.close();
        }
    }

    private static synchronized Survey getSurveyFromCursor(Cursor cursor) throws JSONException {
        // To increase performance first get the index of each column in the cursor
        final int idIndex = cursor.getColumnIndex(COLUMN_ID);
        final int typeIndex = cursor.getColumnIndex(COLUMN_TYPE);
        final int googlePlayRatingIndex = cursor.getColumnIndex(COLUMN_GOOGLE_PLAY_RATING);
        final int titleIndex = cursor.getColumnIndex(COLUMN_TITLE);
        final int tokenIndex = cursor.getColumnIndex(COLUMN_TOKEN);
        final int conditionsOperatorIndex = cursor.getColumnIndex(COLUMN_CONDITIONS_OPERATOR);
        final int answeredIndex = cursor.getColumnIndex(COLUMN_ANSWERED);
        final int dismissedAtIndex = cursor.getColumnIndex(COLUMN_DISMISSED_AT);
        final int shownAtIndex = cursor.getColumnIndex(COLUMN_SHOWN_AT);
        final int isCanceledIndex = cursor.getColumnIndex(COLUMN_IS_CANCELLED);
        final int attemptsCountIndex = cursor.getColumnIndex(COLUMN_ATTEMPT_COUNT);
        final int eventIndexIndex = cursor.getColumnIndex(COLUMN_EVENT_INDEX);
        final int shouldShowAgainIndex = cursor.getColumnIndex(COLUMN_SHOULD_SHOW_AGAIN);
        final int pausedIndex = cursor.getColumnIndex(COLUMN_PAUSED);
        final int sessionsCountIndex = cursor.getColumnIndex(COLUMN_SESSIONS_COUNT);
        final int questionsIndex = cursor.getColumnIndex(COLUMN_QUESTIONS);
        final int thanksListIndex = cursor.getColumnIndex(COLUMN_THANKS_LIST);
        final int targetAudiencesIndex = cursor.getColumnIndex(COLUMN_TARGET_AUDIENCES);
        final int customAttributesIndex = cursor.getColumnIndex(COLUMN_CUSTOM_ATTRIBUTES);
        final int userEventsIndex = cursor.getColumnIndex(COLUMN_USER_EVENTS);
        final int surveyStateIndex = cursor.getColumnIndex(COLUMN_SURVEY_STATE);
        final int surveyTargetIndex = cursor.getColumnIndex(COLUMN_SURVEY_TARGET);
        final int surveyLocalesIndex = cursor.getColumnIndex(COLUMN_SURVEY_LOCALES);
        final int isLocalizedIndex = cursor.getColumnIndex(COLUMN_SURVEY_IS_LOCALIZED);
        final int surveyCurrentLocaleIndex = cursor.getColumnIndex(COLUMN_SURVEY_CURRENT_LOCALE);
        final int sessionIdIndex = cursor.getColumnIndex(COLUMN_SESSION_ID);
        // Read the values of a row in the table using the indexes acquired above
        final Long id = cursor.getLong(idIndex);
        final int type = cursor.getInt(typeIndex);
        final int googlePlayRating = cursor.getInt(googlePlayRatingIndex);
        final String title = cursor.getString(titleIndex);
        final String token = cursor.getString(tokenIndex);
        final String conditionsOperator = cursor.getString(conditionsOperatorIndex);
        final int answered = cursor.getInt(answeredIndex);
        final int dismissedAt = cursor.getInt(dismissedAtIndex);
        final int shownAt = cursor.getInt(shownAtIndex);
        final int isCanceled = cursor.getInt(isCanceledIndex);
        final int attemptsCount = cursor.getInt(attemptsCountIndex);
        final int eventIndex = cursor.getInt(eventIndexIndex);
        final int shouldShowAgain = cursor.getInt(shouldShowAgainIndex);
        final int paused = cursor.getInt(pausedIndex);
        final int sessionsCount = cursor.getInt(sessionsCountIndex);
        final String questions = cursor.getString(questionsIndex);
        final String thanksList = cursor.getString(thanksListIndex);
        final String targetAudiences = cursor.getString(targetAudiencesIndex);
        final String customAttributes = cursor.getString(customAttributesIndex);
        final String userEvents = cursor.getString(userEventsIndex);
        final String surveyState = cursor.getString(surveyStateIndex);
        final String surveyTarget = cursor.getString(surveyTargetIndex);
        final String surveyLocales = cursor.getString(surveyLocalesIndex);
        final int surveyIsLocalized = cursor.getInt(isLocalizedIndex);
        final String surveyCurrentLocale = cursor.getString(surveyCurrentLocaleIndex);
        final String sessionId = cursor.isNull(sessionIdIndex) ? null : cursor.getString(sessionIdIndex);
        Survey survey = new Survey();
        survey.setId(id);
        survey.setType(type);
        survey.setGooglePlayAppRating(googlePlayRating == 1);
        survey.setTitle(title);
        survey.setToken(token);
        survey.setConditionsOperator(conditionsOperator);
        survey.setAnswered(answered == 1);
        survey.setDismissedAt(dismissedAt);
        survey.setShowAt(shownAt);
        survey.setCancelled(isCanceled == 1);
        survey.setAttemptCount(attemptsCount);
        survey.setEventIndex(eventIndex);
        survey.setShouldShowAgain(shouldShowAgain == 1);
        survey.setPaused(paused == 1);
        survey.setSessionCount(sessionsCount);
        survey.setQuestions(Question.fromJson(new JSONArray(questions)));
        survey.setThankYouItems(ThankYouItem.fromJson(new JSONArray(thanksList)));
        survey.setTargetAudiences(Condition.fromJson(new JSONArray(targetAudiences)));
        survey.setCustomAttributes(Condition.fromJson(new JSONArray(customAttributes)));
        survey.setUserEvents(Condition.fromJson(new JSONArray(userEvents)));
        survey.setSurveyState(SyncingStatus.valueOf(surveyState));
        survey.setTarget(Target.fromJsonString(surveyTarget));
        survey.getLocalization().setLocaleFromJson(new JSONArray(surveyLocales));
        survey.getLocalization().setCurrentLocale(surveyCurrentLocale);
        survey.getLocalization().setLocalized(surveyIsLocalized == 1);
        survey.setSessionID(sessionId);
        final int surveyIsDismissibleIndex = cursor.getColumnIndex(COLUMN_SURVEY_IS_DISMISSIBLE);
        final int surveyIsDismissible = cursor.getInt(surveyIsDismissibleIndex);
        survey.setDismissible(surveyIsDismissible == 1);
        return survey;
    }

    /**
     * Delete a specific userAttribute record
     *
     * @param surveyId the key of the userAttribute
     */
    @WorkerThread
    public static synchronized void delete(long surveyId) {
        SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
        String whereClause = COLUMN_ID + "=? ";
        String[] whereArgs = new String[]{String.valueOf(surveyId)};
        db.beginTransaction();
        try {
            db.delete(TABLE_NAME, whereClause, whereArgs);
            db.setTransactionSuccessful();
        } finally {
            db.endTransaction();
            db.close();
        }
    }


    /**
     * Delete all Surveys stored in the database
     * i.e. deleting the whole table
     */
    public static synchronized void deleteAll() {
        SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
        try {
            db.beginTransaction();
            db.delete(TABLE_NAME, null, null);
            db.setTransactionSuccessful();
        } finally {
            db.endTransaction();
            db.close();
        }
    }

    /**
     * This method to update single survey
     *
     * @param survey the new survey which will replace the old one.
     * @return affected rows
     */
    @WorkerThread
    public static synchronized long update(Survey survey) {
        // Gets the data repository in write mode
        SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
        try {
            db.beginTransaction();
            long row = updateSingleSurvey(db, survey);
            db.setTransactionSuccessful();
            return row;
        } finally {
            db.endTransaction();
            db.close();
        }
    }


    /**
     * This method to update multiple survey
     *
     * @param surveys the new surveys which will replace the old ones
     */
    @WorkerThread
    static synchronized void updateBulk(List<Survey> surveys) {
        SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
        try {
            db.beginTransaction();
            for (Survey survey : surveys) {
                updateSingleSurvey(db, survey);
            }
            db.setTransactionSuccessful();
        } finally {
            db.endTransaction();
            db.close();
        }
    }

    /**
     * This method to update survey but note that this method *DOESN'T* open database not begin/close any transactions,
     * the caller method must handle that.
     *
     * @param db     an opened SQLiteDatabaseWrapper
     * @param survey survey to be updated
     * @return row id greater than zero if the survey is updated successfully.
     */
    @WorkerThread
    private static long updateSingleSurvey(@NonNull SQLiteDatabaseWrapper db, Survey survey) {

        String whereClause = COLUMN_ID + "=? ";
        String[] whereArgs = new String[]{String.valueOf(survey.getId())};
        try {
            // Create a new map of values, where column names are the keys
            ContentValues values = new ContentValues();
            values.put(COLUMN_ID, survey.getId());
            values.put(COLUMN_TITLE, survey.getTitle());
            values.put(COLUMN_TYPE, survey.getType());
            values.put(COLUMN_GOOGLE_PLAY_RATING, survey.isGooglePlayAppRating());
            if (survey.getToken() != null)
                values.put(COLUMN_TOKEN, survey.getToken());
            values.put(COLUMN_CONDITIONS_OPERATOR, survey.getConditionsOperator());
            values.put(COLUMN_ANSWERED, survey.isAnswered() ? 1 : 0);
            values.put(COLUMN_DISMISSED_AT, survey.getDismissedAt());
            values.put(COLUMN_SHOWN_AT, survey.getShownAt());
            values.put(COLUMN_IS_CANCELLED, survey.isCancelled() ? 1 : 0);
            values.put(COLUMN_ATTEMPT_COUNT, survey.getAttemptCount());
            values.put(COLUMN_EVENT_INDEX, survey.getEventIndex());
            values.put(COLUMN_SHOULD_SHOW_AGAIN, survey.shouldShowAgain() ? 1 : 0);
            values.put(COLUMN_PAUSED, survey.isPaused() ? 1 : 0);
            values.put(COLUMN_SESSIONS_COUNT, survey.getSessionCounter());
            values.put(COLUMN_QUESTIONS, Question.toJson(survey.getQuestions()).toString());
            values.put(COLUMN_THANKS_LIST, ThankYouItem.toJson(survey.getThankYouItems()).toString());
            values.put(COLUMN_TARGET_AUDIENCES, Condition.toJson(survey.getTargetAudiences())
                    .toString());
            values.put(COLUMN_CUSTOM_ATTRIBUTES, Condition.toJson(survey.getCustomAttributes())
                    .toString());
            values.put(COLUMN_USER_EVENTS, Condition.toJson(survey.getUserEvents()).toString());
            values.put(COLUMN_SURVEY_STATE, survey.getSurveyState().toString());
            values.put(COLUMN_SURVEY_TARGET, survey.getTarget().toJson());
            values.put(COLUMN_SURVEY_TRIGGER_EVENT, survey.getTarget().getTrigger().getUserEvent());
            values.put(COLUMN_SURVEY_IS_LOCALIZED, survey.getLocalization().isLocalized());
            values.put(COLUMN_SURVEY_LOCALES, new JSONArray(survey.getLocalization().getLocales()).toString());
            if (survey.getLocalization() != null && survey.getLocalization().getCurrentLocale() != null)
                values.put(COLUMN_SURVEY_CURRENT_LOCALE, survey.getLocalization().getCurrentLocale());
            putSessionId(values, survey.getSessionID());
            // Insert the new row, returning the primary key value of the new row
            long rowId = db.update(TABLE_NAME, values, whereClause, whereArgs);
            if (rowId > 0) {
                InstabugSDKLogger.d(Constants.LOG_TAG,
                        "survey with id: " + survey.getId() + " has been updated");
            }
            return rowId;
        } catch (JSONException e) {
            IBGDiagnostics.reportNonFatalAndLog(e, "survey updating failed due to " + e.getMessage(), Constants.LOG_TAG);
            return -1;
        }
    }


    /**
     * A method to update a given field with given value, based on survey id condition
     *
     * @param db       database object and this object must be opened and ready for update transaction
     * @param surveyId id
     * @param values   value to be updated with
     * @return row id > 0 if succeed
     */
    private static synchronized long updateField(@NonNull SQLiteDatabaseWrapper db,
                                                 long surveyId, ContentValues values) {
        // Gets the data repository in write mode

        String whereClause = COLUMN_ID + "=? ";
        String[] whereArgs = new String[]{String.valueOf(surveyId)};

        // Insert the new row, returning the primary key value of the new row
        long rowId = db.update(TABLE_NAME, values, whereClause, whereArgs);
        InstabugSDKLogger.d(Constants.LOG_TAG,
                "survey with id: " + surveyId + " has been updated");
        return rowId;

    }

    public static synchronized long updateSessions(Survey survey) {
        // Gets the data repository in write mode
        SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
        String whereClause = COLUMN_ID + "=? ";
        String[] whereArgs = new String[]{String.valueOf(survey.getId())};
        try {
            db.beginTransaction();
            // Create a new map of values, where column names are the keys
            ContentValues values = new ContentValues();
            values.put(COLUMN_SESSIONS_COUNT, survey.getSessionCounter());

            // Insert the new row, returning the primary key value of the new row
            long rowId = db.update(TABLE_NAME, values, whereClause, whereArgs);
            db.setTransactionSuccessful();
            InstabugSDKLogger.d(Constants.LOG_TAG,
                    "survey with id: " + survey.getId() + " has been updated");
            return rowId;
        } finally {
            db.endTransaction();
            db.close();
        }
    }
}