package com.instabug.survey.announcements.network;

import androidx.annotation.VisibleForTesting;

import com.instabug.library.core.InstabugCore;
import com.instabug.library.networkv2.request.Request;
import com.instabug.library.networkv2.request.RequestParameter;
import com.instabug.library.user.UserManagerWrapper;
import com.instabug.survey.announcements.models.Announcement;
import com.instabug.survey.announcements.models.AnnouncementItem;
import com.instabug.survey.common.models.ActionEvent;
import com.instabug.survey.models.State;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;

import static com.instabug.survey.models.State.DISMISSED;

/**
 * Created by Barakat on 24/12/2018
 */
public class AnnouncementSubmittingUtils {
    public static final String RESPONDED_AT_PARAM = "responded_at";
    public static final String ANNOUNCEMENT_ID_PARAM = "announcement_id";
    public static final String NAME_PARAM = "name";
    public static final String EMAIL_PARAM = "email";
    public static final String EVENTS_PARAM = "events";
    public static final String ANNOUNCE_ID_PARAM = "announcement_item_id";
    public static final String RESPONSE_PARAM = "responses";
    public static final String VALUE_PARAM = "value";
    public static final String APP_VERSION = "app_version";
    public static final String PUSH_TOKEN_PARAM = "push_token";

    @VisibleForTesting
    static final String ANNOUNCEMENT_CURRENT_LOCALE = "locale";


    public static void   addParamsToSubmittingAnnouncementRequest(Request.Builder requestBuilder, String appVersion, Announcement announcement) throws JSONException {
        if (announcement == null) return;
        if (announcement.getAnnouncementItems() != null) {
            JSONArray responsesArray = getResponsesFromAnnounceItem(announcement.getAnnouncementItems());
            if (responsesArray.length() > 0) {
                requestBuilder.addParameter(new RequestParameter<>(RESPONSE_PARAM, responsesArray));
            }
        }
        requestBuilder.addParameter(new RequestParameter<>(ANNOUNCEMENT_ID_PARAM, announcement.getId()));
        String userName = InstabugCore.getIdentifiedUsername();
        if (userName != null){
            requestBuilder.addParameter(new RequestParameter<>(NAME_PARAM, userName));
        }
        requestBuilder.addParameter(new RequestParameter<>(EMAIL_PARAM, UserManagerWrapper.getUserEmail()));
        requestBuilder.addParameter(new RequestParameter<>(RESPONDED_AT_PARAM, announcement.getRespondedAt()));
        requestBuilder.addParameter(new RequestParameter<>(APP_VERSION, appVersion));
        if (announcement.getTarget() != null && announcement.getTarget().getActionEvents() != null)
            requestBuilder.addParameter(new RequestParameter<>(EVENTS_PARAM, getAnnouncementEvents(announcement.getTarget().getActionEvents())));
        if (announcement.getLocalization() != null && announcement.getLocalization().getCurrentLocale() != null)
            requestBuilder.addParameter(new RequestParameter<>(ANNOUNCEMENT_CURRENT_LOCALE, announcement.getLocalization().getCurrentLocale()));
        requestBuilder.addParameter(new RequestParameter<>(PUSH_TOKEN_PARAM, InstabugCore.getPushNotificationToken()));
    }

    private static JSONArray getAnnouncementEvents(ArrayList<ActionEvent> actionEvents) throws
            JSONException {
        JSONArray announcementEventsArray = new JSONArray();
        for (ActionEvent event : actionEvents) {
            JSONObject eventObj = new JSONObject();
            eventObj.put(ActionEvent.KEY_EVENT_TYPE, event.getEventType());
            eventObj.put(ActionEvent.KEY_TIMESTAMP, event.getTimestamp());
            eventObj.put(ActionEvent.KEY_INDEX, event.getIndex());
            announcementEventsArray.put(eventObj);
        }
        return announcementEventsArray;
    }

    public static JSONArray getResponsesFromAnnounceItem(ArrayList<AnnouncementItem> announcementItems) throws
            JSONException {
        JSONArray responsesArray = new JSONArray();
        for (AnnouncementItem announcementItem : announcementItems) {
            if (announcementItem.getAnswer() != null && !announcementItem.getAnswer().equals("")) {
                JSONObject questionJsonObject = new JSONObject();
                questionJsonObject.put(VALUE_PARAM, announcementItem.getAnswer());
                questionJsonObject.put(ANNOUNCE_ID_PARAM, announcementItem.getId());
                responsesArray.put(questionJsonObject);
            }
        }
        return responsesArray;
    }

    public static JSONObject getAnnouncementAsResponse(Announcement announcement, @State String announcementState) throws JSONException {
        JSONObject announcementJson = new JSONObject();
        announcementJson.put("id", announcement.getId());
        announcementJson.put("type", announcement.getTypeAsString());
        announcementJson.put("title", announcement.getTitle());
        announcementJson.put("is_announcement", true);
        announcementJson.put("responses", getQuestionsAnswersAsJson(announcement, announcementState));
        return announcementJson;
    }

    public static JSONArray getQuestionsAnswersAsJson(Announcement announcement, @State String announcementState) throws JSONException {
        JSONArray responsesArray = new JSONArray();
        if (announcement.getAnnouncementItems() != null) {
            for (AnnouncementItem announcementItem : announcement.getAnnouncementItems()) {
                if (announcementItem.getAnswer() != null && !announcementItem.getAnswer().equals("")) {
                    JSONObject questionJsonObject = new JSONObject();
                    questionJsonObject.put("question_id", announcementItem.getId());
                    questionJsonObject.put("question_title",
                            announcementItem.getTitle() != null ? announcementItem.getTitle() : "");
                    String questionType =
                            !announcementItem.getTypeAsString().equals("") ? announcementItem.getTypeAsString() : announcement.getTypeAsString();
                    questionJsonObject.put("question_type", questionType);
                    questionJsonObject.put("response_timestamp",
                            announcementState.equals(DISMISSED) ? announcement.getDismissedAt() : announcement.getRespondedAt());
                    questionJsonObject.put("response_value", announcementItem.getAnswer());
                    responsesArray.put(questionJsonObject);
                }
            }
        }
        return responsesArray;
    }
}
