package com.instabug.survey.common.userInteractions;

import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.UserInteractions.COLUMN_ALREADY_SHOWN;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.UserInteractions.COLUMN_ANSWERED;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.UserInteractions.COLUMN_ATTEMPT_COUNT;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.UserInteractions.COLUMN_DISMISSED_AT;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.UserInteractions.COLUMN_EVENT_INDEX;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.UserInteractions.COLUMN_IS_CANCELLED;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.UserInteractions.COLUMN_SESSIONS_COUNT;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.UserInteractions.COLUMN_SHOULD_SHOW_AGAIN;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.UserInteractions.COLUMN_SHOWN_AT;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.UserInteractions.COLUMN_SURVEY_ID;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.UserInteractions.COLUMN_SURVEY_TARGET;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.UserInteractions.COLUMN_USER_INTERACTION_ON_TYPE;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.UserInteractions.COLUMN_UUID_TYPE;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.UserInteractions.TABLE_NAME;

import android.content.ContentValues;
import android.database.Cursor;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.instabug.library.diagnostics.IBGDiagnostics;
import com.instabug.library.internal.storage.cache.db.DatabaseManager;
import com.instabug.library.internal.storage.cache.db.SQLiteDatabaseWrapper;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.survey.Constants;
import com.instabug.survey.common.models.Target;
import com.instabug.survey.common.models.UserInteraction;
import com.instabug.survey.models.UserInteractionOnTypes;

import org.json.JSONException;

import java.util.ArrayList;
import java.util.List;

//import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.UserInteractions.COLUMN_SURVEY_STATE;


public class UserInteractionDbHelper {

    /**
     * Insert survey's user interaction
     *
     * @param userInteraction
     * @return id of inserted row
     */
    public static synchronized long insert(@NonNull UserInteraction userInteraction) {
        if (!isValidUserInteraction(userInteraction)) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Couldn't insert user interaction to DB, user interaction is invalid");
            return -1;
        }
        // Gets the data repository in write mode
        SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
        try {
            db.beginTransaction();

            // Create a new map of values, where column names are the keys
            ContentValues values = getTargetContentValue(userInteraction);


            // Insert the new row, returning the primary key value of the new row
            long rowId = db.insertWithOnConflict(TABLE_NAME, null, values);
            if (rowId == -1) {
                update(userInteraction);
            }
            db.setTransactionSuccessful();
            InstabugSDKLogger.d(Constants.LOG_TAG,
                    "survey user interaction with survey id: " + userInteraction.getSurveyId() +
                            " and uuid: " + userInteraction.getUuid() + " has been added to DB");
            return rowId;
        } catch (JSONException e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "survey insertion failed due to " + e.getMessage());
            IBGDiagnostics.reportNonFatal(e, "survey insertion failed due to " + e.getMessage());
            return -1;
        } finally {
            db.endTransaction();
            db.close();
        }
    }

    /**
     * Insert bulk survey's user interaction
     *
     * @param userInteractions list of user interactions to be inserted
     */
    public static synchronized void insertBulk(@NonNull List<UserInteraction> userInteractions) {

        // Gets the data repository in write mode
        SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
        try {
            db.beginTransaction();

            // Create a new map of values, where column names are the keys
            for (UserInteraction userInteraction : userInteractions) {
                if (!isValidUserInteraction(userInteraction)) {
                    InstabugSDKLogger.e(Constants.LOG_TAG, "Couldn't insert user interaction to DB, user interaction is invalid");
                    continue;
                }
                ContentValues values = getTargetContentValue(userInteraction);
                // Insert the new row, returning the primary key value of the new row
                long rowId = db.insertWithOnConflict(TABLE_NAME, null, values);
                if (rowId == -1) {
                    update(userInteraction);
                }
                InstabugSDKLogger.d(Constants.LOG_TAG, "survey user interaction with survey id: " + userInteraction.getSurveyId() + " and uuid: " + userInteraction.getUuid() + " has been added to " + TABLE_NAME);
            }

            db.setTransactionSuccessful();

        } catch (JSONException e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "survey insertion failed due to " + e.getMessage());
            IBGDiagnostics.reportNonFatal(e, "survey insertion failed due to " + e.getMessage());
        } finally {
            db.endTransaction();
            db.close();
        }
    }


    public static synchronized long update(UserInteraction userInteraction) {
        // Gets the data repository in write mode
        SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
        String whereClause = COLUMN_SURVEY_ID + "=? AND " + COLUMN_UUID_TYPE + "=? AND "
                + COLUMN_USER_INTERACTION_ON_TYPE + "=?";
        String[] whereArgs = new String[]{String.valueOf(userInteraction.getSurveyId()),
                userInteraction.getUuid(), String.valueOf(userInteraction.getActionOnType())};
        try {
            db.beginTransaction();
            // Create a new map of values, where column names are the keys
            ContentValues values = getTargetContentValue(userInteraction);

            // Insert the new row, returning the primary key value of the new row
            long rowId = db.update(TABLE_NAME, values, whereClause, whereArgs);
            db.setTransactionSuccessful();
            InstabugSDKLogger.d(Constants.LOG_TAG,
                    "survey user interaction with survey id: " + userInteraction.getSurveyId() +
                            " and uuid: " + userInteraction.getUuid() + " has been updated ");
            return rowId;
        } catch (JSONException e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "survey updating failed due to " + e.getMessage());
            IBGDiagnostics.reportNonFatal(e, "survey updating failed due to " + e.getMessage());
            return -1;
        } finally {
            db.endTransaction();
            db.close();
        }
    }


    /**
     * Retrieve user's interaction by surveyId, uuid and user interaction type
     *
     * @param surveyId
     * @param uuid
     * @param userInteractionUponType
     * @return a list of Surveys {@link com.instabug.survey.models.Survey}
     */
    @Nullable
    public static synchronized UserInteraction retrieveBySurveyIdAndUUID(Long surveyId, String uuid, @UserInteractionOnTypes
            int userInteractionUponType) {


        // When reading data one should always just get a readable database.
        SQLiteDatabaseWrapper database = DatabaseManager.getInstance().openDatabase();
        Cursor cursor = null;
        try {
            cursor = database.query(
                    // Name of the table to read from
                    TABLE_NAME,
                    // String array of the columns which are supposed to be read
                    null,
                    // The selection argument which specifies which row is read. // ? symbols are
                    // parameters.
                    COLUMN_SURVEY_ID + "=? AND " + COLUMN_UUID_TYPE + "=? AND "
                            + COLUMN_USER_INTERACTION_ON_TYPE + "=?",

                    // The actual parameters values for the selection as a String array. // ? above
                    // take the value
                    // from here
                    new String[]{String.valueOf(surveyId), uuid, String.valueOf(userInteractionUponType)},
                    // GroupBy clause. Specify a column name to group similar values // in that
                    // column together.
                    null,
                    // Having clause. When using the GroupBy clause this allows you to // specify
                    // which groups to
                    // include.
                    null,
                    // OrderBy clause. Specify a column name here to order the results
                    // according to that column. Optionally append ASC or DESC to specify // an
                    // ascending or
                    // descending order.
                    null);
            // To increase performance first get the index of each column in the cursor
            if (cursor == null) return null;

            final int answeredIndex = cursor.getColumnIndex(COLUMN_ANSWERED);
            final int dismissedAtIndex = cursor.getColumnIndex(COLUMN_DISMISSED_AT);
            final int shownAtIndex = cursor.getColumnIndex(COLUMN_SHOWN_AT);
            final int isCanceledIndex = cursor.getColumnIndex(COLUMN_IS_CANCELLED);
            final int attemptsCountIndex = cursor.getColumnIndex(COLUMN_ATTEMPT_COUNT);
            final int eventIndexIndex = cursor.getColumnIndex(COLUMN_EVENT_INDEX);
            final int shouldShowAgainIndex = cursor.getColumnIndex(COLUMN_SHOULD_SHOW_AGAIN);
            final int sessionsCountIndex = cursor.getColumnIndex(COLUMN_SESSIONS_COUNT);
            final int surveyTargetIndex = cursor.getColumnIndex(COLUMN_SURVEY_TARGET);
            final int alreadyShownIndex = cursor.getColumnIndex(COLUMN_ALREADY_SHOWN);


            // If moveToFirst() returns false then cursor is empty
            if (!cursor.moveToFirst()) {
                cursor.close();
                return null;
            }

            final int answered = cursor.getInt(answeredIndex);
            final int dismissedAt = cursor.getInt(dismissedAtIndex);
            final int shownAt = cursor.getInt(shownAtIndex);
            final int isCanceled = cursor.getInt(isCanceledIndex);
            final int attemptsCount = cursor.getInt(attemptsCountIndex);
            final int eventIndex = cursor.getInt(eventIndexIndex);
            final int shouldShowAgain = cursor.getInt(shouldShowAgainIndex);
            final int sessionsCount = cursor.getInt(sessionsCountIndex);
            final String surveyTarget = cursor.getString(surveyTargetIndex);
            final int alreadyShown = cursor.getInt(alreadyShownIndex);

            UserInteraction userInteraction = new UserInteraction(userInteractionUponType);
            userInteraction.setSurveyId(surveyId);
            userInteraction.setUuid(uuid);
            userInteraction.setAttemptCount(attemptsCount);
            userInteraction.setShownAt(shownAt);
            userInteraction.setEventIndex(eventIndex);
            userInteraction.setCancelled(isCanceled == 1);
            userInteraction.setAnswered(answered == 1);
            userInteraction.setDismissedAt(dismissedAt);
            userInteraction.setSessionCounter(sessionsCount);
            userInteraction.setShouldShowAgain(shouldShowAgain == 1);
            userInteraction.setTarget(Target.fromJsonString(surveyTarget));
            userInteraction.setAlreadyShown(alreadyShown == 1);

            return userInteraction;
        } catch (Exception e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "user interaction retrieveBySurveyIdAndUUID failed due to " + e.getMessage());
            IBGDiagnostics.reportNonFatal(e, "user interaction retrieveBySurveyIdAndUUID failed due to " + e.getMessage());
            return null;
        } finally {
            // Don't forget to close the Cursor once you are done to avoid memory leaks.
            // Using a try/finally like in this example is usually the best way to handle this
            if (cursor != null) {
                cursor.close();
            }
            // close the database
            database.close();
        }
    }

    /**
     * Retrieve all user's interactions
     *
     * @return a list of Surveys {@link com.instabug.survey.models.Survey}
     */
    @Nullable
    public static synchronized List<UserInteraction> retrieveAll() {

        // When reading data one should always just get a readable database.
        SQLiteDatabaseWrapper database = DatabaseManager.getInstance().openDatabase();

        Cursor cursor = null;
        try {
            cursor = database.query(
                    // Name of the table to read from
                    TABLE_NAME,
                    // String array of the columns which are supposed to be read
                    null,
                    // The selection argument which specifies which row is read. // ? symbols are
                    // parameters.
                    null,
                    // The actual parameters values for the selection as a String array. // ? above
                    // take the value
                    // from here
                    new String[]{},
                    // GroupBy clause. Specify a column name to group similar values // in that
                    // column together.
                    null,
                    // Having clause. When using the GroupBy clause this allows you to // specify
                    // which groups to
                    // include.
                    null,
                    // OrderBy clause. Specify a column name here to order the results
                    // according to that column. Optionally append ASC or DESC to specify // an
                    // ascending or
                    // descending order.
                    null);

            if (cursor == null) return null;

            // To increase performance first get the index of each column in the cursor
            final int surveyIdIndex = cursor.getColumnIndex(COLUMN_SURVEY_ID);
            final int uuidIndex = cursor.getColumnIndex(COLUMN_UUID_TYPE);
            final int surveyInteractionTypeIndex = cursor.getColumnIndex(COLUMN_USER_INTERACTION_ON_TYPE);
            final int answeredIndex = cursor.getColumnIndex(COLUMN_ANSWERED);
            final int dismissedAtIndex = cursor.getColumnIndex(COLUMN_DISMISSED_AT);
            final int shownAtIndex = cursor.getColumnIndex(COLUMN_SHOWN_AT);
            final int isCanceledIndex = cursor.getColumnIndex(COLUMN_IS_CANCELLED);
            final int attemptsCountIndex = cursor.getColumnIndex(COLUMN_ATTEMPT_COUNT);
            final int eventIndexIndex = cursor.getColumnIndex(COLUMN_EVENT_INDEX);
            final int shouldShowAgainIndex = cursor.getColumnIndex(COLUMN_SHOULD_SHOW_AGAIN);
            final int sessionsCountIndex = cursor.getColumnIndex(COLUMN_SESSIONS_COUNT);
            final int surveyTargetIndex = cursor.getColumnIndex(COLUMN_SURVEY_TARGET);
            final int alreadyShownIndex = cursor.getColumnIndex(COLUMN_ALREADY_SHOWN);

            List<UserInteraction> userInteractions = new ArrayList<>();
            // If moveToFirst() returns false then cursor is empty
            if (!cursor.moveToFirst()) {
                cursor.close();
                return null;
            }
            do {
                final long surveyId = cursor.getLong(surveyIdIndex);
                final String uuid = cursor.getString(uuidIndex);
                final int surveyInteractionType = cursor.getInt(surveyInteractionTypeIndex);
                final int answered = cursor.getInt(answeredIndex);
                final int dismissedAt = cursor.getInt(dismissedAtIndex);
                final int shownAt = cursor.getInt(shownAtIndex);
                final int isCanceled = cursor.getInt(isCanceledIndex);
                final int attemptsCount = cursor.getInt(attemptsCountIndex);
                final int eventIndex = cursor.getInt(eventIndexIndex);
                final int shouldShowAgain = cursor.getInt(shouldShowAgainIndex);
                final int sessionsCount = cursor.getInt(sessionsCountIndex);
                final String surveyTarget = cursor.getString(surveyTargetIndex);
                final int alreadyShown = cursor.getInt(alreadyShownIndex);

                UserInteraction userInteraction = new UserInteraction(surveyInteractionType);
                userInteraction.setSurveyId(surveyId);
                userInteraction.setUuid(uuid);
                userInteraction.setAttemptCount(attemptsCount);
                userInteraction.setShownAt(shownAt);
                userInteraction.setEventIndex(eventIndex);
                userInteraction.setCancelled(isCanceled == 1);
                userInteraction.setAnswered(answered == 1);
                userInteraction.setDismissedAt(dismissedAt);
                userInteraction.setSessionCounter(sessionsCount);
                userInteraction.setShouldShowAgain(shouldShowAgain == 1);
                userInteraction.setTarget(Target.fromJsonString(surveyTarget));
                userInteraction.setAlreadyShown(alreadyShown == 1);

                userInteractions.add(userInteraction);
            } while (cursor.moveToNext());
            return userInteractions;
        } catch (Exception e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, " retrieve user interactions failed: ", e);
            IBGDiagnostics.reportNonFatal(e, " retrieve user interactions failed: " + e.getMessage());
            return null;
        } finally {
            // Don't forget to close the Cursor once you are done to avoid memory leaks.
            // Using a try/finally like in this example is usually the best way to handle this
            if (cursor != null) {
                cursor.close();
            }
            // close the database
            database.close();
        }
    }


    /**
     * Delete a specific user interaction record
     *
     * @param surveyId the key of the user interaction
     */
    public static synchronized void delete(Long surveyId, String uuid,
                                           @UserInteractionOnTypes int userInteractionType) {
        SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
        String whereClause = COLUMN_SURVEY_ID + "=? AND " + COLUMN_UUID_TYPE + "=?" +
                " AND " + COLUMN_USER_INTERACTION_ON_TYPE + "=?";
        String[] whereArgs = new String[]{String.valueOf(surveyId), uuid,
                String.valueOf(userInteractionType)};
        db.beginTransaction();
        try {
            db.delete(TABLE_NAME, whereClause, whereArgs);
            db.setTransactionSuccessful();
        } finally {
            db.endTransaction();
            db.close();
        }
    }


    /**
     * Delete a bulk of specified user interaction record
     *
     * @param userInteractions to be deleted.
     */
    public static synchronized void deleteBulk(@NonNull List<UserInteraction> userInteractions) {

        SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
        db.beginTransaction();
        try {
            for (UserInteraction userInteraction : userInteractions) {
                String whereClause = COLUMN_SURVEY_ID + "=? AND " + COLUMN_UUID_TYPE + "=?" +
                        " AND " + COLUMN_USER_INTERACTION_ON_TYPE + "=?";
                String[] whereArgs = new String[]{String.valueOf(userInteraction.getSurveyId()),
                        userInteraction.getUuid(),
                        String.valueOf(userInteraction.getActionOnType())};
                db.delete(TABLE_NAME, whereClause, whereArgs);
            }
            db.setTransactionSuccessful();
        } finally {
            db.endTransaction();
            db.close();
        }
    }


    /**
     * Delete all user interactions stored in the database
     * i.e. deleting the whole table
     */
    public static synchronized void deleteAll() {
        SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
        try {
            db.beginTransaction();
            db.delete(TABLE_NAME, null, null);
            db.setTransactionSuccessful();
        } finally {
            db.endTransaction();
            db.close();
        }
    }


    private static ContentValues getTargetContentValue(UserInteraction userInteraction) throws JSONException {
        ContentValues values = new ContentValues();
        values.put(COLUMN_SURVEY_ID, userInteraction.getSurveyId());
        values.put(COLUMN_UUID_TYPE, userInteraction.getUuid() != null ? userInteraction.getUuid() : "null");
        values.put(COLUMN_USER_INTERACTION_ON_TYPE, userInteraction.getActionOnType());
        values.put(COLUMN_SURVEY_TARGET, userInteraction.getTarget().toJson());
        values.put(COLUMN_ANSWERED, userInteraction.isAnswered() ? 1 : 0);
        values.put(COLUMN_DISMISSED_AT, userInteraction.getDismissedAt());
        values.put(COLUMN_SHOWN_AT, userInteraction.getShownAt());
        values.put(COLUMN_IS_CANCELLED, userInteraction.isCancelled() ? 1 : 0);
        values.put(COLUMN_ATTEMPT_COUNT, userInteraction.getAttemptCount());
        values.put(COLUMN_EVENT_INDEX, userInteraction.getEventIndex());
        values.put(COLUMN_SHOULD_SHOW_AGAIN, userInteraction.isShouldShowAgain() ? 1 : 0);
        values.put(COLUMN_SESSIONS_COUNT, userInteraction.getSessionCounter());
//        values.put(COLUMN_SURVEY_STATE, userInteraction.getSurveyState().toString());
        values.put(COLUMN_ALREADY_SHOWN, userInteraction.isAlreadyShown() ? 1 : 0);
        return values;
    }

    private static boolean isValidUserInteraction(UserInteraction userInteraction) {
        return userInteraction != null &&
                userInteraction.getSurveyId() != 0 &&
                userInteraction.getUuid() != null && !userInteraction.getUuid().isEmpty();
    }

}
