package com.instabug.survey.announcements.cache;

import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.AnnouncementAssetsEntry.COLUMN_ANNOUNCE_ID;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.AnnouncementAssetsEntry.COLUMN_FILE_PATH;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.AnnouncementAssetsEntry.COLUMN_ID;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.AnnouncementAssetsEntry.TABLE_NAME;

import android.content.ContentValues;
import android.database.Cursor;

import androidx.annotation.Nullable;
import androidx.annotation.WorkerThread;

import com.instabug.library.diagnostics.IBGDiagnostics;
import com.instabug.library.internal.storage.cache.db.DatabaseManager;
import com.instabug.library.internal.storage.cache.db.InstabugDbContract;
import com.instabug.library.internal.storage.cache.db.SQLiteDatabaseWrapper;
import com.instabug.survey.Constants;

/**
 * Created by Barakat on 02/01/2019
 */
public class AnnouncementAssetsDBHelper {

    @WorkerThread
    public synchronized static long insert(long announceId, long featureID, String filePath) {
        SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
        db.beginTransaction();

        // Create a new map of values, where column names are the keys
        ContentValues values = new ContentValues();
        values.put(COLUMN_ID, featureID);
        values.put(COLUMN_ANNOUNCE_ID, announceId);
        values.put(COLUMN_FILE_PATH, filePath);
        // Insert the new row, returning the primary key value of the new row
        long rowId = db.insertWithOnConflict(TABLE_NAME, null, values);
        if (rowId == -1) {
            update(announceId, featureID, filePath);
        }
        db.setTransactionSuccessful();
        db.endTransaction();
        db.close();
        return rowId;
    }

    @WorkerThread
    public static synchronized long update(long announceId, long featureID, String filePath) {
        // Gets the data repository in write mode
        SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
        String whereClause = COLUMN_ID + "=? ";
        String[] whereArgs = new String[]{String.valueOf(featureID)};
        db.beginTransaction();
        // Create a new map of values, where column names are the keys
        ContentValues values = new ContentValues();
        values.put(COLUMN_ID, featureID);
        values.put(COLUMN_ANNOUNCE_ID, announceId);
        values.put(COLUMN_FILE_PATH, filePath);

        // Insert the new row, returning the primary key value of the new row
        long rowId = db.update(TABLE_NAME, values, whereClause, whereArgs);
        db.setTransactionSuccessful();
        db.endTransaction();
        db.close();
        return rowId;
    }

    @WorkerThread
    @Nullable
    public static String retrieveAssetsPathOf(long announceId, long featureId) {
        SQLiteDatabaseWrapper database = DatabaseManager.getInstance().openDatabase();
        String whereClause = COLUMN_ID + "=? " + " AND " + COLUMN_ANNOUNCE_ID + "=? ";
        String[] whereArgs = new String[]{String.valueOf(featureId), String.valueOf(announceId)};
        Cursor cursor = null;
        try {
            cursor = database.query(
                    InstabugDbContract.AnnouncementAssetsEntry.TABLE_NAME,
                    // String array of the columns which are supposed to be read
                    null,
                    // The selection argument which specifies which row is read. // ? symbols are
                    // parameters.
                    whereClause,
                    // The actual parameters values for the selection as a String array. // ? above
                    // take the value
                    // from here
                    whereArgs,
                    // GroupBy clause. Specify a column name to group similar values // in that
                    // column together.
                    null,
                    // Having clause. When using the GroupBy clause this allows you to // specify
                    // which groups to
                    // include.
                    null,
                    // OrderBy clause. Specify a column name here to order the results
                    // according to that column. Optionally append ASC or DESC to specify // an
                    // ascending or
                    // descending order.
                    null);
            if (cursor != null) {
                if (!cursor.moveToFirst()) return null;
                // To increase performance first get the index of each column in the cursor
                final int fileIndex = cursor.getColumnIndex(COLUMN_FILE_PATH);
                // Read the values of a row in the table using the indexes acquired above
                String filePath = cursor.getString(fileIndex);
                if (filePath != null)
                    return filePath;
            }

        } catch (Exception e) {
            IBGDiagnostics.reportNonFatalAndLog(e, "Retrieving assets path for announcement failed due to: " + e.getMessage(), Constants.LOG_TAG);
        } finally {
            if (cursor != null) {
                // Don't forget to close the Cursor once you are done to avoid memory leaks.
                // Using a try/finally like in this example is usually the best way to handle this
                cursor.close();
            }
            // close the database
            database.close();
        }
        return null;
    }

    @WorkerThread
    public synchronized static void delete() {
        SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
        try {
            db.beginTransaction();
            db.delete(TABLE_NAME, null, null);
            db.setTransactionSuccessful();
        } finally {
            db.endTransaction();
            db.close();
        }
    }
}
