package com.instabug.survey.announcements.ui.activity;

import static com.instabug.survey.models.State.DISMISSED;
import static com.instabug.survey.models.State.SUBMITTED;
import static com.instabug.survey.ui.ViewType.SECONDARY;

import androidx.fragment.app.FragmentActivity;

import com.instabug.library.core.ui.BasePresenter;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.TimeUtils;
import com.instabug.library.util.threading.PoolProvider;
import com.instabug.survey.Constants;
import com.instabug.survey.announcements.cache.AnnouncementCacheManager;
import com.instabug.survey.announcements.models.Announcement;
import com.instabug.survey.announcements.models.AnnouncementType;
import com.instabug.survey.announcements.network.AnnouncementSubmittingUtils;
import com.instabug.survey.announcements.network.InstabugAnnouncementSubmitterJob;
import com.instabug.survey.announcements.settings.PersistableSettings;
import com.instabug.survey.callbacks.OnFinishCallback;
import com.instabug.survey.common.LayoutUtils;
import com.instabug.survey.models.State;
import com.instabug.survey.settings.SurveysSettings;

import org.json.JSONException;

public class AnnouncementPresenter extends BasePresenter<AnnouncementActivityContract.View>
        implements AnnouncementActivityContract.Presenter {

    public AnnouncementPresenter(AnnouncementActivityContract.View view) {
        super(view);
    }

    @Override
    public void setLayoutHeightSecondary(boolean withAnimation) {
        AnnouncementActivityContract.View viewRef = view.get();
        if (viewRef != null && viewRef.getViewContext() != null) {
            FragmentActivity activity = viewRef.getViewContext();
            if (activity != null) {
                int heightInPx = LayoutUtils.getLayoutHeight(activity, SECONDARY);
                if (withAnimation) {
                    viewRef.setFrameLayoutHeightWithAnimation(heightInPx);
                } else {
                    viewRef.setFrameLayoutHeightTo(heightInPx);
                }
            }
        }
    }

    @Override
    public void submitAnnouncement(Announcement announcement) {
        if (announcement != null) {
            announcement.setSubmitted();
            handleOnFinishCallback(announcement, SUBMITTED);
            finishAnnouncement(announcement);
        }
    }

    @Override
    public void dismissAnnouncement(Announcement announcement) {
        if (announcement != null) {
            announcement.setDismissed();
            handleOnFinishCallback(announcement, DISMISSED);
            finishAnnouncement(announcement);
        }
    }

    private void handleOnFinishCallback(Announcement announcement, @State String announcementState) {
        OnFinishCallback onFinishCallback = SurveysSettings.getOnFinishCallback();
        if (onFinishCallback != null) {
            try {
                onFinishCallback.onFinish(Long.toString(announcement.getId()),
                        SUBMITTED, AnnouncementSubmittingUtils.getAnnouncementAsResponse(announcement, announcementState));
            } catch (JSONException e) {
                InstabugSDKLogger.e(Constants.LOG_TAG,
                        "Something went wrong during parsing Announcement object in onFinishCallback",
                        e);
            }
        }
    }

    private void finishAnnouncement(final Announcement announcement) {
        PoolProvider.postIOTask(() -> {
            AnnouncementCacheManager.updateAnnouncement(announcement);
            if (PersistableSettings.getInstance() != null)
                PersistableSettings.getInstance().setLastAnnouncementTime(TimeUtils.currentTimeMillis());
            final AnnouncementActivityContract.View viewRef = view.get();
            if (viewRef != null) {
                if (viewRef.getViewContext() != null) {
                    if (announcement.getType() == AnnouncementType.WHAT_IS_NEW) {
                        AnnouncementCacheManager.deleteAnnouncementAssets();
                    }
                    InstabugAnnouncementSubmitterJob.getInstance().start();
                    viewRef.finishAnnouncement(false);
                }
            }
        });


    }
}
