package com.instabug.survey.network.service;

import android.content.Context;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.instabug.library.IBGNetworkWorker;
import com.instabug.library.Instabug;
import com.instabug.library.InstabugNetworkJob;
import com.instabug.library.networkv2.request.Request;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.survey.Constants;
import com.instabug.survey.cache.SurveysCacheManager;
import com.instabug.survey.common.models.ActionEvent;
import com.instabug.survey.common.models.SyncingStatus;
import com.instabug.survey.di.ServiceLocator;
import com.instabug.survey.models.Survey;

import org.json.JSONException;

import java.util.List;

/**
 * Created by moustafa on 12/15/16.
 */

public class InstabugSurveysSubmitterJob extends InstabugNetworkJob {

    @Nullable
    private static InstabugSurveysSubmitterJob INSTANCE;

    private InstabugSurveysSubmitterJob() {
    }

    public synchronized static InstabugSurveysSubmitterJob getInstance() {
        if (INSTANCE == null) {
            INSTANCE = new InstabugSurveysSubmitterJob();
        }

        return INSTANCE;
    }

    private static void submitSurveys(@NonNull Context context) throws JSONException {
        InstabugSDKLogger.d(Constants.LOG_TAG, "submitSurveys started");
        List<Survey> readySurveys = SurveysCacheManager.getReadyToSendSurveys();
        InstabugSDKLogger.d(Constants.LOG_TAG, "ready to send surveys size: " + readySurveys.size());
        if (ServiceLocator.getConfigurationsProvider().isSurveysUsageExceeded()) {
            for (final Survey survey : readySurveys) {
                markSurveyAsSynced(survey);
            }
            SurveysCacheManager.updateBulk(readySurveys);

        } else {
            for (final Survey survey : readySurveys) {
                submitSurvey(context, survey);
            }
        }

    }

    private static void submitSurvey(@NonNull Context context, @NonNull Survey survey) throws JSONException {
        SurveysService.getInstance().submittingSurvey(context, survey, new Request.Callbacks<Boolean, Throwable>() {
            @Override
            public void onSucceeded(@Nullable Boolean response) {
                markSurveyAsSynced(survey);
                SurveysCacheManager.update(survey);
            }

            @Override
            public void onFailed(Throwable error) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "Submitting surveys got error: " + error.getMessage());
            }
        });
    }

    private static void markSurveyAsSynced(@NonNull Survey survey) {
        survey.setSurveyState(SyncingStatus.SYNCED);
        survey.setSessionID(null);
        if (survey.isLastEventSubmit()) {
            survey.clearAnswers();
        }
        if (survey.getSurveyEvents() != null) {
            final List<ActionEvent> events = survey.getSurveyEvents();
            if (!events.isEmpty()) {
                final ActionEvent lastEvent = events.get(events.size() - 1);
                lastEvent.setSynced(true);
                survey.getSurveyEvents().clear();
                survey.getSurveyEvents().add(lastEvent);
            }
        }
    }

    public void start() {
        enqueueJob(IBGNetworkWorker.SURVEYS, new Runnable() {
            @Override
            public void run() {
                if (Instabug.getApplicationContext() != null) {
                    try {
                        submitSurveys(Instabug.getApplicationContext());
                    } catch (Exception e) {
                        InstabugSDKLogger.e(Constants.LOG_TAG, "Error " + e.getMessage() + " occurred while submitting survey", e);
                    }
                } else {
                    InstabugSDKLogger.d(Constants.LOG_TAG, "Context was null while submitting surveys");
                }
            }
        });

    }
}

