package com.instabug.survey.announcements;


import android.app.ActivityManager;
import android.content.ComponentName;
import android.content.Context;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;
import androidx.annotation.WorkerThread;

import com.instabug.library.Feature;
import com.instabug.library.IBGFeature;
import com.instabug.library.Instabug;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.internal.device.InstabugDeviceProperties;
import com.instabug.library.networkv2.request.Request;
import com.instabug.library.user.UserManagerWrapper;
import com.instabug.library.util.DeviceStateProvider;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.LocaleUtils;
import com.instabug.library.util.TimeUtils;
import com.instabug.library.util.threading.PoolProvider;
import com.instabug.survey.Constants;
import com.instabug.survey.announcements.cache.AnnouncementCacheManager;
import com.instabug.survey.announcements.cache.NewFeaturesAssetsHelper;
import com.instabug.survey.announcements.models.Announcement;
import com.instabug.survey.announcements.models.AnnouncementType;
import com.instabug.survey.announcements.network.AnnouncementsService;
import com.instabug.survey.announcements.network.InstabugAnnouncementSubmitterJob;
import com.instabug.survey.announcements.settings.AnnouncementsSettings;
import com.instabug.survey.announcements.settings.PersistableSettings;
import com.instabug.survey.announcements.ui.activity.AnnouncementActivity;
import com.instabug.survey.common.AutoShowingManager;
import com.instabug.survey.common.models.UserInteraction;
import com.instabug.survey.common.userInteractions.UserInteractionCacheManager;
import com.instabug.survey.models.UserInteractionOnTypes;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.List;

/**
 * Created by Barakat on 24/12/2018
 */
public class AnnouncementManager {

    private static AnnouncementManager instance;
    private final Context applicationContext;
    @Nullable
    private AnnouncementValidator announcementValidator;

    public final static String ANNOUNCEMENTS_FETCHING_ERROR = "Announcement Fetching Failed due to ";

    @VisibleForTesting
    boolean hasTokenChanged = false;

    AnnouncementManager(Context applicationContext) {
        this.applicationContext = applicationContext;
        syncReadyToBeSentAnnouncements();
    }

    @NonNull
    private AnnouncementValidator getAnnouncementValidator() {
        if (announcementValidator == null) {
            announcementValidator = new AnnouncementValidator(
                    InstabugDeviceProperties.getAppVersionName(applicationContext), // eg. 1.0.0-beta
                    DeviceStateProvider.getAppVersion(applicationContext)); // eg. 1.0.0-beta (3)
        }
        return announcementValidator;
    }

    public static void init(Context applicationContext) {
        instance = new AnnouncementManager(applicationContext);
    }

    /**
     * Returns the current singleton instance of this class.
     *
     * @return a {@link AnnouncementManager} instance
     */
    public static AnnouncementManager getInstance(Context applicationContext) {
        if (instance == null) {
            init(applicationContext);
        }
        return instance;
    }

    public void setHasTokenChanged(boolean hasTokenChanged) {
        this.hasTokenChanged = hasTokenChanged;
    }

    private void syncReadyToBeSentAnnouncements() {
        if (applicationContext != null) {
            PoolProvider.postIOTask(() -> {
                List<Announcement> readyAnnouncements = AnnouncementCacheManager.getReadyToBeSend();
                if (!readyAnnouncements.isEmpty()) {
                    InstabugAnnouncementSubmitterJob.getInstance().start();
                }
            });
        } else {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Couldn't sync announcements due to null context");
        }

    }

    public static boolean isAnnouncementFeatureAvailable() {
        return InstabugCore.isFeatureAvailable(IBGFeature.ANNOUNCEMENTS);
    }

    public static boolean isAnnouncementFeatureEnabled() {
        return InstabugCore.getFeatureState(IBGFeature.ANNOUNCEMENTS)
                == Feature.State.ENABLED;
    }

    public void startFetching(String locale) {
        if (applicationContext != null) {
            try {
                if (isFeaturesFetchedBefore() && isAnnouncementFeatureEnabled()) {
                    if ((TimeUtils.currentTimeMillis() - AnnouncementsSettings.getInstance().getLastFetchedAt()) > 10000) {
                        AnnouncementsService.getInstance().fetchAnnouncements(locale, new Request.Callbacks<JSONObject, Throwable>() {
                            @Override
                            public void onSucceeded(@Nullable JSONObject response) {
                                try {
                                    AnnouncementsSettings.getInstance().setLastFetchedAt(TimeUtils.currentTimeMillis());
                                    if (response != null) {
                                        List<Announcement> remoteAnnouncements = Announcement.fromJson(response);
                                        onFetchingSucceeded(remoteAnnouncements);
                                    } else {
                                        onFetchingFailed(new NullPointerException("json response is null"));
                                    }
                                } catch (JSONException jSONException) {
                                    onFetchingFailed(jSONException);
                                }
                            }

                            @Override
                            public void onFailed(Throwable error) {
                                onFetchingFailed(error);
                            }
                        });
                    } else {
                        showFirstValidAnnouncement();
                    }
                }
            } catch (JSONException e) {
                onFetchingFailed(e);
                InstabugSDKLogger.e(Constants.LOG_TAG, "Something went wrong while fetching announcements", e);
            }
        }
    }

    private void onFetchingFailed(Throwable error) {
        InstabugSDKLogger.e(Constants.LOG_TAG, ANNOUNCEMENTS_FETCHING_ERROR + error.getMessage());
        startAutomaticAnnouncementTrigger();
    }

    @VisibleForTesting
    void onFetchingSucceeded(final List<Announcement> announcements) {
        InstabugSDKLogger.d(Constants.LOG_TAG, "Announcement Fetching Succeeded");
        if (!Instabug.isEnabled()) {
            return;
        }
        cacheLastRetrievedLocale();
        setShowingIntervals(announcements);
        invalidateAnnouncementUserInteractions(announcements);
        invalidateCachedAnnouncements(announcements);
        migrateAnnouncements(announcements);
        startAutomaticAnnouncementTrigger();
        hasTokenChanged = false;
    }

    private void cacheLastRetrievedLocale() {
        if (applicationContext != null)
            AnnouncementsSettings.setLastRetrievedLocale(LocaleUtils.getCurrentLocaleResolved(applicationContext));
    }

    @WorkerThread
    private void startAutomaticAnnouncementTrigger() {
        List<Announcement> versionAnnouncements = AnnouncementCacheManager.getAnnouncementsByType(AnnouncementType.UPDATE_MSG);
        List<Announcement> whatsNewAnnouncements = AnnouncementCacheManager.getAnnouncementsByType(AnnouncementType.WHAT_IS_NEW);
        if (versionAnnouncements.size() > 0) {
            for (Announcement versionAnnouncement : versionAnnouncements) {
                if (versionAnnouncement.shouldShow()) {
                    showFirstValidAnnouncement();
                    return;
                }
            }
        }
        if (whatsNewAnnouncements.size() > 0) {
            showFirstValidAnnouncement();
        }
    }

    @WorkerThread
    @VisibleForTesting
    void migrateAnnouncements(final List<Announcement> remoteAnnouncements) {
        if (remoteAnnouncements == null) return;

        for (Announcement announcement : remoteAnnouncements) {
            if (announcement != null)
                if (AnnouncementCacheManager.isAnnouncementExist(announcement.getId())) {
                    Announcement cachedAnnouncement = AnnouncementCacheManager.getAnnouncement(announcement.getId());
                    boolean publishStatusChanged = isPublishStatusChanged(announcement, cachedAnnouncement);
                    boolean localeChanged = isLocaleChanged(announcement, cachedAnnouncement);
                    if (announcement.getAssetsStatus() == Announcement.AnnouncementAssetsStatus.NOT_AVAILABLE) {
                        NewFeaturesAssetsHelper.downloadAssets(announcement);
                    }
                    if (publishStatusChanged || localeChanged) {
                        AnnouncementCacheManager.insertOrUpdatePausedOrLocale(announcement, publishStatusChanged,
                                localeChanged);
                    }
                } else if (!announcement.isPaused()) {
                    NewFeaturesAssetsHelper.downloadAssets(announcement);
                    AnnouncementCacheManager.addAnnouncement(announcement);
                }
        }
    }

    @VisibleForTesting
    boolean isLocaleChanged(Announcement remoteAnnouncement, @Nullable Announcement cachedAnnouncement) {
        if (cachedAnnouncement == null) return false;
        return remoteAnnouncement.getLocalization().getCurrentLocale() != null &&
                !remoteAnnouncement.getLocalization().getCurrentLocale()
                        .equals(cachedAnnouncement.getLocalization().getCurrentLocale());
    }

    @VisibleForTesting
    boolean isPublishStatusChanged(Announcement announcement, @Nullable Announcement cachedAnnouncement) {
        if (cachedAnnouncement == null) return false;
        return cachedAnnouncement.isPaused() != announcement.isPaused();
    }

    /**
     * This method to invalidate user interactions if survey is removed then any related user interactions
     * will be removed as well.
     * <p>
     * Order of calling this method must be considered, as this method will be triggered before invalidating surveys
     *
     * @param remoteAnnouncement to check it aganist the cached version.
     */
    @WorkerThread
    @VisibleForTesting
    void invalidateAnnouncementUserInteractions(List<Announcement> remoteAnnouncement) {
        List<Announcement> allAnnouncement = AnnouncementCacheManager.getAllAnnouncement();
        String uuid = UserManagerWrapper.getUserUUID();
        List<UserInteraction> userInteractionsToBeRemoved = new ArrayList<>();
        for (Announcement announcement : allAnnouncement) {
            if (!remoteAnnouncement.contains(announcement)) {
                UserInteraction userInteraction = UserInteractionCacheManager.retrieveUserInteraction(
                        announcement.getId(),
                        uuid,
                        UserInteractionOnTypes.ANNOUNCEMENT
                );
                if (userInteraction != null) {
                    userInteractionsToBeRemoved.add(userInteraction);
                }
            }
        }
        if (!userInteractionsToBeRemoved.isEmpty())
            UserInteractionCacheManager.deleteBulkOfUserInteractions(userInteractionsToBeRemoved);


    }

    @WorkerThread
    public void invalidateCachedAnnouncements(List<Announcement> remoteAnnouncement) {
        List<Announcement> allAnnouncement = AnnouncementCacheManager.getAllAnnouncement();
        for (Announcement announcement : allAnnouncement) {
            if (!remoteAnnouncement.contains(announcement)) {
                AnnouncementCacheManager.deleteAnnouncement(String.valueOf(announcement.getId()));
            }
        }
    }

    @WorkerThread
    private void startAnnouncementActivity(final Announcement announcementToShow) {
        Context applicationContext = Instabug.getApplicationContext();
        if (applicationContext != null) {
            ActivityManager am = (ActivityManager) applicationContext.getSystemService(Context.ACTIVITY_SERVICE);
            ComponentName topActivity = am.getRunningTasks(1).get(0).topActivity;
            String currentActivityName = "";
            if (topActivity != null) {
                currentActivityName = topActivity.getClassName();
            }
            if (!AnnouncementActivity.class.getName().equals(currentActivityName))
                AutoShowingManager.getInstance().showAnnouncement(announcementToShow);
            else {
                InstabugSDKLogger.d(Constants.LOG_TAG, "An announcement is being displayed. Skip showing another one");
            }
        }
    }

    @VisibleForTesting
    void setShowingIntervals(List<Announcement> announcements) {
        for (Announcement announcement : announcements) {
            if (announcement.getType() == AnnouncementType.UPDATE_MSG) {
                AnnouncementsSettings.getInstance().setUpdateMsgShowInterval(
                        announcement.getTarget().getTrigger().getTriggerAfter());
            } else if (announcement.getType() == AnnouncementType.WHAT_IS_NEW) {
                AnnouncementsSettings.getInstance().setWhatIsNewShowInterval(
                        announcement.getTarget().getTrigger().getTriggerAfter());
            }
        }
    }

    private boolean isFeaturesFetchedBefore() {
        return InstabugCore.isFeaturesFetchedBefore();
    }

    public void setAppLatestVersion() {
        if (PersistableSettings.getInstance() == null) return;
        PersistableSettings.getInstance()
                .setAppLatestVersion(DeviceStateProvider.getAppVersion(applicationContext));
    }

    @VisibleForTesting
    void showFirstValidAnnouncement() {
        if (!canShowAnnouncement())
            return;
        final Announcement announcementToShow = getAnnouncementValidator().getFirstValidAnnouncement();
        if (announcementToShow != null) {
            PoolProvider.postIOTask(() -> {
                try {
                    Thread.sleep(announcementToShow.getTarget().getTrigger().getTriggerAfter() * 1000L);
                    startAnnouncementActivity(announcementToShow);
                } catch (InterruptedException e) {
                    if (announcementToShow.getType() == AnnouncementType.UPDATE_MSG)
                        InstabugSDKLogger.e(Constants.LOG_TAG, "Something went wrong while scheduling update msg announcement", e);
                    else if (announcementToShow.getType() == AnnouncementType.WHAT_IS_NEW)
                        InstabugSDKLogger.e(Constants.LOG_TAG, "Something went wrong while scheduling what's new announcement", e);
                }
            });
        }
    }

    @VisibleForTesting
    boolean canShowAnnouncement() {
        return isAnnouncementFeatureAvailable() && isAnnouncementFeatureEnabled() && !hasTokenChanged;
    }

    /**
     * This method to merge user interaction, check cached user interaction for the remote survey
     * if the survey already has user interaction then update this survey, otherwise treat it as a new one.
     *
     * @param remoteAnnouncement .
     */
    @WorkerThread
    private void mergeUserInteraction(List<Announcement> remoteAnnouncement) {
        final String uuid = UserManagerWrapper.getUserUUID();
        List<Announcement> announcementToBeUpdated = new ArrayList<>();
        for (Announcement announcement : remoteAnnouncement) {
            UserInteraction userInteraction = UserInteractionCacheManager.retrieveUserInteraction(
                    announcement.getId(),
                    uuid,
                    UserInteractionOnTypes.ANNOUNCEMENT);
            if (userInteraction != null) {
                announcement.setUserInteraction(userInteraction);
                announcementToBeUpdated.add(announcement);
            }
        }

        if (!remoteAnnouncement.isEmpty()) {
            AnnouncementCacheManager.updateBulk(announcementToBeUpdated);
        }
    }

    public void notifyLogout() {
        UserManagerWrapper.getUUIDAsync(uuid -> {
            // Already called from a BG thread
            final List<Announcement> cachedAnnouncements = AnnouncementCacheManager.getAllAnnouncement();
            if (!cachedAnnouncements.isEmpty()) {
                UserInteractionCacheManager.insertUserInteractions(cachedAnnouncements, uuid);
                AnnouncementCacheManager.resetAnnouncementUserInteraction(cachedAnnouncements);
            }
        });
    }

    /**
     * This method to when user logged in, trigger merge user interactions into announcements
     */
    public void notifyUserLoggedIn() {
        PoolProvider.postIOTask(() -> {
            final List<Announcement> cachedAnnouncements = AnnouncementCacheManager.getAllAnnouncement();
            if (!cachedAnnouncements.isEmpty()) {
                mergeUserInteraction(cachedAnnouncements);
            }
        });
    }

    public void notifyAppVersionChanged() {
        PoolProvider.postIOTask(() -> {
            announcementValidator = null;
            startAutomaticAnnouncementTrigger();
        });
    }
}
