package com.instabug.survey.ui.custom;

import static android.util.TypedValue.COMPLEX_UNIT_DIP;
import static android.util.TypedValue.COMPLEX_UNIT_SP;
import static android.util.TypedValue.applyDimension;

import android.annotation.SuppressLint;
import android.annotation.TargetApi;
import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.CornerPathEffect;
import android.graphics.Paint;
import android.graphics.Path;
import android.graphics.PorterDuff;
import android.graphics.PorterDuffXfermode;
import android.graphics.Rect;
import android.graphics.RectF;
import android.os.Build;
import android.os.Parcel;
import android.os.Parcelable;
import android.util.AttributeSet;
import android.util.Log;
import android.view.MotionEvent;
import android.view.View;

import androidx.annotation.ColorInt;
import androidx.annotation.Dimension;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.core.view.ViewCompat;
import androidx.customview.widget.ExploreByTouchHelper;

import com.instabug.library.InstabugColorTheme;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.internal.device.InstabugDeviceProperties;
import com.instabug.survey.R;

/**
 * Created by zak on 1/28/18.
 */
@SuppressLint("ERADICATE_FIELD_NOT_INITIALIZED")
public abstract class RatingAbstractView extends View implements ChildViewsProvider {
    // Configurable variables
    private @ColorInt
    int borderColor;
    private @ColorInt
    int fillColor;
    private @ColorInt
    int backgroundColor;
    private @ColorInt
    int starBackgroundColor;
    private @ColorInt
    int pressedBorderColor;
    private @ColorInt
    int pressedFillColor;
    private @ColorInt
    int pressedBackgroundColor;
    private @ColorInt
    int pressedStarBackgroundColor;
    private float stepSize;
    private float rating;
    @NonNull
    private Gravity gravity;
    private float starBorderWidth;
    private float starCornerRadius;
    private boolean drawBorderEnabled;

    // Internal variables
    private float currentStarSize;
    @Nullable
    private OnRatingBarChangeListener ratingListener;
    @Nullable
    private OnClickListener clickListener;
    private boolean touchInProgress;
    @Nullable
    private float[] starVertex;
    @Nullable
    private RectF starsDrawingSpace;
    @Nullable
    private RectF starsTouchSpace;

    @Nullable
    private Canvas internalCanvas;
    @Nullable
    private Bitmap internalBitmap;


    private Path starPath;
    private Paint paintStarOutline;
    private CornerPathEffect cornerPathEffect;
    private Paint paintStarBorder;
    private Paint paintStarBackground;
    private Paint paintStarFill;
    private float defaultStarSize;
    private int numberOfStars = 5;
    private float desiredStarSize = Integer.MAX_VALUE, maxStarSize = Integer.MAX_VALUE;
    private float starsSeparation = (int) valueToPixels(4, Dimension.DP);
    private final Rect[] starsRect = new Rect[5];


    public RatingAbstractView(Context context) {
        super(context);
        initView();
    }

    public RatingAbstractView(Context context, AttributeSet attrs) {
        super(context, attrs);
        init();
        initView();
    }

    public RatingAbstractView(Context context, AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        init();
        initView();
    }

    /**
     * Inits paint objects and default values.
     */
    private void initView() {
        starPath = new Path();
        cornerPathEffect = new CornerPathEffect(starCornerRadius);

        paintStarOutline = new Paint(Paint.ANTI_ALIAS_FLAG | Paint.DITHER_FLAG);
        paintStarOutline.setStyle(Paint.Style.FILL_AND_STROKE);
        paintStarOutline.setAntiAlias(true);
        paintStarOutline.setDither(true);
        paintStarOutline.setStrokeJoin(Paint.Join.ROUND);
        paintStarOutline.setStrokeCap(Paint.Cap.ROUND);
        paintStarOutline.setColor(Color.BLACK);
        paintStarOutline.setPathEffect(cornerPathEffect);

        paintStarBorder = new Paint(Paint.ANTI_ALIAS_FLAG | Paint.DITHER_FLAG);
        paintStarBorder.setStyle(Paint.Style.STROKE);
        paintStarBorder.setStrokeJoin(Paint.Join.ROUND);
        paintStarBorder.setStrokeCap(Paint.Cap.ROUND);
        paintStarBorder.setStrokeWidth(starBorderWidth);
        paintStarBorder.setPathEffect(cornerPathEffect);

        paintStarBackground = new Paint(Paint.ANTI_ALIAS_FLAG | Paint.DITHER_FLAG);
        paintStarBackground.setStyle(Paint.Style.FILL_AND_STROKE);
        paintStarBackground.setAntiAlias(true);
        paintStarBackground.setDither(true);
        paintStarBackground.setStrokeJoin(Paint.Join.ROUND);
        paintStarBackground.setStrokeCap(Paint.Cap.ROUND);

        paintStarFill = new Paint(Paint.ANTI_ALIAS_FLAG | Paint.DITHER_FLAG);
        paintStarFill.setStyle(Paint.Style.FILL_AND_STROKE);
        paintStarFill.setAntiAlias(true);
        paintStarFill.setDither(true);
        paintStarFill.setStrokeJoin(Paint.Join.ROUND);
        paintStarFill.setStrokeCap(Paint.Cap.ROUND);

        defaultStarSize = applyDimension(COMPLEX_UNIT_DIP, 30, getResources().getDisplayMetrics());
        ExploreByTouchHelper accessibilityDelegate = new CustomExploreByTouchHelper(new RatingVirtualViewsProvider(this));
        ViewCompat.setAccessibilityDelegate(this, accessibilityDelegate);
        setOnHoverListener((view, event) -> accessibilityDelegate.dispatchHoverEvent(event));
    }

    private void init() {
        borderColor = getResources().getColor(R.color.survey_rate_star_border);
        fillColor = getResources().getColor(R.color.survey_rate_selected);
        starBackgroundColor =
                InstabugCore.getTheme() == InstabugColorTheme.InstabugColorThemeLight
                        ? getResources().getColor(R.color.survey_rate_unselected_light)
                        : getResources().getColor(R.color.survey_rate_unselected_dark);
        backgroundColor = Color.TRANSPARENT;
        pressedBorderColor = borderColor;
        pressedFillColor = fillColor;
        pressedStarBackgroundColor = starBackgroundColor;
        pressedBackgroundColor = backgroundColor;
        numberOfStars = 5;
        starsSeparation = (int) valueToPixels(16, Dimension.DP);
        maxStarSize = (int) valueToPixels(InstabugDeviceProperties.isTablet(getContext()) ? 80 : 52, Dimension.DP);
        desiredStarSize = Integer.MAX_VALUE;
        stepSize = 1f;
        starBorderWidth = getStarBorderWidth();
        starCornerRadius = getStarCornerRadius();
        rating = 0f;
        drawBorderEnabled = shouldDrawBorders();
        gravity = Gravity.fromId(Gravity.Left.id);
    }

    protected abstract float getStarBorderWidth();

    protected abstract float getStarCornerRadius();

    protected abstract boolean shouldDrawBorders();

    @Override
    protected void onMeasure(int widthMeasureSpec, int heightMeasureSpec) {
        int widthMode = MeasureSpec.getMode(widthMeasureSpec);
        int widthSize = MeasureSpec.getSize(widthMeasureSpec);
        int heightMode = MeasureSpec.getMode(heightMeasureSpec);
        int heightSize = MeasureSpec.getSize(heightMeasureSpec);

        int width;
        int height;

        //Measure Width
        if (widthMode == MeasureSpec.EXACTLY) {
            //Must be this size
            width = widthSize;
        } else if (widthMode == MeasureSpec.AT_MOST) {
            //Can't be bigger than...
            if (desiredStarSize != Integer.MAX_VALUE) {
                // user specified a specific star size, so there is a desired width
                int desiredWidth = calculateTotalWidth(desiredStarSize, numberOfStars, starsSeparation, true);
                width = Math.min(desiredWidth, widthSize);
            } else if (maxStarSize != Integer.MAX_VALUE) {
                // user specified a max star size, so there is a desired width
                int desiredWidth = calculateTotalWidth(maxStarSize, numberOfStars, starsSeparation, true);
                width = Math.min(desiredWidth, widthSize);
            } else {
                // using defaults
                int desiredWidth = calculateTotalWidth(defaultStarSize, numberOfStars, starsSeparation, true);
                width = Math.min(desiredWidth, widthSize);
            }
        } else {
            //Be whatever you want
            if (desiredStarSize != Integer.MAX_VALUE) {
                // user specified a specific star size, so there is a desired width
                int desiredWidth = calculateTotalWidth(desiredStarSize, numberOfStars, starsSeparation, true);
                width = desiredWidth;
            } else if (maxStarSize != Integer.MAX_VALUE) {
                // user specified a max star size, so there is a desired width
                int desiredWidth = calculateTotalWidth(maxStarSize, numberOfStars, starsSeparation, true);
                width = desiredWidth;
            } else {
                // using defaults
                int desiredWidth = calculateTotalWidth(defaultStarSize, numberOfStars, starsSeparation, true);
                width = desiredWidth;
            }
        }

        float tentativeStarSize = (width - getPaddingLeft() - getPaddingRight() - starsSeparation * (numberOfStars - 1)) / numberOfStars;

        //Measure Height
        if (heightMode == MeasureSpec.EXACTLY) {
            //Must be this size
            height = heightSize;
        } else if (heightMode == MeasureSpec.AT_MOST) {
            //Can't be bigger than...
            if (desiredStarSize != Integer.MAX_VALUE) {
                // user specified a specific star size, so there is a desired width
                int desiredHeight = calculateTotalHeight(desiredStarSize, numberOfStars, starsSeparation, true);
                height = Math.min(desiredHeight, heightSize);
            } else if (maxStarSize != Integer.MAX_VALUE) {
                // user specified a max star size, so there is a desired width
                int desiredHeight = calculateTotalHeight(maxStarSize, numberOfStars, starsSeparation, true);
                height = Math.min(desiredHeight, heightSize);
            } else {
                // using defaults
                int desiredHeight = calculateTotalHeight(tentativeStarSize, numberOfStars, starsSeparation, true);
                height = Math.min(desiredHeight, heightSize);
            }
        } else {
            //Be whatever you want
            if (desiredStarSize != Integer.MAX_VALUE) {
                // user specified a specific star size, so there is a desired width
                int desiredHeight = calculateTotalHeight(desiredStarSize, numberOfStars, starsSeparation, true);
                height = desiredHeight;
            } else if (maxStarSize != Integer.MAX_VALUE) {
                // user specified a max star size, so there is a desired width
                int desiredHeight = calculateTotalHeight(maxStarSize, numberOfStars, starsSeparation, true);
                height = desiredHeight;
            } else {
                // using defaults
                int desiredHeight = calculateTotalHeight(tentativeStarSize, numberOfStars, starsSeparation, true);
                height = desiredHeight;
            }
        }

        //MUST CALL THIS
        setMeasuredDimension(width, height);
    }

    @Override
    protected void onLayout(boolean changed, int left, int top, int right, int bottom) {
        super.onLayout(changed, left, top, right, bottom);

        int width = getWidth();
        int height = getHeight();
        if (desiredStarSize == Integer.MAX_VALUE) {
            currentStarSize = calculateBestStarSize(width, height);
        } else {
            currentStarSize = desiredStarSize;
        }
        performStarSizeAssociatedCalculations(width, height);
    }

    /**
     * Calculates largest possible star size, based on chosen width and height.
     * If maxStarSize is present, it will be considered and star size will not be greater than this value.
     *
     * @param width
     * @param height
     */
    private float calculateBestStarSize(int width, int height) {
        if (maxStarSize != Integer.MAX_VALUE) {
            float desiredTotalWidth = calculateTotalWidth(maxStarSize, numberOfStars, starsSeparation, true);
            float desiredTotalHeight = calculateTotalHeight(maxStarSize, numberOfStars, starsSeparation, true);
            if (desiredTotalWidth >= width || desiredTotalHeight >= height) {
                // we need to shrink the size of the stars
                float sizeBasedOnWidth = (width - getPaddingLeft() - getPaddingRight() - starsSeparation * (numberOfStars - 1)) / numberOfStars;
                float sizeBasedOnHeight = height - getPaddingTop() - getPaddingBottom();
                return Math.min(sizeBasedOnWidth, sizeBasedOnHeight);
            } else {
                return maxStarSize;
            }
        } else {
            // expand the most we can
            float sizeBasedOnWidth = (width - getPaddingLeft() - getPaddingRight() - starsSeparation * (numberOfStars - 1)) / numberOfStars;
            float sizeBasedOnHeight = height - getPaddingTop() - getPaddingBottom();
            return Math.min(sizeBasedOnWidth, sizeBasedOnHeight);
        }
    }

    /**
     * Performs auxiliary calculations to later speed up drawing phase.
     *
     * @param width
     * @param height
     */
    private void performStarSizeAssociatedCalculations(int width, int height) {
        float totalStarsWidth = calculateTotalWidth(currentStarSize, numberOfStars, starsSeparation, false);
        float totalStarsHeight = calculateTotalHeight(currentStarSize, numberOfStars, starsSeparation, false);
        float startingX = (float) (width - getPaddingLeft() - getPaddingRight()) / 2 - totalStarsWidth / 2 + getPaddingLeft();
        float startingY = (float) (height - getPaddingTop() - getPaddingBottom()) / 2 - totalStarsHeight / 2 + getPaddingTop();
        starsDrawingSpace = new RectF(startingX, startingY, startingX + totalStarsWidth, startingY + totalStarsHeight);
        float aux = starsDrawingSpace.width() * 0.05f;
        starsTouchSpace = new RectF(starsDrawingSpace.left - aux, starsDrawingSpace.top, starsDrawingSpace.right + aux, starsDrawingSpace.bottom);

        float bottomFromMargin = currentStarSize * 0.2f;
        float triangleSide = currentStarSize * 0.35f;
        float half = currentStarSize * 0.5f;
        float tipVerticalMargin = currentStarSize * 0.05f;
        float tipHorizontalMargin = currentStarSize * 0.03f;
        float innerUpHorizontalMargin = currentStarSize * 0.38f;
        float innerBottomHorizontalMargin = currentStarSize * 0.32f;
        float innerBottomVerticalMargin = currentStarSize * 0.6f;
        float innerCenterVerticalMargin = currentStarSize * 0.27f;
        float lowerDeviation = getPointsLowerDeviation();
        float upperDeviation = getPointsUpperDeviation();
        float lowerInnerPointsYUpperDeviation = getLowerInnerPointsYUpperDeviation();

        starVertex = new float[]{
                // 1. outer left (Start point of the path to draw)
                /* x */ tipHorizontalMargin,
                /* y */ innerUpHorizontalMargin,
                // 2. inner left
                /* x */ (tipHorizontalMargin + triangleSide) * lowerDeviation,
                /* y */ innerUpHorizontalMargin * lowerDeviation,
                // 3. top tip
                /* x */ half,
                /* y */ tipVerticalMargin,
                // 4. inner right
                /* x */ (currentStarSize - tipHorizontalMargin - triangleSide) * upperDeviation,
                /* y */ innerUpHorizontalMargin * lowerDeviation,
                // 5. outer right
                /* x */ currentStarSize - tipHorizontalMargin,
                /* y */ innerUpHorizontalMargin,
                // 6. inner bottom right
                /* x */ (currentStarSize - innerBottomHorizontalMargin) * upperDeviation,
                /* y */ innerBottomVerticalMargin * lowerInnerPointsYUpperDeviation,
                // 7. outer bottom right
                /* x */ (currentStarSize - bottomFromMargin),
                /* y */ currentStarSize - tipVerticalMargin,
                // 8. inner bottom tip
                /* x */ half,
                /* y */ (currentStarSize - innerCenterVerticalMargin) * upperDeviation,
                // 9. outer bottom left
                /* x */ bottomFromMargin,
                /* y */currentStarSize - tipVerticalMargin,
                // 10. inner bottom left
                /* x */ innerBottomHorizontalMargin * lowerDeviation,
                /* y */ innerBottomVerticalMargin * lowerInnerPointsYUpperDeviation
        };
    }

    protected abstract float getLowerInnerPointsYUpperDeviation();

    protected abstract float getPointsLowerDeviation();

    protected abstract float getPointsUpperDeviation();


    /**
     * Calculates total width to occupy based on several parameters
     *
     * @param starSize
     * @param numberOfStars
     * @param starsSeparation
     * @param padding
     * @return
     */
    private int calculateTotalWidth(float starSize, int numberOfStars, float starsSeparation, boolean padding) {
        return Math.round(starSize * numberOfStars + starsSeparation * (numberOfStars - 1))
                + (padding ? getPaddingLeft() + getPaddingRight() : 0);
    }

    /**
     * Calculates total height to occupy based on several parameters
     *
     * @param starSize
     * @param numberOfStars
     * @param starsSeparation
     * @param padding
     * @return
     */
    private int calculateTotalHeight(float starSize, int numberOfStars, float starsSeparation, boolean padding) {
        return Math.round(starSize) + (padding ? getPaddingTop() + getPaddingBottom() : 0);
    }

    @Override
    protected void onSizeChanged(int w, int h, int oldw, int oldh) {
        super.onSizeChanged(w, h, oldw, oldh);

        generateInternalCanvas(w, h);
    }

    /**
     * Generates internal canvas on which the ratingbar will be drawn.
     *
     * @param w
     * @param h
     */
    private void generateInternalCanvas(int w, int h) {
        if (internalBitmap != null) {
            // avoid leaking memory after losing the reference
            internalBitmap.recycle();
        }

        if (w > 0 && h > 0) {
            // if width == 0 or height == 0 we don't need internal bitmap, cause view won't be drawn anyway.
            internalBitmap = Bitmap.createBitmap(w, h, Bitmap.Config.ARGB_8888);
            if (internalBitmap != null) {
                internalBitmap.eraseColor(Color.TRANSPARENT);
                internalCanvas = new Canvas(internalBitmap);
            }
        }
    }

    @Override
    protected void onDraw(Canvas canvas) {
        super.onDraw(canvas);

        int height = getHeight();
        int width = getWidth();

        if (width == 0 || height == 0) {
            // don't draw view with width or height equal zero.
            return;
        }

        // clean internal canvas
        if (internalCanvas == null) return;

        internalCanvas.drawColor(0, PorterDuff.Mode.CLEAR);

        // choose colors
        setupColorsInPaint();

        // draw stars
        if (gravity == Gravity.Left) {
            drawFromLeftToRight(internalCanvas);
        } else {
            drawFromRightToLeft(internalCanvas);
        }

        // draw view background color
        if (touchInProgress) {
            canvas.drawColor(pressedBackgroundColor);
        } else {
            canvas.drawColor(backgroundColor);
        }

        if (internalBitmap != null) {
            // draw internal bitmap to definite canvas
            canvas.drawBitmap(internalBitmap, 0, 0, null);
        }
    }

    /**
     * Sets the color for the different paints depending on whether current state is pressed or normal.
     */
    private void setupColorsInPaint() {
        if (touchInProgress) {
            paintStarBorder.setColor(pressedBorderColor);
            paintStarFill.setColor(pressedFillColor);
            if (pressedFillColor != Color.TRANSPARENT) {
                paintStarFill.setXfermode(new PorterDuffXfermode(PorterDuff.Mode.SRC_ATOP));
            } else {
                paintStarFill.setXfermode(new PorterDuffXfermode(PorterDuff.Mode.CLEAR));
            }
            paintStarBackground.setColor(pressedStarBackgroundColor);
            if (pressedStarBackgroundColor != Color.TRANSPARENT) {
                paintStarBackground.setXfermode(new PorterDuffXfermode(PorterDuff.Mode.SRC_ATOP));
            } else {
                paintStarBackground.setXfermode(new PorterDuffXfermode(PorterDuff.Mode.CLEAR));
            }
        } else {
            paintStarBorder.setColor(borderColor);
            paintStarFill.setColor(fillColor);
            if (fillColor != Color.TRANSPARENT) {
                paintStarFill.setXfermode(new PorterDuffXfermode(PorterDuff.Mode.SRC_ATOP));
            } else {
                paintStarFill.setXfermode(new PorterDuffXfermode(PorterDuff.Mode.CLEAR));
            }
            paintStarBackground.setColor(starBackgroundColor);
            if (starBackgroundColor != Color.TRANSPARENT) {
                paintStarBackground.setXfermode(new PorterDuffXfermode(PorterDuff.Mode.SRC_ATOP));
            } else {
                paintStarBackground.setXfermode(new PorterDuffXfermode(PorterDuff.Mode.CLEAR));
            }
        }
    }

    /**
     * Draws the view when gravity is Left
     *
     * @param internalCanvas
     */
    private void drawFromLeftToRight(Canvas internalCanvas) {
        float remainingTotalRating = rating;
        if (starsDrawingSpace != null) {
            float startingX = starsDrawingSpace.left;
            float startingY = starsDrawingSpace.top;
            for (int i = 0; i < numberOfStars; i++) {
                if (remainingTotalRating >= 1) {
                    drawStar(internalCanvas, startingX, startingY, 1f, Gravity.Left);
                    remainingTotalRating -= 1;
                } else {
                    drawStar(internalCanvas, startingX, startingY, remainingTotalRating, Gravity.Left);
                    if (drawBorderEnabled) {
                        internalCanvas.drawPath(starPath, paintStarBorder);
                    }
                    remainingTotalRating = 0;
                }
                startingX += starsSeparation + currentStarSize;
            }
        }
    }

    /**
     * Draws the view when gravity is Right
     *
     * @param internalCanvas
     */
    private void drawFromRightToLeft(Canvas internalCanvas) {
        float remainingTotalRating = rating;
        if (starsDrawingSpace != null) {
            float startingX = starsDrawingSpace.right - currentStarSize;
            float startingY = starsDrawingSpace.top;
            for (int i = 0; i < numberOfStars; i++) {
                if (remainingTotalRating >= 1) {
                    drawStar(internalCanvas, startingX, startingY, 1f, Gravity.Right);
                    remainingTotalRating -= 1;
                } else {
                    drawStar(internalCanvas, startingX, startingY, remainingTotalRating, Gravity.Right);
                    if (drawBorderEnabled) {
                        internalCanvas.drawPath(starPath, paintStarBorder);
                    }
                    remainingTotalRating = 0;
                }
                startingX -= starsSeparation + currentStarSize;
            }
        }
    }


    /**
     * Draws a star in the provided canvas.
     *
     * @param canvas
     * @param x       left of the star
     * @param y       top of the star
     * @param filled  between 0 and 1
     * @param gravity Left or Right
     */
    private void drawStar(Canvas canvas, float x, float y, float filled, Gravity gravity) {
        // calculate fill in pixels
        float fill = currentStarSize * filled;

        if (starVertex == null) return;

        // prepare path for star
        starPath.reset();
        starPath.moveTo(x + starVertex[0], y + starVertex[1]);
        for (int i = 2; i < starVertex.length; i = i + 2) {
            starPath.lineTo(x + starVertex[i], y + starVertex[i + 1]);
        }
        starPath.close();

        // draw star outline
        canvas.drawPath(starPath, paintStarOutline);

        // Note: below, currentStarSize*0.02f is a minor correction so the user won't see a vertical black line in between the fill and empty color
        if (gravity == Gravity.Left) {
            // color star fill
            canvas.drawRect(x, y, x + fill + currentStarSize * 0.02f, y + currentStarSize, paintStarFill);
            // draw star background
            canvas.drawRect(x + fill, y, x + currentStarSize, y + currentStarSize, paintStarBackground);
        } else {
            // color star fill
            canvas.drawRect(x + currentStarSize - (fill + currentStarSize * 0.02f), y, x + currentStarSize, y + currentStarSize, paintStarFill);
            // draw star background
            canvas.drawRect(x, y, x + currentStarSize - fill, y + currentStarSize, paintStarBackground);
        }
    }

    @Override
    public boolean onTouchEvent(MotionEvent event) {
        int action = event.getAction() & MotionEvent.ACTION_MASK;
        switch (action) {
            case MotionEvent.ACTION_DOWN:
            case MotionEvent.ACTION_MOVE:
                // check if action is performed on stars
                if (starsTouchSpace != null && starsTouchSpace.contains(event.getX(), event.getY())) {
                    touchInProgress = true;
                    setNewRatingFromTouch(event.getX(), event.getY());
                } else {
                    if (touchInProgress && ratingListener != null) {
                        ratingListener.onRatingChanged(this, rating, true);
                    }
                    touchInProgress = false;
                    return false;
                }
                break;
            case MotionEvent.ACTION_UP:
                setNewRatingFromTouch(event.getX(), event.getY());
                if (clickListener != null) {
                    clickListener.onClick(this);
                }
                if (ratingListener != null) {
                    ratingListener.onRatingChanged(this, rating, true);
                }
                touchInProgress = false;
                break;
            case MotionEvent.ACTION_CANCEL:
                if (ratingListener != null) {
                    ratingListener.onRatingChanged(this, rating, true);
                }
                touchInProgress = false;
                break;
            default:
                break;

        }

        invalidate();
        return true;
    }

    /**
     * Assigns a rating to the touch event.
     *
     * @param x
     * @param y
     */
    private void setNewRatingFromTouch(float x, float y) {
        // normalize x to inside starsDrawinSpace
        if (gravity != Gravity.Left) {
            x = getWidth() - x;
        }

        if (starsDrawingSpace == null) return;

        // we know that touch was inside starsTouchSpace, but it might be outside starsDrawingSpace
        if (x < starsDrawingSpace.left) {
            rating = 0;
            return;
        } else if (x > starsDrawingSpace.right) {
            rating = numberOfStars;
            return;
        }

        x = x - starsDrawingSpace.left;
        // reduce the width to allow the user reach the top and bottom values of rating (0 and numberOfStars)
        rating = (float) numberOfStars / starsDrawingSpace.width() * x;

        // correct rating in case step size is present
        float mod = rating % stepSize;
        if (mod < stepSize / 4) {
            rating = rating - mod;
            rating = Math.max(0, rating);
        } else {
            rating = rating - mod + stepSize;
            rating = Math.min(numberOfStars, rating);
        }
    }

    @Override
    @Nullable
    protected Parcelable onSaveInstanceState() {
        Parcelable superState = super.onSaveInstanceState();
        if (superState != null) {
            SavedState savedState = new SavedState(superState);
            savedState.rating = getRating();
        }
        return null;
    }

    @Override
    protected void onRestoreInstanceState(Parcelable state) {
        if (state != null) {
            SavedState savedState = (SavedState) state;
            super.onRestoreInstanceState(savedState.getSuperState());
            setRating(savedState.rating, false);
        }
    }

    public float getRating() {
        return rating;
    }

    /**
     * Sets rating.
     * If provided value is less than 0, rating will be set to 0.
     * * If provided value is greater than numberOfStars, rating will be set to numberOfStars.
     *
     * @param rating
     */
    public void setRating(float rating, boolean shouldCallListener) {
        this.rating = normalizeRating(rating);
        // request redraw of the view
        invalidate();
        if (shouldCallListener && ratingListener != null) {
            ratingListener.onRatingChanged(this, rating, false);
        }
    }

    public @ColorInt
    int getFillColor() {
        return fillColor;
    }

    /**
     * Sets fill color of stars in normal state.
     *
     * @param fillColor
     */
    public void setFillColor(@ColorInt int fillColor) {
        this.fillColor = fillColor;
        // request redraw of the view
        invalidate();
    }

    public Gravity getGravity() {
        return gravity;
    }

    /**
     * Sets gravity of fill.
     *
     * @param gravity
     */
    public void setGravity(Gravity gravity) {
        this.gravity = gravity;
        // request redraw of the view
        invalidate();
    }

    /**
     * Convenience method to convert a value in the given dimension to pixels.
     *
     * @param value
     * @param dimen
     * @return
     */
    protected float valueToPixels(float value, @Dimension int dimen) {
        switch (dimen) {
            case Dimension.DP:
                return applyDimension(COMPLEX_UNIT_DIP, value, getResources().getDisplayMetrics());
            case Dimension.SP:
                return applyDimension(COMPLEX_UNIT_SP, value, getResources().getDisplayMetrics());
            default:
                return value;
        }
    }

    /**
     * Convenience method to convert a value from pixels to the given dimension.
     *
     * @param value
     * @param dimen
     * @return
     */
    private float valueFromPixels(float value, @Dimension int dimen) {
        switch (dimen) {
            case Dimension.DP:
                return value / getResources().getDisplayMetrics().density;
            case Dimension.SP:
                return value / getResources().getDisplayMetrics().scaledDensity;
            default:
                return value;
        }
    }

    /**
     * Normalizes rating passed by argument between 0 and numberOfStars.
     *
     * @param rating
     * @return
     */
    private float normalizeRating(float rating) {
        if (rating < 0) {
            Log.w("RatingView", String.format("Assigned rating is less than 0 (%f < 0), I will set it to exactly 0", rating));
            return 0;
        } else if (rating > numberOfStars) {
            Log.w("RatingView", String.format("Assigned rating is greater than numberOfStars (%f > %d), I will set it to exactly numberOfStars", rating, numberOfStars));
            return numberOfStars;
        } else {
            return rating;
        }
    }

    /**
     * Sets OnClickListener.
     *
     * @param listener
     */
    @Override
    public void setOnClickListener(@Nullable OnClickListener listener) {
        this.clickListener = listener;
    }

    /**
     * Sets OnRatingBarChangeListener.
     *
     * @param listener
     */
    public void setOnRatingBarChangeListener(OnRatingBarChangeListener listener) {
        this.ratingListener = listener;
    }

    @Override
    public int childPositionAt(float x, float y) {
        for (int i = 0; i < starsRect.length; i++) {
            Rect rect = starsRect[i];
            if (rect != null && rect.contains((int) x, (int) y)) {
                return i + 1;
            }
        }
        return ExploreByTouchHelper.INVALID_ID;
    }


    @Nullable
    @Override
    public Rect childBoundsAtPosition(int position) {
        int star = position > 0 ? position - 1 : position;
        if (starsDrawingSpace == null) return null;
        float startingX = starsDrawingSpace.left + (star * (starsSeparation + currentStarSize));
        float startingY = starsDrawingSpace.top;

        Rect rect = new Rect();
        rect.top = (int) startingY;
        rect.left = (int) (startingX);
        rect.bottom = (int) ((startingY) + currentStarSize);
        rect.right = (int) ((startingX) + currentStarSize);
        int index = position - 1;
        starsRect[index] = rect;
        return rect;
    }

    /**
     * Represents gravity of the fill in the bar.
     */
    public enum Gravity {
        /**
         * Left gravity is default: the bar will be filled starting from left to right.
         */
        Left(0),
        /**
         * Right gravity: the bar will be filled starting from right to left.
         */
        Right(1);

        int id;

        Gravity(int id) {
            this.id = id;
        }

        static Gravity fromId(int id) {
            for (Gravity f : values()) {
                if (f.id == id) return f;
            }
            // default value
            Log.w("RatingView", String.format("Gravity chosen is neither 'left' nor 'right', I will set it to Left"));
            return Left;
        }
    }

    public interface OnRatingBarChangeListener {

        /**
         * Notification that the rating has changed. Clients can use the
         * fromUser parameter to distinguish user-initiated changes from those
         * that occurred programmatically. This will not be called continuously
         * while the user is dragging, only when the user finalizes a rating by
         * lifting the touch.
         *
         * @param RatingView The RatingBar whose rating has changed.
         * @param rating     The current rating. This will be in the range
         *                   0..numStars.
         * @param fromUser   True if the rating change was initiated by a user's
         *                   touch gesture or arrow key/horizontal trackbell movement.
         */
        void onRatingChanged(RatingAbstractView RatingView, float rating, boolean fromUser);

    }

    private static class SavedState extends BaseSavedState {
        public static final Parcelable.Creator<SavedState> CREATOR = new Creator<SavedState>() {
            @Override
            public SavedState createFromParcel(Parcel parcel) {
                return new SavedState(parcel);
            }

            @Override
            public SavedState[] newArray(int size) {
                return new SavedState[size];
            }
        };
        private float rating = 0.0f;

        protected SavedState(Parcel source) {
            super(source);
            rating = source.readFloat();
        }

        @TargetApi(Build.VERSION_CODES.N)
        protected SavedState(Parcel source, ClassLoader loader) {
            super(source, loader);
        }

        protected SavedState(Parcelable superState) {
            super(superState);
        }

        @Override
        public void writeToParcel(Parcel out, int flags) {
            super.writeToParcel(out, flags);
            out.writeFloat(rating);
        }
    }

}
