package com.instabug.survey.settings;

import android.annotation.SuppressLint;
import android.content.Context;
import android.content.SharedPreferences;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.instabug.library.Instabug;
import com.instabug.library.internal.servicelocator.CoreServiceLocator;

/**
 * @author tarek
 */
public class PersistableSettings {

    private static final String IB_LAST_SURVEY_TIME = "last_survey_time";
    private static final String IB_SURVEY_LAST_FETCH = "survey_last_fetch_time";
    private static final String IB_SURVEY_COUNTRY_CODE_RESOLVER_API = "survey_resolve_country_code";
    private static final String IB_SURVEY_COUNTRY_CODE_RESOLVER_LAST_FETCH = "survey_resolve_country_code_last_fetch";

    private static final String INSTABUG_SURVEY_SHARED_PREF_NAME = "instabug_survey";
    /**
     * For old cache file deletion
     */
    private static final String IB_SURVEYS_HAS_BEEN_MIGRATED = "should_remove_old_survey_cache_file";

    @Nullable
    private SharedPreferences sharedPreferences;
    @Nullable
    private SharedPreferences.Editor editor;
    @Nullable
    private static PersistableSettings persistableSettings;

    @SuppressLint("CommitPrefEdits")
    private PersistableSettings(@NonNull Context context) {
        if (context == null) return;
        sharedPreferences =
                CoreServiceLocator.getInstabugSharedPreferences(context,
                        INSTABUG_SURVEY_SHARED_PREF_NAME);
        if (sharedPreferences != null) {
            editor = sharedPreferences.edit();
        }
    }

    public static void init(@Nullable Context context) {
        if (context == null) return;

        persistableSettings = new PersistableSettings(context);
    }

    @Nullable
    public static PersistableSettings getInstance() {
        if (persistableSettings == null)
            init(Instabug.getApplicationContext());
        return persistableSettings;
    }

    public long getLastSurveyTime() {
        if (sharedPreferences != null) {
            return sharedPreferences.getLong(IB_LAST_SURVEY_TIME, 0L);
        }
        return 0L;
    }

    public void setLastSurveyTime(long time) {
        if (editor != null) {
            editor.putLong(IB_LAST_SURVEY_TIME, time);
            editor.apply();
        }
    }

    public long getLastFetchedAt() {
        if (sharedPreferences != null) {
            return sharedPreferences.getLong(IB_SURVEY_LAST_FETCH, 0L);
        }
        return 0l;
    }

    public void setLastFetchedAt(long time) {
        if (editor != null) {
            editor.putLong(IB_SURVEY_LAST_FETCH, time);
            editor.apply();
        }
    }

    @Nullable
    public String getCountryCode() {
        if (sharedPreferences != null) {
            return sharedPreferences.getString(IB_SURVEY_COUNTRY_CODE_RESOLVER_API, null);
        }
        return null;
    }

    public void setCountryCode(String countryCode) {
        if (editor != null) {
            editor.putString(IB_SURVEY_COUNTRY_CODE_RESOLVER_API, countryCode);
            editor.apply();
        }
    }

    public long getCountryCodeLastFetch() {
        if (sharedPreferences != null) {
            return sharedPreferences.getLong(IB_SURVEY_COUNTRY_CODE_RESOLVER_LAST_FETCH, 0L);
        }
        return 0L;
    }

    public void setCountryCodeLastFetch(long timestamp) {
        if (editor != null) {
            editor.putLong(IB_SURVEY_COUNTRY_CODE_RESOLVER_LAST_FETCH, timestamp);
            editor.apply();
        }
    }

    public static void release() {
        persistableSettings = null;
    }

    /**
     * This to check if we deleted the survey old cache file or not
     */
    public boolean hasBeenMigrated() {
        if (sharedPreferences != null) {
            return sharedPreferences.getBoolean(IB_SURVEYS_HAS_BEEN_MIGRATED, false);
        }
        return false;
    }

    /**
     * This to mark that we deleted the survey cache file
     */
    public void markSurveysFilesMigrated() {
        if (editor != null) {
            editor.putBoolean(IB_SURVEYS_HAS_BEEN_MIGRATED, true);
            editor.apply();
        }
    }
}