package com.instabug.survey.announcements.ui.fragment.whatsnew;

import android.app.Activity;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.TextView;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.core.view.ViewCompat;
import androidx.recyclerview.widget.RecyclerView;

import com.instabug.library.util.BitmapUtils;
import com.instabug.library.view.ViewUtils;
import com.instabug.survey.R;
import com.instabug.survey.announcements.cache.AnnouncementCacheManager;
import com.instabug.survey.announcements.models.AnnouncementItem;
import com.instabug.survey.announcements.models.NewFeature;

import org.jetbrains.annotations.NotNull;

public class WhatsNewFeaturesAdapter extends RecyclerView.Adapter<WhatsNewFeaturesAdapter.NewFeatureViewHolder> {
    private final LayoutInflater inflater;
    private final AnnouncementItem announcementItem;

    WhatsNewFeaturesAdapter(Activity activity, AnnouncementItem announcementItem) {
        inflater = LayoutInflater.from(activity);
        this.announcementItem = announcementItem;
    }

    @NotNull
    @Override
    public NewFeatureViewHolder onCreateViewHolder(@NonNull ViewGroup viewGroup, int viewType) {
        return new NewFeatureViewHolder(inflater.inflate(
                R.layout.instabug_announcement_new_feature_item, viewGroup, false));
    }

    @Override
    public void onBindViewHolder(@NonNull final NewFeatureViewHolder newFeatureViewHolder, int position) {
        NewFeature newFeature = getItem(position);
        if (newFeature != null) {
            newFeatureViewHolder.bindFeatureText(newFeature);
            if (announcementItem != null) {
                newFeatureViewHolder.handleAnnouncementIcon(newFeature);
            }
        }
    }

    @Nullable
    private NewFeature getItem(int position) {
        if (announcementItem.getNewFeatures() == null) return null;

        return announcementItem.getNewFeatures().get(position);
    }

    @Override
    public long getItemId(int position) {
        return position;
    }

    @Override
    public int getItemCount() {
        return announcementItem != null && announcementItem.getNewFeatures() != null ? announcementItem.getNewFeatures().size() : 0;
    }

    class NewFeatureViewHolder extends RecyclerView.ViewHolder {

        @Nullable
        private final LinearLayout textContainer;
        @Nullable
        private final TextView title;
        @Nullable
        private final TextView description;
        @Nullable
        private final ImageView icon;

        NewFeatureViewHolder(@NonNull View itemView) {
            super(itemView);
            textContainer = itemView.findViewById(R.id.feature_content_container);
            title = itemView.findViewById(R.id.new_feature_title);
            description = itemView.findViewById(R.id.new_feature_description);
            icon = itemView.findViewById(R.id.new_feature_img);
        }

        void bindFeatureText(@NonNull NewFeature feature) {
            setTextContainerSpacing();
            if (title != null) {
                title.setText(feature.getTitle() != null ? feature.getTitle() : "");
            }
            if (description != null) {
                description.setText(feature.getDescription() != null ? feature.getDescription() : "");
            }
        }

        private void setTextContainerSpacing() {
            int containerStartPadding = 0;
            if (!announcementItem.hasNoIcons() && textContainer != null) {
                containerStartPadding = ViewUtils.convertDpToPx(itemView.getContext(), 16);
            }
            if (ViewCompat.getLayoutDirection(itemView) == ViewCompat.LAYOUT_DIRECTION_RTL && textContainer != null) {
                textContainer.setPadding(0, 0, containerStartPadding, 0);
            } else if (ViewCompat.getLayoutDirection(itemView) == ViewCompat.LAYOUT_DIRECTION_LTR && textContainer != null) {
                textContainer.setPadding(containerStartPadding, 0, 0, 0);
            }
        }

        void handleAnnouncementIcon(@NonNull NewFeature feature) {
            if (announcementItem != null && announcementItem.hasNoIcons() && icon != null) {
                icon.setPadding(0, icon.getPaddingTop(), 0, icon.getPaddingBottom());
                icon.setVisibility(View.GONE);
            } else if (announcementItem != null && !announcementItem.hasNoIcons() && icon != null) {
                icon.setVisibility(View.VISIBLE);
                String pathname = AnnouncementCacheManager.getAnnouncementAsset(announcementItem.getId(), feature.getId());
                if (pathname != null) {
                    BitmapUtils.loadBitmapWithFallback(pathname, icon, R.drawable.ibg_survey_ic_star_icon_placholder);
                } else {
                    icon.setImageResource(R.drawable.ibg_survey_ic_star_icon_placholder);
                }
            }
        }

    }
}