package com.instabug.survey.announcements.ui.activity;

import android.animation.ValueAnimator;
import android.os.Bundle;
import android.os.Handler;
import android.view.ViewGroup;
import android.widget.FrameLayout;
import android.widget.RelativeLayout;

import com.instabug.library._InstabugActivity;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.core.plugin.Plugin;
import com.instabug.library.core.ui.BackPressHandler;
import com.instabug.library.core.ui.BaseFragmentActivity;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.StatusBarUtils;
import com.instabug.survey.Constants;
import com.instabug.survey.R;
import com.instabug.survey.SurveyPlugin;
import com.instabug.survey.SurveysManager;
import com.instabug.survey.announcements.models.Announcement;
import com.instabug.survey.announcements.ui.fragment.versionupdate.UpdateMessageFragment;
import com.instabug.survey.utils.SurveysUtils;
import com.instabug.survey.utils.ThemeResolver;

import androidx.annotation.Nullable;
import androidx.core.content.ContextCompat;
import androidx.fragment.app.Fragment;

public class AnnouncementActivity extends BaseFragmentActivity<AnnouncementPresenter> implements
        AnnouncementActivityContract.View,
        _InstabugActivity, AnnouncementActivityCallback {

    public static final String KEY_ANNOUNCEMENT = "announcement";
    boolean isResumed = false;

    private FrameLayout fragmentContainer;
    @Nullable
    private Announcement announcement;
    @Nullable
    private Handler finishHandler;
    @Nullable
    private Runnable finishRunnable;
    @Nullable
    private Runnable fragmentContainerRunnable;

    @Override
    public void onCreate(@Nullable final Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);

        setTheme(ThemeResolver.resolveTheme(InstabugCore.getTheme()));
        StatusBarUtils.setStatusBarForDialog(this);

        fragmentContainer = findViewById(R.id.instabug_fragment_container);
        RelativeLayout activityContainer = findViewById(R.id.survey_activity_container);
        activityContainer.setFocusableInTouchMode(true);

        presenter = new AnnouncementPresenter(this);
        presenter.setLayoutHeightSecondary(false);
        // this fixes https://instabug.atlassian.net/browse/IBG-3232
        // Prevent showing announcement if the app is closed
        fragmentContainerRunnable = new Runnable() {
            @Override
            public void run() {
                if (InstabugCore.getStartedActivitiesCount() <= 1) {
                    InstabugSDKLogger.d(Constants.LOG_TAG, "Announcement Error: StartedActivitiesCount <= 1");
                    finish();
                    return;
                }
                try {
                    if (!isFinishing()) {
                        if (isResumed) {
                            Announcement announcement = (Announcement) getIntent().getSerializableExtra(KEY_ANNOUNCEMENT);
                            AnnouncementActivity.this.announcement = announcement;
                            if (savedInstanceState == null && announcement != null) {
                                AnnouncementNavigator.navigateToAnnouncement(getSupportFragmentManager(),
                                        announcement);
                            }
                        } else {
                            finish();
                        }
                    }
                } catch (Exception e) {
                    InstabugSDKLogger.e(Constants.LOG_TAG,
                            "Announcement has not been shown due to this error: "
                                    + e.getMessage());
                }
            }
        };
        fragmentContainer.postDelayed(fragmentContainerRunnable, 500);
    }

    @Override
    protected void onResume() {
        super.onResume();
        isResumed = true;
        SurveyPlugin plugin = (SurveyPlugin) InstabugCore.getXPlugin(SurveyPlugin.class);
        if (plugin != null) {
            plugin.setState(Plugin.STATE_FOREGROUND);
        }
    }

    @Override
    protected void onPause() {
        isResumed = false;
        super.onPause();
        overridePendingTransition(0, 0);
    }

    @Override
    protected void onDestroy() {
        super.onDestroy();
        SurveyPlugin plugin = (SurveyPlugin) InstabugCore.getXPlugin(SurveyPlugin.class);
        if (plugin != null) {
            plugin.setState(Plugin.STATE_BACKGROUND);
        }
        if (finishRunnable != null && finishHandler != null) {
            finishHandler.removeCallbacks(finishRunnable);
            finishHandler = null;
            finishRunnable = null;
        }
        if (fragmentContainer != null && fragmentContainerRunnable != null) {
            fragmentContainer.removeCallbacks(fragmentContainerRunnable);
            fragmentContainerRunnable = null;
            fragmentContainer.clearAnimation();
        }
        Fragment fragmentById = getSupportFragmentManager().findFragmentById(R.id.instabug_fragment_container);
        if (fragmentById instanceof UpdateMessageFragment) {
            ((UpdateMessageFragment) fragmentById).onDestroy();
        }
        if (SurveysManager.getInstance() != null) {
            SurveysManager.getInstance().registerSurveysTriggerEvents();
        }
        if (presenter != null) {
            presenter.onDestroy();
        }
    }

    @Override
    protected int getLayout() {
        return R.layout.instabug_survey_activity;
    }

    @Override
    protected void initViews() {
        //We don't need it
    }

    @Override
    public void setFrameLayoutHeightTo(int height) {
        RelativeLayout.LayoutParams params = (RelativeLayout.LayoutParams)
                fragmentContainer.getLayoutParams();
        params.height = height;
        fragmentContainer.setLayoutParams(params);
    }

    @Override
    public void finishAnnouncement(boolean showThanksFragment) {
        runOnUiThread(new Runnable() {
            @Override
            public void run() {
                if (getSupportFragmentManager() == null) return;

                Fragment fragment = getSupportFragmentManager().findFragmentById(R.id
                        .instabug_fragment_container);

                if (fragment != null && isResumed) {
                    getSupportFragmentManager().beginTransaction()
                            .setCustomAnimations(0, R.anim.instabug_anim_flyout_to_bottom)
                            .remove(fragment).commit();
                }
                finishHandler = new Handler();
                finishRunnable = new Runnable() {
                    @Override
                    public void run() {
                        SurveysUtils.executeRunnableAfterShowingSurvey();
                        finish();
                    }
                };
                finishHandler.postDelayed(finishRunnable, 300);
            }
        });
    }

    @Override
    public void setFrameLayoutHeightWithAnimation(int heightInPx) {
        ValueAnimator anim = ValueAnimator.ofInt(fragmentContainer.getMeasuredHeight(), heightInPx);
        anim.addUpdateListener(new ValueAnimator.AnimatorUpdateListener() {
            @Override
            public void onAnimationUpdate(ValueAnimator valueAnimator) {
                int val = (Integer) valueAnimator.getAnimatedValue();
                ViewGroup.LayoutParams layoutParams = fragmentContainer.getLayoutParams();
                layoutParams.height = val;
                fragmentContainer.setLayoutParams(layoutParams);
            }
        });
        anim.setDuration(300);
        anim.start();
    }

    public void setLayoutHeightSecondary(boolean withAnimation) {
        if (presenter != null) {
            presenter.setLayoutHeightSecondary(withAnimation);
        }

    }

    public void setBackgroundTransparent(boolean shouldBeTransparent) {
        getWindow().getDecorView().setBackgroundColor(ContextCompat.getColor(this,
                shouldBeTransparent ? R.color.instabug_transparent_color
                        : R.color.instabug_dialog_bg_color));
    }

    @Override
    public void submitAnnouncement(Announcement announcement) {
        if (presenter != null) {
            presenter.submitAnnouncement(announcement);
        }
    }

    @Override
    public void dismissAnnouncement(Announcement announcement) {
        if (presenter != null) {
            presenter.dismissAnnouncement(announcement);
        }
    }

    @Override
    public void onBackPressed() {
        boolean topFragmentHandledBackPress = delegateBackPressToTopFragment();
        if (!topFragmentHandledBackPress) {
            super.onBackPressed();
        }
    }

    @Nullable
    public Announcement getAnnouncement() {
        return announcement;
    }

    @Nullable
    private Fragment getTopFragment() {
        return getSupportFragmentManager().findFragmentById(R.id.instabug_fragment_container);
    }

    private boolean delegateBackPressToTopFragment() {
        Fragment fragment = getTopFragment();
        if (fragment instanceof BackPressHandler) {
            BackPressHandler backPressHandler = (BackPressHandler) fragment;
            return backPressHandler.onBackPress();
        }
        return false;
    }


}

