package com.instabug.survey.announcements.models;

import androidx.annotation.IntDef;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.instabug.library.internal.storage.cache.Cacheable;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.TimeUtils;
import com.instabug.survey.Constants;
import com.instabug.survey.common.models.ActionEvent;
import com.instabug.survey.common.models.BaseLocalization;
import com.instabug.survey.common.models.IUserInteraction;
import com.instabug.survey.common.models.SyncingStatus;
import com.instabug.survey.common.models.Target;
import com.instabug.survey.common.models.UserInteraction;
import com.instabug.survey.models.UserInteractionOnTypes;
import com.instabug.survey.utils.DateUtils;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.TimeUnit;


import static com.instabug.survey.announcements.models.AnnouncementType.UPDATE_MSG;
import static com.instabug.survey.announcements.models.AnnouncementType.WHAT_IS_NEW;

/**
 * Created by Barakat on 23/12/2018
 */
public class Announcement implements Cacheable, Serializable, IUserInteraction {

    // keys used by toJson() & fromJson() methods
    private static final String KEY_PUBLISHED = "published";
    private static final String KEY_PAUSED = "paused";
    private static final String KEY_ID = "id";
    private static final String KEY_TITLE = "title";
    private static final String KEY_TOKEN = "token";
    private static final String KEY_TARGET = "target";
    private static final String KEY_ANSWERED = "answered";
    private static final String KEY_DISMISSED_AT = "dismissed_at";
    private static final String KEY_IS_CANCELLED = "is_cancelled";
    private static final String KEY_ANNOUNCE_EVENTS = "events";
    private static final String KEY_ANNOUNCE_STATE = "announcement_state";
    private static final String KEY_SHOULD_SHOW_AGAIN = "should_show_again";
    private static final String KEY_SESSION_COUNTER = "session_counter";
    private static final String KEY_TYPE = "type";
    private static final String KEY_ANNOUNCEMENTS = "announcement_items";

    private long id;
    @Nullable
    private String title;
    private int type;
    @Nullable
    private ArrayList<AnnouncementItem> announcementItems;
    private boolean paused = false;
    private @AnnouncementAssetsStatus
    int assetsStatus;
    private BaseLocalization localization;
    private UserInteraction userInteraction;


    public Announcement() {
        assetsStatus = AnnouncementAssetsStatus.NOT_AVAILABLE;
        localization = new BaseLocalization();
        userInteraction = new UserInteraction(UserInteractionOnTypes.ANNOUNCEMENT);
    }

    public static List<Announcement> getPausedAnnouncementsFromJson(JSONObject response) throws JSONException {
        JSONArray jsonArray = response.getJSONArray(KEY_PAUSED);
        List<Announcement> pausedAnnouncements = new ArrayList<>(jsonArray.length());
        for (int i = 0; i < jsonArray.length(); i++) {
            Announcement announcement = new Announcement();
            announcement.setId(jsonArray.getLong(i));
            announcement.setPaused(true);
            pausedAnnouncements.add(announcement);
        }
        return pausedAnnouncements;
    }

    public static List<Announcement> fromJson(JSONObject announcementsJsonObject) throws JSONException {
        JSONArray jsonArray = announcementsJsonObject.getJSONArray(KEY_PUBLISHED);
        List<Announcement> announcements = new ArrayList<>();
        for (int i = 0; i < jsonArray.length(); i++) {
            JSONObject announcementJsonObject = jsonArray.getJSONObject(i);
            Announcement announcement = new Announcement();
            announcement.fromJson(announcementJsonObject.toString());
            announcements.add(announcement);
        }
        return announcements;
    }

    public Target getTarget() {
        return userInteraction.getTarget();
    }

    public void setTarget(Target target) {
        userInteraction.setTarget(target);
    }

    public boolean isPaused() {
        return paused;
    }

    public void setPaused(boolean paused) {
        this.paused = paused;
    }

    @AnnouncementType
    public int getType() {
        return type;
    }

    public void setType(@AnnouncementType int type) {
        this.type = type;
    }

    @Nullable
    public String getTitle() {
        return title;
    }

    public void setTitle(@Nullable String title) {
        this.title = title;
    }

    public void setSubmitted() {
        setCancelled(false);
        setAnswered(true);
        setAlreadyShown(true);


        ActionEvent submitEvent;
        //Add rate Event only if the user clicked on the rateUsOnPlayStore button
        submitEvent = new ActionEvent(ActionEvent.EventType.SUBMIT,
                TimeUtils.currentTimeSeconds(), 1);

        setAnnouncementState(SyncingStatus.READY_TO_SEND);

        //Ignore adding submit event if the last added event was also submit
        Target target = userInteraction.getTarget();
        if (target.getActionEvents().size() > 0
                && target.getActionEvents().get(target.getActionEvents().size() - 1).getEventType()
                == ActionEvent.EventType.SUBMIT
                && submitEvent.getEventType() == ActionEvent.EventType.SUBMIT) {
            return;
        }

        target.getActionEvents().add(submitEvent);
    }

    @Nullable
    public ArrayList<AnnouncementItem> getAnnouncementItems() {
        return announcementItems;
    }

    public void setAnnouncementItems(@Nullable ArrayList<AnnouncementItem> announcementItems) {
        this.announcementItems = announcementItems;
    }

    public ArrayList<ActionEvent> getAnnouncementEvents() {
        return userInteraction.getTarget().getActionEvents();
    }

    public int getAssetsStatus() {
        return assetsStatus;
    }

    public void setAssetsStatus(int assetsStatus) {
        this.assetsStatus = assetsStatus;
    }

    public String getConditionsOperator() {
        return userInteraction.getTarget().getConditionsOperator();
    }

    public void setConditionsOperator(String conditionOperator) {
        this.userInteraction.getTarget().setConditionsOperator(conditionOperator);
    }

    public int getEventIndex() {
        return userInteraction.getEventIndex();
    }

    public void setEventIndex(int eventIndex) {
        userInteraction.setEventIndex(eventIndex);
    }

    public void setDismissed() {
        setAnnouncementState(SyncingStatus.READY_TO_SEND);

        this.userInteraction.setDismissedAt(TimeUtils.currentTimeSeconds());
        setAnswered(true);
        setCancelled(true);
        setAlreadyShown(true);
        //Ignore adding dismissEvent if the last added event was also dismissEvent
        Target target = userInteraction.getTarget();
        if (target.getActionEvents().size() > 0
                && target.getActionEvents().get(target.getActionEvents().size() - 1).getEventType()
                == ActionEvent.EventType.DISMISS) {
            return;
        }
        ActionEvent dismissEvent = new ActionEvent(ActionEvent.EventType.DISMISS,
                userInteraction.getDismissedAt(), getEventIndex());
        target.getActionEvents().add(dismissEvent);
    }

    public void addShowEvent() {
        setShowAt(TimeUtils.currentTimeSeconds());
        ActionEvent showEvent = new ActionEvent(ActionEvent.EventType.SHOW,
                TimeUtils.currentTimeSeconds(), incrementEventIndex());
        userInteraction.getTarget().getActionEvents().add(showEvent);
    }

    public boolean isCancelled() {
        return userInteraction.isCancelled();
    }

    public void setCancelled(boolean cancelled) {
        userInteraction.setCancelled(cancelled);
    }

    private int incrementEventIndex() {
        return userInteraction.incrementEventIndex();
    }


    public long getRespondedAt() {
        Target target = userInteraction.getTarget();
        if (target.getActionEvents() != null && target.getActionEvents().size() > 0) {
            for (ActionEvent announcementEvent : target.getActionEvents()) {
                if (announcementEvent.getEventType() == ActionEvent.EventType.SUBMIT
                        || announcementEvent.getEventType() == ActionEvent.EventType.DISMISS) {
                    return announcementEvent.getTimestamp();
                }
            }
        }
        return 0;
    }

    @Override
    public String toJson() throws JSONException {
        JSONObject announcementJsonObject = new JSONObject();
        announcementJsonObject.put(KEY_ID, id)
                .put(KEY_TYPE, type)
                .put(KEY_TITLE, title)
                .put(KEY_ANNOUNCEMENTS, AnnouncementItem.toJson(announcementItems))
                .put(KEY_TARGET, Target.toJson(userInteraction.getTarget()))
                .put(KEY_ANNOUNCE_EVENTS, ActionEvent.toJson(userInteraction.getTarget().getActionEvents()))
                .put(KEY_ANSWERED, userInteraction.isAnswered())
                .put(KEY_DISMISSED_AT, getDismissedAt())
                .put(KEY_IS_CANCELLED, userInteraction.isCancelled())
                .put(KEY_ANNOUNCE_STATE, getAnnouncementState().toString())
                .put(KEY_SHOULD_SHOW_AGAIN, shouldShow())
                .put(KEY_SESSION_COUNTER, getSessionCounter());
        localization.localizationToJson(announcementJsonObject);
        return announcementJsonObject.toString();
    }


    public boolean shouldShow() {
        Target target = userInteraction.getTarget();
        boolean shouldShowEveryTime = target.getFrequency().shouldShowEveryTime();
        boolean notShownBefore = !userInteraction.isAlreadyShown();
        boolean isFrequent = !target.getFrequency().shouldShowOnce();
        boolean isFrequencyPeriodPassed = DateUtils.getDifferenceInDaysFromSeconds(getShownAt()) >= target.getFrequency().getShowingInterval();
        return shouldShowEveryTime || notShownBefore || (isFrequent && isFrequencyPeriodPassed);
    }

    @SuppressWarnings("javadoc")
    /**
     If the announcement has been already dismissed but has no value in {@param shownAt}
     it sets the shownAt as the same as dismissedAt
     This happens if the current SDK version is above {@link com.instabug.library.internal.storage.cache.db.InstabugDBVersions#DATABASE_VERSION_16}
     and the announcement was dissmissed on an SDK running with DB schema version equla or below {@link com.instabug.library.internal.storage.cache.db.InstabugDBVersions#DATABASE_VERSION_16}

     @return shownAt time stamp in seconds
     */
    public long getShownAt() {
        if (userInteraction.getShownAt() == 0 && userInteraction.getDismissedAt() != 0) {
            setShowAt(userInteraction.getDismissedAt());
        }
        return userInteraction.getShownAt();
    }

    public void setShowAt(long shownAt) {
        this.userInteraction.setShownAt(shownAt);
    }

    private int getDaysSinceShow() {
        return (int) TimeUnit.SECONDS.toDays(TimeUtils.currentTimeSeconds() - getShownAt());
    }

    public boolean isAlreadyShown() {
        return userInteraction.isAlreadyShown();
    }

    public void setAlreadyShown(boolean alreadyShown) {
        userInteraction.setAlreadyShown(alreadyShown);
    }

    public long getDismissedAt() {
        return userInteraction.getDismissedAt();
    }

    public void setDismissedAt(long dismissedAt) {
        userInteraction.setDismissedAt(dismissedAt);
    }

    public SyncingStatus getAnnouncementState() {
        return userInteraction.getSurveyState();
    }

    public void setAnnouncementState(SyncingStatus announcementState) {
        userInteraction.setSurveyState(announcementState);
    }

    public int getSessionCounter() {
        return userInteraction.getSessionCounter();
    }

    private void setSessionCounter(int sessionCounter) {
        userInteraction.setSessionCounter(sessionCounter);
    }

    @Override
    public void fromJson(String modelAsJson) throws JSONException {
        JSONObject announcementJsonObject = new JSONObject(modelAsJson);
        if (announcementJsonObject.has(KEY_ID)) {
            setId(announcementJsonObject.getLong(KEY_ID));
        }
        if (announcementJsonObject.has(KEY_TYPE)) {
            @AnnouncementType int type = announcementJsonObject.getInt(KEY_TYPE);
            setType(type);
        }
        if (announcementJsonObject.has(KEY_TITLE)) {
            setTitle(announcementJsonObject.getString(KEY_TITLE));
        }
        if (announcementJsonObject.has(KEY_ANNOUNCE_EVENTS)) {
            userInteraction.getTarget().setActionEvents(ActionEvent.fromJson(announcementJsonObject.getJSONArray(KEY_ANNOUNCE_EVENTS)));
        }
        if (announcementJsonObject.has(KEY_ANNOUNCEMENTS)) {
            setAnnouncementItems(AnnouncementItem.fromJson(announcementJsonObject.getJSONArray(KEY_ANNOUNCEMENTS)));
        } else {
            setAnnouncementItems(new ArrayList<AnnouncementItem>());
        }
        if (announcementJsonObject.has(KEY_TARGET)) {
            JSONObject targetJsonObject = announcementJsonObject.getJSONObject(KEY_TARGET);
            userInteraction.getTarget().fromJson(targetJsonObject.toString().replace("\\", ""));
        }
        if (announcementJsonObject.has(KEY_ANSWERED)) {
            setAnswered(announcementJsonObject.getBoolean(KEY_ANSWERED));
        }
        if (announcementJsonObject.has(KEY_IS_CANCELLED)) {
            setCancelled(announcementJsonObject.getBoolean(KEY_IS_CANCELLED));
        }
        if (announcementJsonObject.has(KEY_ANNOUNCE_STATE)) {
            setAnnouncementState(SyncingStatus.valueOf(announcementJsonObject.getString(KEY_ANNOUNCE_STATE)));
        }
        if (announcementJsonObject.has(KEY_SESSION_COUNTER)) {
            setSessionCounter(announcementJsonObject.getInt(KEY_SESSION_COUNTER));
        }
        if (announcementJsonObject.has(KEY_DISMISSED_AT)) {
            setDismissedAt(announcementJsonObject.getInt(KEY_DISMISSED_AT));
        }
        localization.localizationFromJson(announcementJsonObject);
    }

    public long getId() {
        return id;
    }

    public Announcement setId(long id) {
        this.id = id;
        return this;
    }

    public boolean isAnswered() {
        return userInteraction.isAnswered();
    }

    public void setAnswered(boolean answered) {
        userInteraction.setAnswered(answered);
    }

    public BaseLocalization getLocalization() {
        return localization;
    }

    public void setLocalization(BaseLocalization localization) {
        this.localization = localization;
    }

    @Override
    public long getSurveyId() {
        return id;
    }

    @Override
    public UserInteraction getUserInteraction() {
        return userInteraction;
    }

    @Override
    public int hashCode() {
        return String.valueOf(getId()).hashCode();
    }

    @Override
    public boolean equals(@Nullable Object announcement) {
        if (announcement == null) return false;

        if (announcement instanceof Announcement) {
            Announcement comparedAnnouncement = (Announcement) announcement;
            return comparedAnnouncement.getId() == getId();
        } else {
            return false;
        }
    }

    @NonNull
    @Override
    public String toString() {
        try {
            return toJson();
        } catch (JSONException e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Error: " + e.getMessage() + " while parsing announcement", e);
        }
        return super.toString();
    }

    /**
     * This method to rest user interaction but since it holds target that has some data other than user's action
     * so we need to hold it and only clear user's actions.
     */
    public void resetUserInteractions() {
        Target target = userInteraction.getTarget();
        target.setActionEvents(new ArrayList<ActionEvent>());
        userInteraction = new UserInteraction(UserInteractionOnTypes.SURVEY);
        userInteraction.setTarget(target);
    }

    public void setUserInteraction(UserInteraction userInteraction) {
        this.userInteraction = userInteraction;
    }

    public String getTypeAsString() {
        switch (type) {
            case WHAT_IS_NEW:
                return "WhatsNew";
            case UPDATE_MSG:
                return "UpdateMessage";
            default:
                return "";
        }
    }


    @IntDef({
            AnnouncementAssetsStatus.NOT_AVAILABLE,
            AnnouncementAssetsStatus.DOWNLOAD_SUCCEED,
            AnnouncementAssetsStatus.DOWNLOAD_FAILED
    })
    public @interface AnnouncementAssetsStatus {
        int NOT_AVAILABLE = 0;
        int DOWNLOAD_SUCCEED = 1;
        int DOWNLOAD_FAILED = 2;
    }
}