package com.instabug.survey.announcements.cache;

import androidx.annotation.Nullable;

import com.instabug.library.Instabug;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.internal.storage.cache.AssetsCacheManager;
import com.instabug.library.model.AssetEntity;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.threading.PoolProvider;
import com.instabug.survey.Constants;
import com.instabug.survey.announcements.models.Announcement;
import com.instabug.survey.announcements.models.AnnouncementItem;
import com.instabug.survey.announcements.models.NewFeature;

import java.util.ArrayList;
import java.util.List;

import io.reactivexport.Observable;
import io.reactivexport.ObservableEmitter;
import io.reactivexport.ObservableOnSubscribe;
import io.reactivexport.observers.DisposableObserver;

/**
 * Created by Barakat on 02/01/2019
 */
public class NewFeaturesAssetsHelper {
    private static final String EMPTY_ICON_PATH = "";

    public static void downloadAssets(final Announcement announcement) {
        InstabugSDKLogger.d(Constants.LOG_TAG, "downloading announcement assets for: " + announcement.getId());
        List<Observable<AssetEntity>> downloadingObservables = null;
        if (announcement.getAnnouncementItems() != null) {
            downloadingObservables = getDownloadingObservables(announcement.getAnnouncementItems().get(0));
        }
        if (downloadingObservables == null) return;

        Observable.merge(downloadingObservables).subscribe(new DisposableObserver<AssetEntity>() {

            @Override
            public void onNext(AssetEntity assetEntity) {
                InstabugSDKLogger.d(Constants.LOG_TAG, "downloading announcement " + announcement.getId() + " asset started");
            }

            @Override
            public void onError(Throwable e) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "downloading announcement " + announcement.getId() + " assets failed");
                PoolProvider.postIOTask(new Runnable() {
                    @Override
                    public void run() {
                        AnnouncementCacheManager.updateAssetStatus(announcement.getId(), Announcement.AnnouncementAssetsStatus.DOWNLOAD_FAILED);
                    }
                });
            }

            @Override
            public void onComplete() {
                InstabugSDKLogger.d(Constants.LOG_TAG, "downloading announcement " + announcement.getId() + " assets completed");
                announcement.setAssetsStatus(Announcement.AnnouncementAssetsStatus.DOWNLOAD_SUCCEED);
                PoolProvider.postIOTask(new Runnable() {
                    @Override
                    public void run() {
                        AnnouncementCacheManager.updateAssetStatus(announcement.getId(), Announcement.AnnouncementAssetsStatus.DOWNLOAD_SUCCEED);
                    }
                });
            }
        });
    }

    @Nullable
    private static List<Observable<AssetEntity>> getDownloadingObservables(AnnouncementItem announcementItem) {
        if (announcementItem.getNewFeatures() == null) return null;
        List<Observable<AssetEntity>> observables = new ArrayList<>(announcementItem.getNewFeatures().size());
        for (int i = 0; i < announcementItem.getNewFeatures().size(); i++) {
            NewFeature feature = announcementItem.getNewFeatures().get(i);
            if (feature.getIconUrl() != null && !feature.getIconUrl().equals(EMPTY_ICON_PATH)) {
                observables.add(getFeatureObservable(announcementItem.getId(), feature));
            }
        }
        return observables;
    }

    private static Observable<AssetEntity> getFeatureObservable(final long announceId, final NewFeature feature) {
        return Observable.create(new ObservableOnSubscribe<AssetEntity>() {
            @Override
            public void subscribe(final ObservableEmitter<AssetEntity> emitter) {
                if (Instabug.getApplicationContext() == null) return;
                if (feature.getIconUrl() != null) {
                    AssetEntity assetEntity = AssetsCacheManager.createEmptyEntity(
                            Instabug.getApplicationContext(), feature.getIconUrl(), AssetEntity.AssetType.IMAGE);
                    AssetsCacheManager.getAssetEntity(assetEntity,
                            new AssetsCacheManager.OnDownloadFinished() {
                                @Override
                                public void onSuccess(final AssetEntity assetEntity) {
                                    InstabugCore.doOnBackground(new Runnable() {
                                        @Override
                                        public void run() {
                                            AnnouncementCacheManager.insertAnnouncementAsset(
                                                    announceId,
                                                    feature.getId(),
                                                    assetEntity.getFile().getPath());
                                            emitter.onNext(assetEntity);
                                            emitter.onComplete();
                                        }
                                    });
                                }

                                @Override
                                public void onFailed(Throwable error) {
                                    if (!emitter.isDisposed()) {
                                        emitter.onError(error);
                                    } else {
                                        InstabugSDKLogger.e(Constants.LOG_TAG, "Assets Request got error: " + error);
                                    }
                                }
                            });
                }
            }
        });
    }
}
