package com.instabug.survey.common.userInteractions;

import com.instabug.survey.common.models.IUserInteraction;
import com.instabug.survey.common.models.UserInteraction;
import com.instabug.survey.models.UserInteractionOnTypes;

import java.util.ArrayList;
import java.util.List;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

/**
 * A class Repository that works as a middleware to preform CRUD operations
 */
public class UserInteractionCacheManager {

    /**
     * This method to add user interaction, first it setup the user interaction object then delegate
     * it to {@link UserInteractionDbHelper}
     *
     * @param iUerInteraction is a survey or announcement which will get from it the user interaction.
     * @param uuid            user id
     */
    public static <T extends IUserInteraction> void insertUserInteraction(@NonNull T iUerInteraction,
                                                                          @NonNull String uuid) {
        UserInteraction userInteraction = iUerInteraction.getUserInteraction();
        userInteraction.setSurveyId(iUerInteraction.getSurveyId());
        userInteraction.setUuid(uuid);
        UserInteractionDbHelper.insert(userInteraction);
    }

    /**
     * This method to insert bulk of  user interactions, first it setup the user interaction object then delegate
     * it to {@link UserInteractionDbHelper}
     *
     * @param iUerInteractions are list of  surveys or announcements which will get from their the user interactions.
     * @param uuid             user id
     */
    public static <T extends IUserInteraction> void insertUserInteractions(@NonNull List<T> iUerInteractions,
                                                                           @NonNull String uuid) {
        List<UserInteraction> userInteractions = new ArrayList<>();
        for (T iUerInteraction : iUerInteractions) {
            UserInteraction userInteraction = iUerInteraction.getUserInteraction();
            userInteraction.setSurveyId(iUerInteraction.getSurveyId());
            userInteraction.setUuid(uuid);
            userInteractions.add(userInteraction);
        }

        UserInteractionDbHelper.insertBulk(userInteractions);
    }

    /**
     * This method to update user interaction, first it setup the user interaction object then delegate
     * it to {@link UserInteractionDbHelper}
     *
     * @param iUerInteraction is a survey or announcement which will get from it the user interaction.
     * @param uuid            user id
     */
    public static <T extends IUserInteraction> void updateUserInteraction(@NonNull T iUerInteraction,
                                                                          @NonNull String uuid) {
        UserInteraction userInteraction = iUerInteraction.getUserInteraction();
        userInteraction.setSurveyId(iUerInteraction.getSurveyId());
        userInteraction.setUuid(uuid);
        UserInteractionDbHelper.update(userInteraction);
    }


    /**
     * This method to retrieve user interaction by its unique keys, surveyId, uuid and interaction on type.
     *
     * @param surveyId
     * @param uuid
     * @param interactionOnType
     * @return user interaction if found otherwise null
     */
    @Nullable
    public static UserInteraction retrieveUserInteraction(long surveyId,
                                                          @NonNull String uuid,
                                                          @UserInteractionOnTypes int interactionOnType) {

        return UserInteractionDbHelper.retrieveBySurveyIdAndUUID(surveyId, uuid, interactionOnType);
    }


    public static void deleteUserInteraction(long surveyId,
                                             @NonNull String uuid,
                                             @UserInteractionOnTypes int interactionOnType) {
        UserInteractionDbHelper.delete(surveyId, uuid, interactionOnType);
    }


    public static void deleteBulkOfUserInteractions(List<UserInteraction> userInteractions) {
        UserInteractionDbHelper.deleteBulk(userInteractions);


    }

}
