package com.instabug.survey.announcements.network;

import static com.instabug.library.IBGNetworkWorker.ANNOUNCEMENTS;

import android.content.Context;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.instabug.library.Instabug;
import com.instabug.library.InstabugNetworkJob;
import com.instabug.library.networkv2.request.Request;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.survey.Constants;
import com.instabug.survey.announcements.cache.AnnouncementCacheManager;
import com.instabug.survey.announcements.models.Announcement;
import com.instabug.survey.common.models.SyncingStatus;
import com.instabug.survey.di.ServiceLocator;

import org.json.JSONException;

import java.util.List;

public class InstabugAnnouncementSubmitterJob extends InstabugNetworkJob {

    @Nullable
    private static InstabugAnnouncementSubmitterJob INSTANCE;

    private InstabugAnnouncementSubmitterJob() {
    }

    public synchronized static InstabugAnnouncementSubmitterJob getInstance() {
        if (INSTANCE == null) {
            INSTANCE = new InstabugAnnouncementSubmitterJob();
        }

        return INSTANCE;
    }


    private static void submitAnnouncement(@NonNull Context context) throws JSONException {
        InstabugSDKLogger.d(Constants.LOG_TAG, "submitAnnouncements started");
        List<Announcement> readyAnnouncements = AnnouncementCacheManager.getReadyToBeSend();
        InstabugSDKLogger.d(Constants.LOG_TAG, "ready to send Announcements size: " + readyAnnouncements.size());
        if (ServiceLocator.getConfigurationsProvider().isSurveysUsageExceeded()) {
            markAnnouncementsAsSynced(readyAnnouncements);
            return;
        }
        for (final Announcement announcement : readyAnnouncements) {
            AnnouncementsService.getInstance().submittingAnnouncement(context, announcement, new Request.Callbacks<Boolean, Throwable>() {
                @Override
                public void onSucceeded(@Nullable Boolean response) {
                    announcement.setAnnouncementState(SyncingStatus.SYNCED);
                    announcement.getAnnouncementEvents().clear();
                    AnnouncementCacheManager.updateAnnouncement(announcement);
                }

                @Override
                public void onFailed(Throwable error) {
                    InstabugSDKLogger.e(Constants.LOG_TAG, "Submitting announcement got error: " + error.getMessage());
                }
            });
        }
    }

    private static void markAnnouncementsAsSynced(@NonNull List<Announcement> announcements) {
        for (final Announcement announcement : announcements) {
            announcement.setAnnouncementState(SyncingStatus.SYNCED);
            announcement.getAnnouncementEvents().clear();
        }
        AnnouncementCacheManager.updateBulk(announcements);
    }

    public void start() {
        enqueueJob(ANNOUNCEMENTS, new Runnable() {
            @Override
            public void run() {
                if (Instabug.getApplicationContext() != null) {
                    try {
                        submitAnnouncement(Instabug.getApplicationContext());
                    } catch (Exception e) {
                        InstabugSDKLogger.e(Constants.LOG_TAG, "Error " + e.getMessage() + " occurred while submitting announcements", e);
                    }
                } else {
                    InstabugSDKLogger.d(Constants.LOG_TAG, "Context was null while trying to submit announcements");
                }
            }

        });
    }
}

