package com.instabug.survey;

import android.content.Context;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;

import com.instabug.library.Feature;
import com.instabug.library.IBGFeature;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.core.eventbus.coreeventbus.IBGCoreEventSubscriber;
import com.instabug.library.core.eventbus.coreeventbus.IBGSdkCoreEvent;
import com.instabug.library.core.eventbus.coreeventbus.IBGSdkCoreEvent.CacheDumped;
import com.instabug.library.core.eventbus.coreeventbus.IBGSdkCoreEvent.FeaturesFetched;
import com.instabug.library.core.eventbus.coreeventbus.IBGSdkCoreEvent.NetworkActivated;
import com.instabug.library.core.eventbus.coreeventbus.IBGSdkCoreEvent.User;
import com.instabug.library.core.eventbus.eventpublisher.IBGCompositeDisposable;
import com.instabug.library.core.eventbus.eventpublisher.IBGDisposable;
import com.instabug.library.core.plugin.Plugin;
import com.instabug.library.diagnostics.IBGDiagnostics;
import com.instabug.library.internal.storage.cache.OnDiskCache;
import com.instabug.library.tokenmapping.MappedTokenChangedEventBus;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.LocaleUtils;
import com.instabug.library.util.threading.PoolProvider;
import com.instabug.survey.announcements.AnnouncementManager;
import com.instabug.survey.announcements.cache.AnnouncementCacheManager;
import com.instabug.survey.announcements.cache.AnnouncementsDBHelper;
import com.instabug.survey.announcements.models.Announcement;
import com.instabug.survey.announcements.network.InstabugAnnouncementSubmitterJob;
import com.instabug.survey.announcements.settings.AnnouncementsSettings;
import com.instabug.survey.cache.SurveysCacheManager;
import com.instabug.survey.configuration.SurveysConfigurationsProvider;
import com.instabug.survey.di.ServiceLocator;
import com.instabug.survey.models.CountryInfo;
import com.instabug.survey.models.Survey;
import com.instabug.survey.network.service.InstabugSurveysSubmitterJob;
import com.instabug.survey.settings.PerSessionSettings;
import com.instabug.survey.settings.PersistableSettings;
import com.instabug.survey.settings.SurveysSettings;
import com.instabug.survey.utils.PlayStoreUtils;
import com.instabug.survey.utils.SurveysUtils;

import java.util.List;
import java.util.Locale;

/**
 * Created by tarek on 3/25/17.
 */

public class SurveyPlugin extends Plugin {
    @Nullable
    IBGCompositeDisposable disposables;
    @Nullable
    private AnnouncementManager announcementManager;
    private final SurveysConfigurationsProvider configurationsProvider = ServiceLocator.getConfigurationsProvider();

    @VisibleForTesting
    @Nullable
    IBGDisposable mappedTokenChangeDisposable;

    @Override
    public void init(final Context context) {
        super.init(context);
    }

    @Override
    public void stop() {
        PersistableSettings.release();
        PerSessionSettings.release();
        unSubscribeOnSDKEvents();
        unsubscribeFromMappedTokenChangedEvent();
    }

    @Override
    public void start(final Context context) {
        PoolProvider.postIOTaskWithCheck(() -> {
            PersistableSettings.init(context);
            initAnnouncementSettings(context);
            subscribeOnMappedTokenChangedEvent();
            subscribeOnSDKEvents();
        });
    }

    private void initAnnouncementSettings(Context context) {
        announcementManager = AnnouncementManager.getInstance(context);
        com.instabug.survey.announcements.settings.PersistableSettings.init(context);
    }

    @Override
    public void wake() {
        removeOldSurveys();
        SurveysManager.init();
        if (SurveysManager.getInstance() != null)
            SurveysManager.getInstance().updateDismissedSurveysSessionCount();
        checkAppStatus();

    }

    @Override
    public void sleep() {
        if (announcementManager != null) {
            announcementManager.setAppLatestVersion();
        }
        if (SurveysManager.getInstance() != null)
            SurveysManager.getInstance().release();
    }

    @Override
    public long getLastActivityTime() {
        if (PersistableSettings.getInstance() == null) return -1;

        return PersistableSettings.getInstance().getLastSurveyTime();
    }

    @Override
    public void onLocaleChanged(Locale oldLocale, Locale newLocale) {
        super.onLocaleChanged(oldLocale, newLocale);
        SurveysSettings.setLastFetchedAt(0L);
        AnnouncementsSettings.getInstance().setLastFetchedAt(0L);
        if (shouldReFetch() && getAppContext() != null) {
            String resolvedLocale = LocaleUtils.resolveLocale(getAppContext(), newLocale);
            startFetchingAnnouncements(resolvedLocale);
            fetchSurveysImmediately(resolvedLocale);
        }
    }

    @Override
    public boolean isFeatureEnabled() {
        return InstabugCore.isFeatureEnabled(IBGFeature.SURVEYS);
    }

    @VisibleForTesting
    void subscribeOnSDKEvents() {
        getOrCreateDisposables().add(subscribeToSDKCoreEvents());
    }

    @NonNull
    private IBGDisposable subscribeToSDKCoreEvents() {
        return IBGCoreEventSubscriber.subscribe(this::handleCoreEvents);
    }

    private void handleCoreEvents(IBGSdkCoreEvent coreEvent) {
        if (coreEvent instanceof NetworkActivated)
            handleNetworkActivated();
        else if (coreEvent instanceof User)
            handleUserEvent((User) coreEvent);
        else if (coreEvent instanceof CacheDumped)
            handleCacheDumped();
        else if (coreEvent instanceof FeaturesFetched)
            handleFeaturesFetched((FeaturesFetched) coreEvent);
        else if (coreEvent instanceof IBGSdkCoreEvent.Features.Fetched &&
                SurveysUtils.isSurveysFeatureEnabled())
            startFetchingRequests();
        else if (coreEvent instanceof IBGSdkCoreEvent.CodePushVersionChanged)
            handleAppVersionChanged();
    }

    private void handleFeaturesFetched(FeaturesFetched coreEvent) {
        ServiceLocator.getConfigurationHandler()
                .handleConfiguration(coreEvent.getResponse());
    }

    private void handleCacheDumped() {
        if (SurveysUtils.isSurveysFeatureEnabled())
            PoolProvider.postIOTask(AnnouncementsDBHelper::resetAssetsStatus);
    }

    private void handleUserEvent(User coreEvent) {
        if (coreEvent instanceof User.LoggedIn)
            handleUserLogIn();
        else
            handleUserLogOut();
    }

    private void handleUserLogOut() {
        clearUserActivities();
        if (contextWeakReference == null || contextWeakReference.get() == null) return;
        if (SurveysManager.getInstance() != null) {
            SurveysManager.getInstance().notifyLogout();
            AnnouncementManager.getInstance(contextWeakReference.get()).notifyLogout();
        }
    }

    private void handleUserLogIn() {
        if (SurveysManager.getInstance() == null) return;
        SurveysManager.getInstance().notifyUserLoggedIn();

        if (contextWeakReference == null || contextWeakReference.get() == null) return;

        AnnouncementManager.getInstance(contextWeakReference.get()).notifyUserLoggedIn();

    }

    private void handleNetworkActivated() {
        startSubmittingPendingSurveys();
        startSubmittingPendingAnnouncements();
    }

    private void handleAppVersionChanged() {
        Context ctx = null;
        if (contextWeakReference != null)
            ctx = contextWeakReference.get();
        if (ctx != null) {
            AnnouncementManager.getInstance(ctx).notifyAppVersionChanged();
        }
        SurveysManager surveysManager = SurveysManager.getInstance();
        if (surveysManager != null)
            surveysManager.notifyAppVersionChanged();
    }

    private IBGCompositeDisposable getOrCreateDisposables() {
        return disposables != null ? disposables : (disposables = new IBGCompositeDisposable());
    }

    private void subscribeOnMappedTokenChangedEvent() {
        if (mappedTokenChangeDisposable == null) {
            mappedTokenChangeDisposable = MappedTokenChangedEventBus.INSTANCE.subscribe(event -> {
                if (contextWeakReference != null && contextWeakReference.get() != null) {
                    AnnouncementManager manager = AnnouncementManager.getInstance(contextWeakReference.get());
                    if (manager != null) {
                        manager.setHasTokenChanged(true);
                    }
                }
                SurveysManager surveysManager = SurveysManager.getInstance();
                if (surveysManager != null) {
                    surveysManager.setHasTokenChanged(true);
                }
                AnnouncementsSettings.getInstance().setLastFetchedAt(0);
                fetchSurveysImmediately(getLocaleResolved());
                startFetchingAnnouncements(getLocaleResolved());
                resolveCountryInfo(new CountryInfo(), true);
            });
        }
    }


    private void startFetchingRequests() {
        PoolProvider.postIOTask(() -> {
            startFetchingSurveys(getLocaleResolved());
            startFetchingAnnouncements(getLocaleResolved());
            resolveCountryInfo(new CountryInfo(), false);
        });
    }

    private void startSubmittingPendingAnnouncements() {
        if (InstabugCore.getFeatureState(IBGFeature.ANNOUNCEMENTS) == Feature.State.ENABLED &&
                configurationsProvider.isAnnouncementsAvailable()) {

            if (contextWeakReference != null) {
                final Context context = contextWeakReference.get();
                if (context != null) {
                    PoolProvider.postIOTask(() -> {
                                List<Announcement> readyAnnouncements = AnnouncementCacheManager.getReadyToBeSend();
                                if (!readyAnnouncements.isEmpty()) {
                                    InstabugAnnouncementSubmitterJob.getInstance().start();
                                }
                            }
                    );
                } else {
                    InstabugSDKLogger.e(Constants.LOG_TAG, "Couldn't submit announcements due to null context");
                }
            } else {
                InstabugSDKLogger.e(Constants.LOG_TAG, "Couldn't submit announcements due to null context");
            }
        }
    }

    @VisibleForTesting
    void resolveCountryInfo(CountryInfo countryInfo, boolean forceResolve) {
        if (SurveysUtils.isSurveysFeatureAvailable()) {
            if (contextWeakReference != null
                    && contextWeakReference.get() != null
                    && SurveysManager.getInstance() != null) {
                InstabugSDKLogger.d(Constants.LOG_TAG, "Getting Country Code...");
                SurveysManager.getInstance().resolveCountryInfo(countryInfo, forceResolve);
            }
        }
    }

    private void unSubscribeOnSDKEvents() {
        if (disposables != null) disposables.dispose();
    }

    private void unsubscribeFromMappedTokenChangedEvent() {
        if (mappedTokenChangeDisposable != null) {
            mappedTokenChangeDisposable.dispose();
            mappedTokenChangeDisposable = null;
        }
    }

    private void startSubmittingPendingSurveys() {
        if (InstabugCore.getFeatureState(IBGFeature.SURVEYS) == Feature.State.ENABLED &&
                configurationsProvider.isSurveysAvailable()) {
            if (contextWeakReference != null) {
                final Context context = contextWeakReference.get();
                if (context != null) {
                    PoolProvider.postIOTask(() -> {
                                List<Survey> readySurveys = SurveysCacheManager.getReadyToSendSurveys();
                                if (contextWeakReference != null && !readySurveys.isEmpty()) {
                                    InstabugSurveysSubmitterJob.getInstance().start();
                                }
                            }
                    );
                } else {
                    InstabugSDKLogger.e(Constants.LOG_TAG, "Couldn't submit surveys due to null context");
                }
            } else {
                InstabugSDKLogger.e(Constants.LOG_TAG, "Couldn't submit surveys due to null context");
            }
        }
    }

    private static void clearUserActivities() {
        if (PersistableSettings.getInstance() == null) return;

        PersistableSettings.getInstance().setLastSurveyTime(0L);
        PersistableSettings.getInstance().setCountryCodeLastFetch(0L);
    }

    /**
     * This method will remove any surveys the surveys cache file store locally.
     * This was long time ago we now use db, this is only defensive as i noticed we still don't delete
     * the file but only clear its contents.
     */
    private void removeOldSurveys() {
        PoolProvider.postIOTask(new Runnable() {
            @Override
            public void run() {
                if (!SurveysSettings.hasBeenMigrated()) {
                    if (contextWeakReference != null && contextWeakReference.get() != null) {
                        OnDiskCache<Survey> surveysOnDiskCache = new OnDiskCache<>(
                                contextWeakReference.get(),
                                SurveysCacheManager.SURVEYS_DISK_CACHE_KEY,
                                SurveysCacheManager.SURVEYS_DISK_CACHE_FILE_NAME, Survey.class);
                        if (surveysOnDiskCache != null) {
                            surveysOnDiskCache.delete();
                            SurveysSettings.markSurveysFilesMigrated();
                        }
                    }
                }
            }
        });
    }

    private void checkAppStatus() {
        if (contextWeakReference != null) {
            Context contextRef = contextWeakReference.get();
            if (contextRef != null) {
                PlayStoreUtils.isLiveApp(contextRef);
            }
        }
    }


    @VisibleForTesting
    void startFetchingSurveys(String locale) {
        // ignore fetching surveys if the Features are not fetched yet or the Surveys Feature is disabled
        if (InstabugCore.isAppOnForeground() && InstabugCore.isFeaturesFetchedBefore() && SurveysUtils.isSurveysFeatureEnabled() &&
                configurationsProvider.isSurveysAvailableAndUsageNotExceeded()) {

            if (contextWeakReference != null
                    && contextWeakReference.get() != null
                    && SurveysManager.getInstance() != null) {
                SurveysManager.getInstance().startFetching(locale);
            }
        }
    }

    void fetchSurveysImmediately(String locale) {
        // ignore fetching surveys if the Features are not fetched yet or the Surveys Feature is disabled
        if (InstabugCore.isAppOnForeground() && InstabugCore.isFeaturesFetchedBefore() && SurveysUtils.isSurveysFeatureEnabled() &&
                configurationsProvider.isSurveysAvailable() &&
                !configurationsProvider.isSurveysUsageExceeded()) {
            if (contextWeakReference != null
                    && contextWeakReference.get() != null
                    && SurveysManager.getInstance() != null) {
                SurveysManager.getInstance().fetchImmediately(locale);
            }
        }
    }

    @VisibleForTesting
    void startFetchingAnnouncements(String locale) {
        try {
            if (InstabugCore.isAppOnForeground()
                    && contextWeakReference != null && contextWeakReference.get() != null
                    && InstabugCore.getFeatureState(IBGFeature.ANNOUNCEMENTS) == Feature.State.ENABLED &&
                    configurationsProvider.isAnnouncementsAvailable()) {
                AnnouncementManager.getInstance(contextWeakReference.get()).startFetching(locale);
            }
        } catch (Exception ex) {
            IBGDiagnostics.reportNonFatal(ex, "Error while fetching and processing announcements: " + ex.getMessage());
        }
    }

    private String getLocaleResolved() {
        if (getAppContext() == null) return "default";

        return LocaleUtils.getCurrentLocaleResolved(getAppContext());
    }

    @VisibleForTesting
    boolean shouldReFetch() {
        return !getLocaleResolved().equals(SurveysSettings.getLastFetchedLocaleCode());
    }
}
