package com.instabug.featuresrequest.ui.newfeature;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.instabug.featuresrequest.Constants;
import com.instabug.featuresrequest.R;
import com.instabug.featuresrequest.models.FeatureRequest;
import com.instabug.featuresrequest.network.service.AddNewFeatureService;
import com.instabug.featuresrequest.settings.FeaturesRequestSettings;
import com.instabug.library.Instabug;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.core.ui.BasePresenter;
import com.instabug.library.networkv2.request.Request;
import com.instabug.library.user.UserManagerWrapper;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.threading.PoolProvider;

/**
 * @author hossam.
 */

public class AddNewFeaturePresenter extends BasePresenter<AddNewFeatureContract.View>
        implements AddNewFeatureContract.Presenter {

    @Nullable
    private final AddNewFeatureContract.View viewInstance;
    @Nullable
    private volatile String enteredEmail = null;
    private final FeaturesRequestSettings settings;

    public AddNewFeaturePresenter(AddNewFeatureContract.View view) {
        super(view);
        settings = FeaturesRequestSettings.getInstance();
        viewInstance = this.view.get();
        if (settings.isUserIdentificationStateEnabled())
            fillUserDataViews();
        else
            hideUserDataViews();
    }

    private void fillUserDataViews() {
        PoolProvider.postIOTask(() -> {
            final String email = InstabugCore.getEnteredEmail();
            enteredEmail = email;
            final String userName = getEnteredUsername();
            PoolProvider.postMainThreadTask(() -> {
                if (viewInstance != null) {
                    if (email != null)
                        viewInstance.setUserEmail(email);
                    if (userName != null)
                        viewInstance.setUserName(userName);
                }
            });
        });
    }

    private void hideUserDataViews() {
        if (viewInstance != null) {
            viewInstance.hideUserEmail();
            viewInstance.hideUserName();
        }
    }

    @Override
    public void onAddButtonClicked() {
        if (viewInstance == null) return;
        if (viewInstance.getFeatureTitle() == null) return;
        if (settings.isUserIdentificationStateEnabled() && isEmailVerificationRequired()) {
            if (viewInstance.getValidEmail() != null) addFeature();
        } else {
            addFeature();
        }
    }

    private boolean isEmailVerificationRequired() {

        return settings.isNewFeatureEmailFieldRequired()
                || (viewInstance != null && !viewInstance.getEmailString().isEmpty());

    }

    @Override
    @Nullable
    public String getEnteredEmail() {
        return enteredEmail != null ? enteredEmail : InstabugCore.getEnteredEmail();
    }

    @Override
    @Nullable
    public String getEnteredUsername() {
        return InstabugCore.getEnteredUsername();
    }

    @Override
    public void handelRequiredFieldsHints() {
        if (viewInstance != null) {
            viewInstance.setEmailHintStringWithAsterisk(FeaturesRequestSettings.getInstance().isNewFeatureEmailFieldRequired());
        }
    }

    private void addFeature() {
        if (viewInstance != null) {
            // set email if user data is enabled
            if (settings.isUserIdentificationStateEnabled()) {
                InstabugCore.setEnteredEmail(viewInstance.getEmailString());
                InstabugCore.setEnteredUsername(viewInstance.getNameString());
            }

            viewInstance.showProgressbarDialog();
            final FeatureRequest featureRequest = new FeatureRequest(getUserName(), getUserEmail(), InstabugCore.getPushNotificationToken());
            featureRequest.setTitle(viewInstance.getFeatureTitle() != null ? viewInstance.getFeatureTitle() : "");
            featureRequest.setDescription(viewInstance.getFeatureDescription());


            if (Instabug.getApplicationContext() == null) return;

            AddNewFeatureService.getInstance().sendNewFeature(featureRequest, new Request
                    .Callbacks<Boolean, Throwable>() {
                @Override
                public void onSucceeded(@Nullable Boolean response) {
                    InstabugSDKLogger.d(Constants.LOG_TAG, "featureRequest synced successfully");

                    PoolProvider.postMainThreadTask(new Runnable() {
                        @Override
                        public void run() {
                            if (viewInstance == null) return;

                            viewInstance.hideProgressbarDialog();
                            viewInstance.navigateToThanksScreen();
                        }
                    });
                }

                @Override
                public void onFailed(Throwable throwable) {
                    InstabugSDKLogger.e(Constants.LOG_TAG, "Something went wrong " +
                            "while sending featureRequest: " + featureRequest, throwable);
                    notifySyncFailed();
                }

                @Override
                public void onDisconnected() {
                    InstabugSDKLogger.e(Constants.LOG_TAG, "Something went wrong " +
                            "while sending featureRequest: " + featureRequest + " Device is offline");
                    notifySyncFailed();
                }

                private void notifySyncFailed() {
                    PoolProvider.postMainThreadTask(() -> {
                        if (viewInstance == null) return;
                        viewInstance.hideProgressbarDialog();
                        viewInstance.showCustomToast(
                                R.string.feature_request_str_add_comment_error
                        );
                    });
                }
            });


        }
    }

    @NonNull
    private String getUserEmail() {
        return settings.isUserIdentificationStateEnabled() ? UserManagerWrapper.getUserEmail() : "";

    }

    @Nullable
    private String getUserName() {
        return settings.isUserIdentificationStateEnabled() ? UserManagerWrapper.getUserName() : "";

    }

}
