package com.instabug.featuresrequest.ui.base.featureslist;

import android.annotation.SuppressLint;
import android.graphics.PorterDuff;
import android.os.Build;
import android.os.Bundle;
import android.view.View;
import android.view.ViewStub;
import android.widget.AbsListView;
import android.widget.Button;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.ListView;
import android.widget.ProgressBar;
import android.widget.Toast;

import androidx.annotation.CallSuper;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.StringRes;
import androidx.swiperefreshlayout.widget.SwipeRefreshLayout;

import com.instabug.featuresrequest.Constants;
import com.instabug.featuresrequest.R;
import com.instabug.featuresrequest.listeners.OnFeatureClickListener;
import com.instabug.featuresrequest.listeners.OnSortActionChangedListener;
import com.instabug.featuresrequest.models.FeatureRequest;
import com.instabug.featuresrequest.ui.featuredetails.FeatureRequestsDetailsFragment;
import com.instabug.featuresrequest.ui.newfeature.AddNewFeatureFragment;
import com.instabug.featuresrequest.utils.DrawableUtils;
import com.instabug.library.Feature;
import com.instabug.library.IBGFeature;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.core.ui.InstabugBaseFragment;
import com.instabug.library.settings.SettingsManager;
import com.instabug.library.util.InstabugSDKLogger;


/**
 * @author hossam.
 */
@SuppressLint("ERADICATE_FIELD_NOT_INITIALIZED")
public abstract class FeaturesListFragment extends InstabugBaseFragment<FeaturesListPresenter> implements
        FeaturesListContract.View, OnFeatureClickListener, View.OnClickListener,
        OnSortActionChangedListener, OnVoteChangeListener, SwipeRefreshLayout.OnRefreshListener {

    protected static final String KEY_SORT_BY_TOP_VOTED = "sort_by_top_voted";
    protected static final String KEY_MY_POSTS = "my_posts";
    private static final String KEY_EMPTY_STATE = "empty_state";
    private static final String KEY_ERROR_STATE = "error_state";

    @Nullable
    ListView listView;
    FeatureAdapter featureAdapter;
    @Nullable
    private ViewStub emptyStateViewStub;
    @Nullable
    private ViewStub errorStateViewStub;
    protected boolean shouldShowMyPosts = false;
    // list footer
    @Nullable
    private View footerView;
    @Nullable
    private ProgressBar loadMoreProgressBar;
    @Nullable
    private LinearLayout poweredByFooter;
    private boolean footerInflated = false;
    @Nullable
    private SwipeRefreshLayout swipeRefreshLayout;
    private boolean isLoading = false;

    @Override
    public void onCreate(@Nullable Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setRetainInstance(true);
    }

    @Override
    public void showToast(String message) {
        if (message != null && this.getViewContext().getContext() != null) {
            Toast.makeText(this.getViewContext().getContext(), message, Toast.LENGTH_SHORT).show();
        }
    }

    @Override
    public void showToast(@StringRes int resourceId) {
        if (this.getViewContext().getContext() != null) {
            Toast.makeText(this.getViewContext().getContext(), getLocalizedString(resourceId), Toast.LENGTH_SHORT).show();
        }
    }

    @NonNull
    public abstract FeaturesListPresenter getPresenter();

    @Override
    public void showEmptyView() {
        if (emptyStateViewStub != null) {
            if (emptyStateViewStub.getParent() != null) {
                View emptyStateView = emptyStateViewStub.inflate();
                Button addYourIdea = emptyStateView.findViewById(R.id.ib_empty_state_action);
                ImageView icon = emptyStateView.findViewById(R.id.ib_empty_state_icon);
                if (icon != null)
                    icon.setImageResource(R.drawable.ibg_fr_ic_features_empty_state);
                DrawableUtils.setColor(addYourIdea, SettingsManager.getInstance().getPrimaryColor());
                if (addYourIdea != null)
                    addYourIdea.setOnClickListener(this);
            } else {
                emptyStateViewStub.setVisibility(View.VISIBLE);
            }
        }
    }

    @Override
    public void hideErrorView() {
        if (errorStateViewStub != null) {
            errorStateViewStub.setVisibility(View.GONE);
        }
    }

    @Override
    public void showFeatures() {
        if (listView != null) {
            initLoadingMoreView();
            notifyDataSetChanged();
        }
        // Use a local allocation to overcome SpotBugs false-positive possible NPE warnings
        ProgressBar loadMoreProgressBar = this.loadMoreProgressBar;
        if (presenter != null && loadMoreProgressBar != null) {
            if (presenter.hasNext()) {
                loadMoreProgressBar.setVisibility(View.VISIBLE);
            } else {
                unRegisterLoadMoreListener();
                loadMoreProgressBar.setVisibility(View.GONE);
            }
        }
        this.loadMoreProgressBar = loadMoreProgressBar;
        isLoading = false;
    }

    @Override
    public void showErrorView() {
        if (errorStateViewStub != null) {
            if (errorStateViewStub.getParent() != null) {
                View view = errorStateViewStub.inflate();
                view.setOnClickListener(this);
            } else {
                errorStateViewStub.setVisibility(View.VISIBLE);
            }
        }
    }

    @Override
    public void hideEmptyView() {
        if (emptyStateViewStub != null) {
            emptyStateViewStub.setVisibility(View.GONE);
        }
    }

    @Override
    public void notifyDataSetChanged() {
        if (featureAdapter != null) {
            featureAdapter.notifyDataSetChanged();
        }
    }

    @Override
    public void navigateToTheDetailsScreen(FeatureRequest featureRequest) {
        if (getActivity() == null) return;
        getActivity().getSupportFragmentManager()
                .beginTransaction()
                .add(R.id.instabug_fragment_container, FeatureRequestsDetailsFragment.newInstance
                        (featureRequest, this))
                .addToBackStack("feature_requests_details")
                .commit();
    }

    @Override
    protected int getLayout() {
        return R.layout.ib_fr_features_list_fragment;
    }

    @Override
    @CallSuper
    protected void initViews(View rootView, @Nullable Bundle savedInstanceState) {
        emptyStateViewStub = findViewById(R.id.ib_empty_state_stub);
        errorStateViewStub = findViewById(R.id.error_state_stub);
        listView = findViewById(R.id.features_request_list);
        registerLoadMoreListener();
        swipeRefreshLayout = findViewById(R.id.swipeRefreshLayout);
        if (swipeRefreshLayout == null) return;

        swipeRefreshLayout.setColorSchemeColors(SettingsManager.getInstance().getPrimaryColor());
        swipeRefreshLayout.setOnRefreshListener(this);
        if (getArguments() != null) {
            shouldShowMyPosts = getArguments().getBoolean(KEY_MY_POSTS, false);
        }

        // Use a local allocation to overcome SpotBugs false-positive possible NPE warnings
        FeaturesListPresenter presenter = this.presenter;
        if (savedInstanceState == null || presenter == null) {
            presenter = getPresenter();
        } else {
            footerInflated = false;
            if (savedInstanceState.getBoolean(KEY_EMPTY_STATE)
                    && presenter.getFeaturesCount() == 0) {
                showEmptyView();
            }
            if (savedInstanceState.getBoolean(KEY_ERROR_STATE)
                    && presenter.getFeaturesCount() == 0) {
                showErrorView();
            }
            if (presenter.getFeaturesCount() > 0) {
                initLoadingMoreView();
            }
        }
        this.presenter = presenter;

        featureAdapter = new FeatureAdapter(presenter, this);
        //Fix CCE occurs on android prior 18
        if (Build.VERSION.SDK_INT < Build.VERSION_CODES.JELLY_BEAN_MR2) {
            initLoadingMoreView();
        }
        if (listView != null) {
            listView.setAdapter(featureAdapter);
        }

    }

    @Override
    public void onSaveInstanceState(Bundle outState) {
        super.onSaveInstanceState(outState);
        if (emptyStateViewStub != null) {
            outState.putBoolean(KEY_EMPTY_STATE, emptyStateViewStub.getParent() == null);
        }
        if (errorStateViewStub != null) {
            outState.putBoolean(KEY_ERROR_STATE, errorStateViewStub.getParent() == null);
        }
    }

    private void registerLoadMoreListener() {
        if (listView != null) {
            listView.setOnScrollListener(new AbsListView.OnScrollListener() {
                @Override
                public void onScrollStateChanged(AbsListView view, int scrollState) {

                }

                @Override
                public void onScroll(AbsListView view, int firstVisibleItem, int visibleItemCount,
                                     int totalItemCount) {
                    int lastInScreen = firstVisibleItem + visibleItemCount;

                    if (totalItemCount > 0 && (lastInScreen == totalItemCount) && !isLoading) {
                        isLoading = true;
                        if (presenter != null) {
                            presenter.onLoadMoreTriggered();
                        }
                    }

                }
            });
        }
    }

    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    private void unRegisterLoadMoreListener() {
        if (listView != null) {
            listView.setOnScrollListener(null);
        }
    }

    @Override
    public void onVote(FeatureRequest feature) {
        if (presenter != null) {
            presenter.onUpVoteFeatureClicked(feature);
        }
    }

    @Override
    public void onUnVote(FeatureRequest feature) {
        if (presenter != null) {
            presenter.onUnVoteFeatureClicked(feature);
        }
    }

    @Override
    public void onFeatureClicked(int position) {
        if (presenter != null) {
            presenter.onFeatureClicked(position);
        }
    }

    @Override
    public void onClick(View view) {
        int id = view.getId();
        if (presenter == null) return;

        if (id == R.id.ib_empty_state_action) {
            presenter.onEmptyViewPositiveButtonClicked();
        } else if (errorStateViewStub != null && id == errorStateViewStub.getInflatedId()) {
            presenter.onErrorStateClicked();
        }
    }

    @Override
    public void navigateToAddNewFeatureScreen() {
        if (getActivity() == null) return;

        getActivity().getSupportFragmentManager()
                .beginTransaction()
                .add(R.id.instabug_fragment_container, new AddNewFeatureFragment())
                .addToBackStack("search_features")
                .commit();
    }

    @Override
    public boolean shouldShowMyPosts() {
        return shouldShowMyPosts;
    }

    @Override
    public void onSortActionChanged(Boolean sortByTopVoted) {
        if (listView != null) {
            listView.smoothScrollToPosition(0);
        }
        registerLoadMoreListener();

        if (presenter != null) {
            presenter.onSortActionChanged();
        }
    }

    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    private void initLoadingMoreView() {
        // Use a local allocation to overcome SpotBugs false-positive possible NPE warnings
        ListView listView = this.listView;
        FeaturesListPresenter presenter = this.presenter;
        if (getContext() == null || listView == null || presenter == null || footerView == null)
            return;

        try {
            if (footerInflated) {
                listView.removeFooterView(footerView);
                listView.addFooterView(footerView);
                return;
            }
            footerView = View.inflate(getContext(), R.layout.ib_fr_pull_to_refresh_footer_view,
                    null);

            if (footerView == null) return;
            loadMoreProgressBar = footerView.findViewById(R.id.ib_loadmore_progressbar);

            poweredByFooter = footerView.findViewById(R.id.instabug_pbi_container);

            if (loadMoreProgressBar != null) {
                loadMoreProgressBar.setVisibility(View.INVISIBLE);
                loadMoreProgressBar
                        .getIndeterminateDrawable()
                        .setColorFilter(SettingsManager.getInstance().getPrimaryColor(), PorterDuff.Mode.SRC_IN);
            }

            listView.addFooterView(footerView);
            presenter.handlePbiFooter();

            footerInflated = true;
        } catch (Exception e) {
            InstabugSDKLogger.e(Constants.LOG_TAG,
                    "exception occurring while setting up the loadMore views", e);
        } finally {
            this.listView = listView;
            this.presenter = presenter;
        }
    }

    @Override
    public void handlePoweredByFooter() {
        if (getActivity() != null && footerView != null && poweredByFooter != null) {
            if (InstabugCore.getFeatureState(IBGFeature.WHITE_LABELING) == Feature.State.ENABLED
                    && !SettingsManager.getInstance().isCustomBrandingEnabled()) {
                poweredByFooter.setVisibility(View.INVISIBLE);
                return;
            }
            InstabugCore.handlePbiFooter(footerView);
            InstabugCore.handlePbiFooterThemeColor(
                    footerView, R.color.pbi_footer_color_dark, R.color.pbi_footer_color_light
            );
        }
    }

    @Override
    public void showFooterView() {
        if (loadMoreProgressBar != null) {
            loadMoreProgressBar.setVisibility(View.VISIBLE);
        }
    }

    @Override
    public void hideFooterView() {
        if (loadMoreProgressBar != null) {
            loadMoreProgressBar.setVisibility(View.GONE);
        }
    }

    @Override
    public void onDestroy() {
        super.onDestroy();
        if (presenter != null) {
            presenter.onDestroy();
        }
    }

    @Override
    public void onVoteChange() {
        if (featureAdapter != null)
            featureAdapter.notifyDataSetChanged();
    }

    @Override
    public void onRefresh() {
        registerLoadMoreListener();
        if (presenter != null) {
            presenter.onPullToRefreshTriggered();
        }
    }

    @Override
    public void setRefreshing(boolean isRefreshing) {
        if (swipeRefreshLayout != null) {
            swipeRefreshLayout.setRefreshing(isRefreshing);
        }
    }

    @Override
    public void hideLoadMore() {
        hideFooterView();
    }

    @Override
    public void updateItemsOrder() {
        if (listView != null) {
            listView.smoothScrollToPosition(0);
        }
        registerLoadMoreListener();
        if (presenter != null) {
            presenter.onSortActionChanged();
        }
    }

    @Override
    public void showErrorToastMessage() {
        if (getActivity() != null) {
            showToast(getLocalizedString(R.string.feature_requests_error_state_sub_title));
        }
    }
}