package com.instabug.terminations.sync

import android.annotation.SuppressLint
import androidx.annotation.VisibleForTesting
import com.instabug.commons.di.CommonsLocator.configurationsProvider
import com.instabug.library.diagnostics.IBGDiagnostics
import com.instabug.library.featuresflags.EnhancementRequestBodyParams
import com.instabug.library.featuresflags.di.FeaturesFlagServiceLocator.featuresFlagsConfigsProvider
import com.instabug.library.model.Attachment
import com.instabug.library.model.State
import com.instabug.library.model.State.KEY_CURRENT_ACTIVITY
import com.instabug.library.networkv2.request.Endpoints
import com.instabug.library.networkv2.request.FileToUpload
import com.instabug.library.networkv2.request.Header
import com.instabug.library.networkv2.request.Request
import com.instabug.library.networkv2.request.RequestMethod
import com.instabug.library.networkv2.request.RequestParameter
import com.instabug.library.networkv2.request.RequestType
import com.instabug.library.networkv2.request.getTokenFromState
import com.instabug.terminations.di.ServiceLocator.terminationsConfigurationProvider
import com.instabug.terminations.model.Termination
import org.json.JSONObject

private const val KEY_EXCEPTION = "exception"
private const val KEY_NAME = "name"
private const val KEY_MESSAGE = "message"
private const val KEY_STACK_TRACE = "stackTrace"
private const val KEY_TITLE = "title"
private const val KEY_ERROR = "error"
private const val TERMINATION_MESSAGE =
    "The user terminated the app then relaunched it within %d seconds"
private const val TERMINATION_NAME = "User Termination"
private const val KEY_ID = "id"
private const val KEY_REPORTED_AT = "reported_at"

private const val PARAM_FILE_TYPE = "metadata[file_type]"
private const val PARAM_DURATION = "metadata[duration]"

@VisibleForTesting
const val PARAM_ATTACHMENTS_COUNT = "attachments_count"

class TerminationRequestBuilder {

    @SuppressLint("WrongConstant")
    fun reportRequest(termination: Termination): Request {
        val builder = Request.Builder()
            .endpoint(Endpoints.REPORT_USER_TERMINATION)
            .method(RequestMethod.POST)
            .getTokenFromState(termination.state)

        termination.metadata.uuid?.let { uuid ->
            with(builder) {
                addHeader(RequestParameter(Header.ID, uuid))
                addParameter(RequestParameter(KEY_ID, uuid))
            }
        }
        val userIdentificationEnabled = configurationsProvider.userIdentificationEnabled
        val stateItems = EnhancementRequestBodyParams().getModifiedStateItemsList(
            termination.state?.getStateItems(userIdentificationEnabled),
            featuresFlagsConfigsProvider.mode
        ).toMutableMap()

        stateItems.remove(KEY_CURRENT_ACTIVITY)
        for ((key, value) in stateItems) {
            if (value != null)
                builder.addParameter(RequestParameter(key, value))
        }
        updateReportedAtIfNeeded(builder, termination)

        JSONObject().apply {
            JSONObject().apply {
                put(KEY_NAME, TERMINATION_NAME)
                val threshold = terminationsConfigurationProvider.threshold / 1000
                put(KEY_EXCEPTION, "$TERMINATION_NAME: ${TERMINATION_MESSAGE.format(threshold)}")
                put(KEY_MESSAGE, TERMINATION_MESSAGE.format(threshold))
                put(KEY_STACK_TRACE, "")
            }.also { errorObject -> put(KEY_ERROR, errorObject) }
        }.also { title -> builder.addParameter(RequestParameter(KEY_TITLE, title.toString())) }

        builder.addParameter(
            RequestParameter(PARAM_ATTACHMENTS_COUNT, termination.attachments.size)
        )

        return builder.build()
    }

    private fun updateReportedAtIfNeeded(builder: Request.Builder, termination: Termination) {
        termination.state?.run { if (!isMinimalState && reportedAt != 0L) return }
        runCatching {
            //Termination id is basically a UTC time in milliseconds. So, it can be used as the report time.
            RequestParameter(KEY_REPORTED_AT, (termination.id))
                .let { requestParam -> builder.addParameter(requestParam) }
        }.getOrElse { throwable ->
            IBGDiagnostics.reportNonFatal(
                throwable,
                "Failed to update reported_at in termination reporting request."
            )
        }
    }

    fun uploadLogsRequest(termination: Termination): Request {
        val builder = Request.Builder()
            .endpoint(termination.temporaryServerToken?.let { token ->
                Endpoints.CRASH_LOGS.replace(":crash_token".toRegex(), token)
            })
            .method(RequestMethod.POST)
            .getTokenFromState(termination.state)

        termination.state?.logsItems
            ?.filterNot { item -> item.key == null }
            ?.forEach { (key, value) -> builder.addParameter(RequestParameter(key, value ?: "")) }

        return builder.build()
    }

    fun singleAttachmentRequest(termination: Termination, attachment: Attachment): Request? =
        termination.temporaryServerToken?.let { token ->
            val endpoint = Endpoints.ADD_CRASH_ATTACHMENT.replace(":crash_token".toRegex(), token)
            val builder = Request.Builder()
                .endpoint(endpoint)
                .method(RequestMethod.POST)
                .type(RequestType.MULTI_PART)
                .getTokenFromState(termination.state)
            attachment.type?.let { builder.addParameter(RequestParameter(PARAM_FILE_TYPE, it)) }
            attachment.duration?.takeIf { attachment.type == Attachment.Type.AUDIO }
                ?.let { builder.addParameter(RequestParameter(PARAM_DURATION, it)) }
            val name = attachment.name
            val localPath = attachment.localPath
            if (name != null && localPath != null) {
                builder.fileToUpload(FileToUpload("file", name, localPath, attachment.fileType))
            }
            builder.build()
        }
}

private operator fun <V> State.StateItem<V>.component1(): String? = key

private operator fun <V> State.StateItem<V>.component2(): V? = value

