package com.instabug.terminations

import android.app.ActivityManager.RunningAppProcessInfo
import android.content.Context
import android.os.Build
import androidx.annotation.RequiresApi
import com.instabug.commons.DefaultOSExitInfoExtractor
import com.instabug.commons.OSExitInfo
import com.instabug.commons.OSExitInfoExtractor
import com.instabug.commons.caching.SessionCacheDirectory
import com.instabug.commons.isAnr
import com.instabug.commons.isOfVisibleImportance
import com.instabug.commons.isUserTermination
import com.instabug.commons.logging.logError
import com.instabug.commons.logging.logVerbose
import com.instabug.commons.snapshot.StateSnapshotCaptor
import com.instabug.commons.snapshot.ifNotExists
import com.instabug.commons.snapshot.readJsonObject
import com.instabug.commons.utils.dropReproStepsIfNeeded
import com.instabug.commons.utils.getScreenshotsDir
import com.instabug.commons.utils.modifyWithHubData
import com.instabug.library.IssueType
import com.instabug.library.SpansCacheDirectory
import com.instabug.library.frustratingexperience.FrustratingExperienceEvent
import com.instabug.library.frustratingexperience.FrustratingExperienceEventBus
import com.instabug.library.frustratingexperience.FrustratingExperienceType
import com.instabug.library.model.State
import com.instabug.library.tracking.FirstFGTimeProvider
import com.instabug.terminations.Constants.Preferences
import com.instabug.terminations.cache.TerminationsCacheDir
import com.instabug.terminations.cache.TerminationsCacheDir.Companion.BASELINE_FILE_SUFFIX
import com.instabug.terminations.cache.TerminationsCacheDir.Companion.DETECTED_FILE_SUFFIX
import com.instabug.terminations.cache.TerminationsCacheDir.Companion.GROUND_STATE_BG
import com.instabug.terminations.cache.TerminationsCacheDir.Companion.GROUND_STATE_FG
import com.instabug.terminations.cache.TerminationsCacheDir.Companion.MIGRATED_FILE_SUFFIX
import com.instabug.terminations.cache.TerminationsCacheDir.Companion.VALIDATED_FILE_SUFFIX
import com.instabug.terminations.cache.TerminationsCacheDir.Companion.getBaselineFile
import com.instabug.terminations.cache.TerminationsCacheDir.Companion.getDetectedFile
import com.instabug.terminations.cache.TerminationsCacheDir.Companion.getOldTerminationSnapshotFile
import com.instabug.terminations.cache.TerminationsCacheDir.Companion.getTerminationsDir
import com.instabug.terminations.cache.TerminationsCacheDir.Companion.getValidatedFile
import com.instabug.terminations.cache.TerminationsCacheDir.Companion.markBaselineFileAsDetected
import com.instabug.terminations.cache.TerminationsCacheDir.Companion.markDetectedFileAsValidated
import com.instabug.terminations.cache.TerminationsCacheDir.Companion.markDetectionFileAsMigrated
import com.instabug.terminations.cache.TerminationsCacheDir.Companion.markTerminationSnapshotFileAs
import com.instabug.terminations.cache.TerminationsCachingManager
import com.instabug.terminations.model.Termination
import java.io.File

sealed class MigrationResult {
    object Failed : MigrationResult()
    class Migrated(
        val incidents: List<Termination>,
        val migratedSessions: List<String>
    ) : MigrationResult()
}

fun interface TerminationMigrator {
    operator fun invoke(): MigrationResult
}

@RequiresApi(Build.VERSION_CODES.R)
class PostAndroidRMigrator(
    private val ctx: Context?,
    private val crashesCacheDir: SessionCacheDirectory,
    private val validator: TerminationsValidator,
    private val firstFGProvider: FirstFGTimeProvider,
    private val cachingManager: TerminationsCachingManager,
    private val reproScreenshotsDir: SpansCacheDirectory
) : TerminationMigrator {

    private var currentSessionDirectory: File? = null
    private lateinit var oldSessionsDirectories: List<File>
    private var firstFGTime: Long? = null

    override fun invoke(): MigrationResult {
        ctx ?: run {
            "Couldn't start terminations migration (lack of Context)".logError()
            return MigrationResult.Failed
        }
        currentSessionDirectory = crashesCacheDir.currentSessionDirectory
        oldSessionsDirectories = crashesCacheDir.oldSessionsDirectories
        firstFGTime = firstFGProvider.firstFGTime
        DefaultOSExitInfoExtractor().extract(ctx, Preferences.exitInfoBaselineSpecs)
            .also(this::markCurrentSessionWithBaseline)
            .also(this::markNewDetections)
        val result = oldSessionsDirectories.asSequence()
            .onEach(this::validateOldDetection)
            .mapNotNull(this::migrate)
            .toList()
            .let(this::composeMigratedResult)

        return firstFGTime?.let { result } ?: MigrationResult.Failed
    }

    private fun markCurrentSessionWithBaseline(result: OSExitInfoExtractor.Result) {
        runCatching {
            currentSessionDirectory
                ?.also { getTerminationsDir(it).ifNotExists { mkdirs() } }
                ?.also { sessionDir ->
                    val baselineFile = getBaselineFile(sessionDir, result.currentBaseline)
                    getBaselineFile(sessionDir)
                        ?.takeIf { it.exists() }
                        ?.renameTo(baselineFile)
                        ?: baselineFile.ifNotExists { createNewFile() }
                }?.also { "Trm Migrator-> Marked current session with Baseline".logVerbose() }
        }
    }

    private fun markNewDetections(result: OSExitInfoExtractor.Result) {
        runCatching {
            val latestOSDetection = result.infoList
                .also { "Trm Migrator-> info list: $it".logVerbose() }
                .firstOrNull(this::isTargetReason)
                ?: run { "Trm Migrator-> no valid exit info found, skipping ..".logVerbose(); return }
            val state = latestOSDetection.importance
                .takeIf { it == RunningAppProcessInfo.IMPORTANCE_FOREGROUND }
                ?.let { GROUND_STATE_FG } ?: GROUND_STATE_BG
            oldSessionsDirectories
                .map { getBaselineFile(it, result.oldBaseline) }
                .firstOrNull { it.exists() }
                ?.also { markBaselineFileAsDetected(it, state, latestOSDetection.timestamp) }
                ?.let { "Trm Migrator-> Marked detection for bl ${it.absolutePath}".logVerbose() }
        }
    }

    private fun isTargetReason(info: OSExitInfo) =
        with(info) { isUserTermination() || (isAnr() && isOfVisibleImportance()) }

    private fun validateOldDetection(sessionDir: File) = firstFGTime?.let { fgTime ->
        runCatching {
            getDetectedFile(sessionDir)?.run {
                val (detectionStateFG, stateSuffix) = name.contains(GROUND_STATE_FG)
                    .takeIf { it }
                    ?.let { (true to GROUND_STATE_FG) } ?: (false to GROUND_STATE_BG)
                val detectionTime = name.removeSuffix("$stateSuffix$DETECTED_FILE_SUFFIX").toLong()
                val isDetectionOnForeground =
                    isDetectionOnForeground(sessionDir) || detectionStateFG
                        .also { "Trm Migrator-> detection on foreground $it".logVerbose() }
                validator(fgTime, detectionTime)
                    .takeIf { isValid -> isValid && isDetectionOnForeground }
                    ?.let { markDetectedFileAsValidated(this, stateSuffix) }
                    ?.also { "Trm Migrator-> Marked $absolutePath as valid".logVerbose() }
                    ?: run {
                        "Trm Migrator-> Detection $absolutePath is not valid".logVerbose()
                        markDetectionFileAsMigrated(this, "$stateSuffix$DETECTED_FILE_SUFFIX")
                    }
            }
        }
    }

    private fun isDetectionOnForeground(sessionDir: File) =
        getTerminationSnapshot(sessionDir)?.foregroundTimeline
            ?.reduce { acc, isOnForeground -> acc || isOnForeground } ?: true

    private fun migrate(sessionDir: File): Termination? = runCatching {
        val validatedDetectionFile = getValidatedFile(sessionDir) ?: run {
            getBaselineFile(sessionDir)?.let {
                markDetectionFileAsMigrated(it, BASELINE_FILE_SUFFIX)
            }
            return null
        }
        val id = validatedDetectionFile.name.removeSuffix(VALIDATED_FILE_SUFFIX).toLong()
        val state = StateSnapshotCaptor.getStateSnapshot(sessionDir)
            ?.also { updateSessionCompositeId(it, sessionDir) }
            .apply { modifyWithHubData() }
            .also { it.dropReproStepsIfNeeded(IssueType.ForceRestart) }
        val screenshotsDir = state?.getScreenshotsDir(reproScreenshotsDir, IssueType.ForceRestart)
        "Trm Migrator-> Migrating ${validatedDetectionFile.absolutePath}".logVerbose()
        Termination.Factory(ctx, id, sessionDir.name, state, screenshotsDir).apply {
            ctx?.let { cachingManager.insertAndTrim(ctx, this) }
                .also {
                    FrustratingExperienceEventBus.post(
                        FrustratingExperienceEvent.Detected(
                            FrustratingExperienceType.FORCE_RESTART,
                            this.id
                        )
                    )
                }
            markDetectionFileAsMigrated(validatedDetectionFile, VALIDATED_FILE_SUFFIX)
            markTerminationSnapshotFileAs(sessionDir, MIGRATED_FILE_SUFFIX)
        }
    }.getOrNull()

    private fun composeMigratedResult(incidents: List<Termination>) =
        MigrationResult.Migrated(incidents, oldSessionsDirectories.map { it.name })


    private fun getTerminationSnapshot(sessionDirectory: File): PostAndroidRTerminationSnapshot? =
        getTerminationSnapshotFile(sessionDirectory)?.let(File::readJsonObject)
            ?.let(PostAndroidRTerminationSnapshot::fromJson)

    private fun getTerminationSnapshotFile(sessionDirectory: File): File? =
        getTerminationsDir(sessionDirectory).takeIf { it.exists() }?.let { terminationsDir ->
            TerminationsCacheDir.getTerminationSnapshotFile(terminationsDir).takeIf { it.exists() }
                ?: getOldTerminationSnapshotFile(terminationsDir).takeIf { it?.exists() == true }
        }

    private fun updateSessionCompositeId(state: State, sessionDir: File) {
        state.takeUnless { it.sessionId != null }
            ?.apply { sessionId = getTerminationSnapshot(sessionDir)?.sessionCompositeId }
    }
}

class PreAndroidRMigrator(
    private val ctx: Context?,
    private val crashesCacheDir: SessionCacheDirectory,
    private val validator: TerminationsValidator,
    private val firstFGProvider: FirstFGTimeProvider,
    private val cachingManager: TerminationsCachingManager,
    private val reproScreenshotsDir: SpansCacheDirectory
) : TerminationMigrator {

    private lateinit var oldSessionsDirectories: List<File>
    private var firstFGTime: Long? = null
    override fun invoke(): MigrationResult {
        oldSessionsDirectories = crashesCacheDir.oldSessionsDirectories
        firstFGTime = firstFGProvider.firstFGTime
        val result = oldSessionsDirectories.asSequence()
            .onEach(this::validate)
            .mapNotNull(this::migrate)
            .toList()
            .let(this::composeMigratedResult)
        return firstFGTime?.let { result } ?: MigrationResult.Failed
    }

    private fun validate(sessionDirectory: File) {
        runCatching {
            getTerminationSnapshot(sessionDirectory)?.run {
                val hasNDKCrash = getNDKFiles(sessionDirectory).isNotEmpty()
                val wasCrashing = hasNDKCrash || hasCrashed
                val isOnForeground = foregroundTimeline.reduce { acc, isOnFG -> acc || isOnFG }
                val hasNoTrmIndication = !isInAnr && !isOnForeground
                if (wasCrashing || hasNoTrmIndication) {
                    markTerminationSnapshotFileAs(sessionDirectory, MIGRATED_FILE_SUFFIX)
                    "Trm Migrator-> Snapshot $this for session ${sessionDirectory.name} is not eligible for validation".logVerbose()
                    return
                }
                "Trm Migrator-> Snapshot $this for session ${sessionDirectory.name} is eligible for validation".logVerbose()
                firstFGTime?.let { fgTime -> validator(fgTime, timestamp) }
                    ?.takeIf { isValid -> isValid }
                    ?.let { markTerminationSnapshotFileAs(sessionDirectory, VALIDATED_FILE_SUFFIX) }
                    ?.also { "Trm Migrator-> Validated session ${sessionDirectory.name}".logVerbose() }
                    ?: markTerminationSnapshotFileAs(sessionDirectory, MIGRATED_FILE_SUFFIX)
            }
        }
    }

    private fun migrate(sessionDir: File): Termination? = runCatching {
        val validatedSnapshot: PreAndroidRTerminationSnapshot =
            getValidatedFile(sessionDir)?.let(File::readJsonObject)
                ?.let(PreAndroidRTerminationSnapshot::fromJson)
                ?: return null
        val id = validatedSnapshot.timestamp
        val state = StateSnapshotCaptor.getStateSnapshot(sessionDir)
            ?.also { updateSessionCompositeId(it, validatedSnapshot) }
            .apply { modifyWithHubData() }
            .also { it.dropReproStepsIfNeeded(IssueType.ForceRestart) }
        val screenshotsDir = state?.getScreenshotsDir(reproScreenshotsDir, IssueType.ForceRestart)
        Termination.Factory(ctx, id, sessionDir.name, state, screenshotsDir).apply {
            ctx?.let { cachingManager.insertAndTrim(ctx, this) }
                .also {
                    FrustratingExperienceEventBus.post(
                        FrustratingExperienceEvent.Detected(
                            FrustratingExperienceType.FORCE_RESTART,
                            this.id
                        )
                    )
                }
            markTerminationSnapshotFileAs(sessionDir, MIGRATED_FILE_SUFFIX)
        }
    }.getOrNull()

    private fun composeMigratedResult(incidents: List<Termination>) = run {
        val sessionIdsWithIncidents = incidents.mapNotNull { it.sessionId }
        val sessionIdsWithNoIncidents = oldSessionsDirectories.map { it.name }
            .filterNot { it in sessionIdsWithIncidents }
        MigrationResult.Migrated(incidents, sessionIdsWithNoIncidents)
    }

    private fun getNDKFiles(sessionDirectory: File): List<String> =
        sessionDirectory.run { File("$absolutePath${File.separator}ndk") }
            .takeIf { it.exists() }?.list()?.toList() ?: emptyList()

    private fun getTerminationSnapshot(sessionDirectory: File): PreAndroidRTerminationSnapshot? =
        getTerminationSnapshotFile(sessionDirectory)?.let(File::readJsonObject)
            ?.let(PreAndroidRTerminationSnapshot::fromJson)

    private fun getTerminationSnapshotFile(sessionDirectory: File): File? =
        getTerminationsDir(sessionDirectory).takeIf { it.exists() }?.let { terminationsDir ->
            TerminationsCacheDir.getTerminationSnapshotFile(terminationsDir).takeIf { it.exists() }
                ?: getOldTerminationSnapshotFile(terminationsDir).takeIf { it?.exists() == true }
        }

    private fun updateSessionCompositeId(state: State, snapshot: PreAndroidRTerminationSnapshot) {
        state.takeUnless { it.sessionId != null }
            ?.apply { sessionId = snapshot.sessionCompositeId }
    }
}
