package com.instabug.fatalhangs.model

import android.content.Context
import android.net.Uri
import androidx.annotation.WorkerThread
import com.instabug.commons.AttachmentsHolder
import com.instabug.commons.BasicAttachmentsHolder
import com.instabug.commons.caching.DiskHelper
import com.instabug.commons.caching.DiskHelper.getIncidentStateFile
import com.instabug.commons.caching.DiskHelper.getReproScreenshotsZipPath
import com.instabug.commons.models.Incident
import com.instabug.commons.models.Incident.Type
import com.instabug.commons.models.IncidentMetadata
import com.instabug.commons.utils.updateScreenShotAnalytics
import com.instabug.crash.Constants
import com.instabug.fatalhangs.constants.Constants.FATAL_HANG_MESSAGE
import com.instabug.fatalhangs.constants.Constants.FATAL_HANG_MESSAGE_PLACE_HOLDER
import com.instabug.fatalhangs.di.FatalHangsServiceLocator
import com.instabug.library.Feature
import com.instabug.library.IBGFeature
import com.instabug.library.core.InstabugCore
import com.instabug.library.internal.storage.AttachmentsUtility
import com.instabug.library.internal.storage.DiskUtils
import com.instabug.library.internal.storage.cache.db.userAttribute.UserAttributesDbHelper
import com.instabug.library.internal.storage.operation.WriteStateToFileDiskOperation
import com.instabug.library.logging.InstabugLog
import com.instabug.library.model.Attachment
import com.instabug.library.model.State
import com.instabug.library.settings.SettingsManager
import com.instabug.library.util.InstabugSDKLogger
import com.instabug.library.util.ReportHelper
import com.instabug.library.util.memory.MemoryUtils
import org.json.JSONException
import org.json.JSONObject
import java.io.File

class FatalHang(
    val id: String,
    override val metadata: IncidentMetadata
) : Incident, AttachmentsHolder by BasicAttachmentsHolder() {
    companion object {
        private const val TAG = "FatalHang"
        private const val FATAL_HANG_STATE = "fatal_hang_state"
    }

    var mainThreadData: String? = null
    var restOfThreadsData: String? = null

    @FatalHangState
    var fatalHangState = FatalHangState.READY_TO_BE_SENT
    var temporaryServerToken: String? = null
    var state: State? = null
    var stateUri: Uri? = null
    var message: String? = null
    var lastActivity: String = "NA"

    override val type: Type = Type.FatalHang
    override fun getSavingDirOnDisk(ctx: Context): File =
        DiskHelper.getIncidentSavingDirectory(ctx, type.name, id)

    object Factory {
        @WorkerThread
        fun createFatalHang(
            context: Context?,
            hangDuration: Long,
            mainThreadData: JSONObject,
            threadsData: String,
            metadata: IncidentMetadata
        ): FatalHang? {
            context?.let { _context ->
                return FatalHang(System.currentTimeMillis().toString(), metadata).apply {
                    this.message = FATAL_HANG_MESSAGE.replace(
                        FATAL_HANG_MESSAGE_PLACE_HOLDER,
                        hangDuration.toString()
                    )
                    val error: JSONObject? = mainThreadData.optJSONObject("error")
                    error
                        ?.put("name", "Fatal Hang")
                        ?.put("exception", "Fatal Hang: $message")
                        ?.put("message", "Fatal Hang: $message")
                    var stacktrace = error?.optString("stackTrace")
                    stacktrace?.let {
                        stacktrace = "Fatal Hang: $message$stacktrace"
                        error?.put("stackTrace", stacktrace)
                    }
                    error?.let {
                        mainThreadData.put("error", it)
                    }
                    this.mainThreadData = mainThreadData.toString()
                    this.restOfThreadsData = threadsData
                    FatalHangsServiceLocator.trackingDelegate.currentRealActivity
                        ?.javaClass?.name?.let { activityName -> this.lastActivity = activityName }
                    this.state = State.getState(_context)
                    updateState(_context, this.state)
                    val report = ReportHelper.getReport(InstabugCore.getOnReportCreatedListener())
                    ReportHelper.update(this.state, report)
                    this.stateUri =
                        persistStateToFile(_context, this.state, getSavingDirOnDisk(_context))
                    this.state = null
                    addReproScreenshotsAttachmentIfApplicable(this, _context)
                    updateAttachmentsList(context, this)
                }
            }
            InstabugSDKLogger.v(
                TAG,
                "Couldn't create a new instance of FatalHang due to a null context."
            )
            return null
        }

        private fun updateAttachmentsList(context: Context, fatalHang: FatalHang) {
            // Add File attachments
            if (InstabugCore.getExtraAttachmentFiles() != null &&
                InstabugCore.getExtraAttachmentFiles()!!.size >= 1
            ) {
                for ((key, value) in InstabugCore.getExtraAttachmentFiles()!!) {
                    val attachmentUri = AttachmentsUtility.getNewFileAttachmentUri(
                        context,
                        key,
                        value
                    )
                    if (attachmentUri != null) {
                        fatalHang.addAttachment(attachmentUri, Attachment.Type.ATTACHMENT_FILE)
                    }
                }
            }
        }

        private fun persistStateToFile(_context: Context, state: State?, savingDir: File): Uri? {
            val file = getIncidentStateFile(savingDir, FATAL_HANG_STATE)
            return DiskUtils.with(_context)
                .writeOperation(WriteStateToFileDiskOperation(file, state?.toJson()))
                .execute()
        }

        private fun updateState(context: Context, state: State?) {
            state?.apply {
                if (!MemoryUtils.isLowMemory(context) && InstabugCore.getFeatureState(
                        IBGFeature.USER_EVENTS
                    ) == Feature.State.ENABLED
                ) {
                    try {
                        updateUserEvents()
                    } catch (e: JSONException) {
                        InstabugSDKLogger.e(
                            Constants.LOG_TAG, "Got error while parsing " +
                                    "user events logs", e
                        )
                    }
                }
                if (SettingsManager.getInstance().onReportCreatedListener == null) {
                    tags = InstabugCore.getTagsAsString()
                    if (InstabugCore.getFeatureState(IBGFeature.USER_DATA)
                        == Feature.State.ENABLED
                    ) {
                        userData = InstabugCore.getUserData()
                    }
                    if (InstabugCore.getFeatureState(IBGFeature.INSTABUG_LOGS)
                        == Feature.State.ENABLED
                    ) {
                        instabugLog = InstabugLog.getLogs()
                    }
                }
                userAttributes = UserAttributesDbHelper.getSDKUserAttributes()
                if (FatalHangsServiceLocator.reproConfigurationsProvider.isReproStepsEnabled) {
                    updateVisualUserSteps()
                }
                this.updateScreenShotAnalytics()

            }
        }

        @WorkerThread
        private fun addReproScreenshotsAttachmentIfApplicable(fatalHang: FatalHang, ctx: Context) {
            if (!FatalHangsServiceLocator.reproConfigurationsProvider.isReproScreenshotsEnabled) return
            val reproScreenshotsDir =
                FatalHangsServiceLocator.reproScreenshotsCacheDir.currentSpanDirectory ?: return
            val (zipPath, isEncrypted) = getReproScreenshotsZipPath(
                ctx,
                fatalHang.id,
                fatalHang.getSavingDirOnDisk(ctx),
                reproScreenshotsDir
            )
            zipPath?.let { path ->
                fatalHang.addAttachment(
                    Uri.parse(path),
                    Attachment.Type.VISUAL_USER_STEPS,
                    isEncrypted
                )
            }
        }
    }
}