package com.instabug.crash.network;

import static com.instabug.crash.utils.DeleteCrashUtilsKt.deleteAttachment;
import static com.instabug.library.networkv2.request.RequestExtKt.getTokenFromState;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;

import com.instabug.commons.di.CommonsLocator;
import com.instabug.crash.models.Crash;
import com.instabug.crash.models.IBGNonFatalException;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.diagnostics.IBGDiagnostics;
import com.instabug.library.featuresflags.EnhancementRequestBodyParams;
import com.instabug.library.featuresflags.di.FeaturesFlagServiceLocator;
import com.instabug.library.internal.storage.AttachmentsUtility;
import com.instabug.library.model.Attachment;
import com.instabug.library.model.State;
import com.instabug.library.networkv2.NetworkManager;
import com.instabug.library.networkv2.RateLimitedException;
import com.instabug.library.networkv2.RequestResponse;
import com.instabug.library.networkv2.request.Endpoints;
import com.instabug.library.networkv2.request.FileToUpload;
import com.instabug.library.networkv2.request.Header;
import com.instabug.library.networkv2.request.Request;
import com.instabug.library.networkv2.request.RequestMethod;
import com.instabug.library.networkv2.request.RequestParameter;
import com.instabug.library.networkv2.request.RequestType;
import com.instabug.library.util.InstabugSDKLogger;

import org.json.JSONException;
import org.json.JSONObject;

import java.io.File;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * @author mesbah
 */
public class CrashesService {
    private static final String TAG = "CrashesService";

    private static final String PARAM_TITLE = "title";
    private static final String PARAM_ATTACHMENTS_COUNT = "attachments_count";
    private static final String PARAM_FILE_TYPE = "metadata[file_type]";
    private static final String PARAM_DURATION = "metadata[duration]";
    private static final String PARAM_HANDLED = "handled";
    private static final String PARAM_THREADS_DETAILS = "threads_details";
    private static final String PARAM_GROUPING_STRING = "grouping_string";
    private static final String PARAM_LEVEL = "level";
    private static final String PARAM_ID = "id";
    private static final String PARAM_REPORTED_AT = "reported_at";
    @Nullable
    private static CrashesService INSTANCE;
    private final NetworkManager networkManager;

    private CrashesService() {
        networkManager = NetworkManager.newInstance();
    }

    /**
     * Returns the current singleton instance of this class.
     *
     * @return singleton instance of CrashesService
     */
    public static CrashesService getInstance() {
        if (INSTANCE == null) {
            INSTANCE = new CrashesService();
        }
        return INSTANCE;
    }

    public void reportCrash(final Crash crash,
                            final Request.Callbacks<String, Throwable> reportingCrashCallbacks)
            throws JSONException {
        reportCrash(crash, reportingCrashCallbacks, true);
    }

    public void reportCrash(final Crash crash,
                            final Request.Callbacks<String, Throwable> reportingCrashCallbacks,
                            final boolean canWriteToCache
    ) throws JSONException {
        InstabugSDKLogger.d(com.instabug.crash.Constants.LOG_TAG, "Reporting crash with crash message: " + crash.getCrashMessage());
        // build crash request
        Request reportingCrashRequest = buildCrashReportingRequest(crash);
        // do request with NORMAL request type.
        networkManager.doRequestOnSameThread(RequestType.NORMAL, reportingCrashRequest, new Request.Callbacks<RequestResponse, Throwable>() {
            @Override
            public void onSucceeded(RequestResponse requestResponse) {
                InstabugSDKLogger.d(com.instabug.crash.Constants.LOG_TAG, "reportingCrashRequest Succeeded, Response code: " + requestResponse.getResponseCode());
                InstabugSDKLogger.v(com.instabug.crash.Constants.LOG_TAG, "reportingCrashRequest Succeeded, Response body: " + requestResponse.getResponseBody());
                try {
                    if (requestResponse.getResponseBody() != null) {
                        reportingCrashCallbacks.onSucceeded(new JSONObject((String) requestResponse
                                .getResponseBody()).getString("id"));
                    } else {
                        reportingCrashCallbacks.onFailed(
                                new JSONException("requestResponse.getResponseBody() returned null"));
                    }
                } catch (JSONException e) {
                    e.printStackTrace();
                    reportingCrashCallbacks.onFailed(e);
                }
            }

            @Override
            public void onFailed(Throwable error) {
                if (error instanceof RateLimitedException) {
                    reportingCrashCallbacks.onFailed(error);
                    return;
                }
                IBGDiagnostics.reportNonFatalAndLog(error, "Reporting crash got error: " + error.getMessage(), com.instabug.crash.Constants.LOG_TAG);
                InstabugSDKLogger.e(TAG, "reportingCrashRequest got error: ", error);
                InstabugCore.reportError(error, "Reporting crash got error: " + error.getMessage());
                if (canWriteToCache) {
                    AttachmentsUtility.encryptAttachmentsAndUpdateDb(crash.getAttachments());
                }

                reportingCrashCallbacks.onFailed(error);
            }
        });
    }

    @NonNull
    @VisibleForTesting
    public Request buildCrashReportingRequest(Crash crash) throws JSONException {
        String endpoint = Endpoints.REPORT_CRASH;
        if (crash.isHandled()) endpoint = Endpoints.REPORT_NON_FATAL;
        Request.Builder requestBuilder = new Request.Builder()
                .endpoint(endpoint)
                .method(RequestMethod.POST);

        getTokenFromState(requestBuilder, crash.getState());

        if (crash.getMetadata().getUuid() != null)
            requestBuilder.addHeader(new RequestParameter<>(Header.ID, crash.getMetadata().getUuid()));

        State crashState = crash.getState();
        if (crashState != null) {
            boolean userIdentificationEnabled = CommonsLocator.getConfigurationsProvider().getUserIdentificationEnabled();
            EnhancementRequestBodyParams enhancementRequestBodyParams = new EnhancementRequestBodyParams();
            Map<String, Object> result = enhancementRequestBodyParams.getModifiedStateItemsList(crashState.getStateItems(userIdentificationEnabled), FeaturesFlagServiceLocator.getFeaturesFlagsConfigsProvider().getMode());
            for (Map.Entry<String, Object> entry : result.entrySet()) {
                requestBuilder.addParameter(new RequestParameter(entry.getKey(), entry.getValue()));
            }
        }
        updateReportedAtIfNeeded(requestBuilder, crash);
        String crashMessage = crash.getCrashMessage();
        if (crashMessage != null)
            requestBuilder.addParameter(new RequestParameter<>(PARAM_TITLE, crashMessage));
        requestBuilder.addParameter(new RequestParameter<>(PARAM_HANDLED, crash.isHandled()));
        String threadsDetails = crash.getThreadsDetails();
        if (threadsDetails != null)
            requestBuilder.addParameter(new RequestParameter<>(PARAM_THREADS_DETAILS, threadsDetails));
        String fingerprint = crash.getFingerprint();
        if (fingerprint != null)
            requestBuilder.addParameter(new RequestParameter<>(PARAM_GROUPING_STRING, new JSONObject(fingerprint)));
        IBGNonFatalException.Level level = crash.getLevel();
        if (level != null)
            requestBuilder.addParameter(new RequestParameter<>(PARAM_LEVEL, level.getSeverity()));
        String id = crash.getMetadata().getUuid();
        if (id != null)
            requestBuilder.addParameter(new RequestParameter<>(PARAM_ID, id));
        if (crash.getAttachments() != null && crash.getAttachments().size() > 0) {
            requestBuilder.addParameter(new RequestParameter<>(PARAM_ATTACHMENTS_COUNT, crash
                    .getAttachments().size()));
        }
        return requestBuilder.build();
    }

    private void updateReportedAtIfNeeded(Request.Builder builder, Crash crash) {
        State state = crash.getState();
        if (state != null && !state.isMinimalState() && state.getReportedAt() != 0) return;
        try {
            //Crash id is basically a UTC time in milliseconds. So, it can be used as the report time.
            long reportedAt = crash.getId() != null ? Long.parseLong(crash.getId()) : 0L;
            if (reportedAt != 0L) {
                builder.addParameter(new RequestParameter<>(PARAM_REPORTED_AT, reportedAt));
            }
        } catch (Exception e) {
            IBGDiagnostics.reportNonFatal(e, "Failed to update reported_at in crash reporting request.");
        }
    }

    public void uploadCrashAttachments(final Crash crash, final Request
            .Callbacks<Boolean, Crash> uploadCrashAttachmentsCallbacks) throws JSONException {


        final List<Attachment> synced = new ArrayList<>();
        if (crash.getAttachments().size() == 0) {
            uploadCrashAttachmentsCallbacks.onSucceeded(true);
            return;
        }
        for (int i = 0; i < crash.getAttachments().size(); i++) {
            // create attachment request.
            final Attachment attachment = crash.getAttachments().get(i);
            boolean isAttachmentDecrypted = AttachmentsUtility.decryptAttachmentAndUpdateDb(attachment);
            if (isAttachmentDecrypted) {
                Request uploadingCrashAttachmentRequest = buildCrashSingleAttachmentRequest(crash, attachment);

                if (attachment.getLocalPath() != null) {
                    File file = new File(attachment.getLocalPath());
                    if (file.exists() && file.length() > 0) {
                        attachment.setAttachmentState(Attachment.AttachmentState.SYNCED);
                        networkManager.doRequestOnSameThread(RequestType.MULTI_PART, uploadingCrashAttachmentRequest, new Request.Callbacks<RequestResponse, Throwable>() {
                            @Override
                            public void onSucceeded(RequestResponse requestResponse) {

                                InstabugSDKLogger.d(com.instabug.crash.Constants.LOG_TAG, "uploadingCrashAttachmentRequest succeeded, Response code:" + requestResponse.getResponseCode());
                                InstabugSDKLogger.v(com.instabug.crash.Constants.LOG_TAG, "uploadingCrashAttachmentRequest succeeded, Response body:" + requestResponse.getResponseBody());


                                if (attachment.getLocalPath() != null) {
                                    deleteAttachment(attachment, crash.getId());
                                    synced.add(attachment);

                                }

                                if (synced.size() == crash.getAttachments().size()) {
                                    uploadCrashAttachmentsCallbacks.onSucceeded(true);
                                }
                            }

                            @Override
                            public void onFailed(Throwable error) {
                                InstabugSDKLogger.e(com.instabug.crash.Constants.LOG_TAG, "uploadingCrashAttachmentRequest got error: " + error.getMessage());
                                uploadCrashAttachmentsCallbacks.onFailed(crash);
                            }
                        });
                    } else {
                        InstabugSDKLogger.w(com.instabug.crash.Constants.LOG_TAG, "Skipping attachment file of type "
                                + attachment.getType() + " because it's either not found or empty file");
                    }
                } else {
                    InstabugSDKLogger.w(com.instabug.crash.Constants.LOG_TAG, "Skipping attachment file of type "
                            + attachment.getType() + " because it's either not found or empty file");
                }
            } else {
                InstabugSDKLogger.w(com.instabug.crash.Constants.LOG_TAG, "Skipping attachment file of type "
                        + attachment.getType() + " because it was not decrypted successfully");
            }
        }
    }

    @NonNull
    @VisibleForTesting
    public Request buildCrashSingleAttachmentRequest(Crash crash, Attachment attachment) throws
            JSONException {

        Request.Builder requestBuilder = null;
        requestBuilder = new Request.Builder()
                .method(RequestMethod.POST)
                .type(RequestType.MULTI_PART);

        getTokenFromState(requestBuilder, crash.getState());

        if (crash.getTemporaryServerToken() != null) {
            requestBuilder.endpoint(Endpoints.ADD_CRASH_ATTACHMENT.replaceAll(":crash_token", crash.getTemporaryServerToken()));
        }


        if (attachment.getType() != null)
            requestBuilder.addParameter(new RequestParameter<>(PARAM_FILE_TYPE, attachment.getType()));
        if (attachment.getType() == Attachment.Type.AUDIO && attachment.getDuration() != null) {
            requestBuilder.addParameter(new RequestParameter<>(PARAM_DURATION, attachment
                    .getDuration()));
        }
        if (attachment.getName() != null && attachment.getLocalPath() != null) {
            requestBuilder.fileToUpload(new FileToUpload("file",
                    attachment.getName(), attachment.getLocalPath(), attachment.getFileType()));
        }
        return requestBuilder.build();
    }

    public void uploadCrashLogs(final Crash crash, final Request
            .Callbacks<Boolean, Crash> callbacks) {
        InstabugSDKLogger.d(com.instabug.crash.Constants.LOG_TAG, "START uploading all logs related to this crash id = " + crash.getId());
        // create logs request
        try {
            // build crash logs request
            Request crashLogsRequest = buildCrashLogsRequest(crash);
            // do network request
            networkManager.doRequestOnSameThread(RequestType.NORMAL, crashLogsRequest, new Request.Callbacks<RequestResponse, Throwable>() {
                @Override
                public void onSucceeded(RequestResponse requestResponse) {
                    InstabugSDKLogger.d(com.instabug.crash.Constants.LOG_TAG, "Uploading crash logs succeeded, Response code: " + requestResponse.getResponseCode());
                    InstabugSDKLogger.v(com.instabug.crash.Constants.LOG_TAG, "uploading crash logs onNext, Response body: " + requestResponse.getResponseBody());

                    callbacks.onSucceeded(true);
                }

                @Override
                public void onFailed(Throwable error) {
                    InstabugSDKLogger.e(com.instabug.crash.Constants.LOG_TAG, "uploading crash logs got error: " + error.getMessage());
                    callbacks.onFailed(crash);
                }
            });
        } catch (JSONException e) {
            InstabugSDKLogger.e(com.instabug.crash.Constants.LOG_TAG, "uploading crash logs got Json error: " + e.getMessage());
            callbacks.onFailed(crash);
        }
    }

    @NonNull
    @VisibleForTesting
    public Request buildCrashLogsRequest(Crash crash) throws JSONException {

        Request.Builder requestBuilder = new Request.Builder()
                .endpoint(Endpoints.CRASH_LOGS.replaceAll(":crash_token",
                        crash.getTemporaryServerToken() != null ? crash.getTemporaryServerToken() : ""))
                .method(RequestMethod.POST);

        getTokenFromState(requestBuilder, crash.getState());

        State crashState = crash.getState();
        if (crashState != null) {
            ArrayList<State.StateItem> logsItems = crashState.getLogsItems();
            if (logsItems != null && logsItems.size() > 0) {
                for (State.StateItem logItem : logsItems) {
                    if (logItem.getKey() != null) {
                        requestBuilder.addParameter(new RequestParameter<>(logItem.getKey(),
                                logItem.getValue() != null ? logItem.getValue() : ""));
                    }
                }
            }
        }
        return requestBuilder.build();
    }
    @VisibleForTesting
    public static void reset() {
        INSTANCE = null;
    }
}
