package com.instabug.commons.threading

import android.os.Looper
import com.instabug.commons.di.CommonsLocator
import com.instabug.commons.logging.getOrReportError
import com.instabug.commons.logging.logVerbose
import com.instabug.crash.utils.ExceptionFormatter
import org.json.JSONArray
import org.json.JSONObject

class CrashDetailsParser @JvmOverloads constructor(
    threadParsingStrategy: ThreadParsingStrategy,
    errorParsingStrategy: ErrorParsingStrategy,
    crashingThread: Thread? = null,
    threads: Set<Thread> = Thread.getAllStackTraces().keys,
    threadsLimit: Int = CommonsLocator.threadingLimitsProvider.provideThreadsLimit(),
    framesLimit: Int = CommonsLocator.threadingLimitsProvider.provideFramesLimit()
) {

    val crashDetails: JSONObject
    val threadsDetails: JSONArray

    init {
        val terminatedCount = threads.count { it.state == Thread.State.TERMINATED }
        val maintainedThreads = threads.extractMaintainedThreads(crashingThread)
        val operableThreads = threads.getOperableThreads(
            crashingThread = crashingThread,
            maintainedThreads = maintainedThreads,
            threadsLimitExclusive = threadsLimit - maintainedThreads.size
        )
        val droppedThreads =
            (threads.size - terminatedCount - operableThreads.size).takeUnless { it < 0 } ?: 0
        "Original threads' count = ${threads.size}, Terminated threads' count = $terminatedCount, Dropped threads' count = $droppedThreads".logVerbose()
        "First original thread ${threads.firstOrNull()}".logVerbose()
        "Last original thread ${threads.lastOrNull()}".logVerbose()
        crashDetails = runCatching {
            JSONObject().apply {
                threadParsingStrategy()?.let { threadObject -> put("thread", threadObject) }
                errorParsingStrategy()?.let { detailsObject -> put("error", detailsObject) }
                put("droppedThreads", droppedThreads)
                put("terminatedThreads", terminatedCount)
            }
        }.getOrReportError(JSONObject(), "Failed parsing crash details")
        threadsDetails = operableThreads.toFormattedData(crashingThread, framesLimit)
    }

    private fun Set<Thread>.extractMaintainedThreads(crashingThread: Thread?): Set<Thread> =
        asSequence().filter { it.isMainThread || it.isCrashing(crashingThread) }.toSet()

    private fun Set<Thread>.getOperableThreads(
        crashingThread: Thread?,
        maintainedThreads: Set<Thread>,
        threadsLimitExclusive: Int
    ): Set<Thread> = asSequence()
        .filterNot { it.state == Thread.State.TERMINATED }
        .filterNot { it.isCrashing(crashingThread) }
        .filterNot { it.isMainThread }
        .sortedByDescending { it.id }.take(threadsLimitExclusive)
        .toMutableSet().apply { addAll(maintainedThreads) }
        .sortedBy { thread -> thread.id }.toSet()

    /**
     * Represents the strategy of parsing the crashing thread object.
     */
    sealed class ThreadParsingStrategy {
        open operator fun invoke(): JSONObject? = null

        /**
         * Generates no crashing thread object.
         */
        object None : ThreadParsingStrategy()

        /**
         * Generates a crashing thread object with the main thread details.
         */
        object Main : ThreadParsingStrategy() {
            override fun invoke(): JSONObject = runCatching {
                Looper.getMainLooper().thread.formattedData
            }.getOrReportError(JSONObject(), "Failed parsing main thread data")

        }

        /**
         * Generates a crashing thread object with the actual crashing thread details.
         */
        class Crashing(private val thread: Thread) : ThreadParsingStrategy() {
            override fun invoke(): JSONObject = runCatching { thread.formattedData }
                .getOrReportError(JSONObject(), "Failed parsing crashing thread")
        }
    }

    /**
     * Represents the strategy of parsing the error and stacktrace object
     */
    sealed class ErrorParsingStrategy {
        open operator fun invoke(): JSONObject? = null

        /**
         * Generates no error object
         */
        object None : ErrorParsingStrategy()

        /**
         * Generates and error object with the stacktrace of the main thread
         * and customized name and exception.
         */
        class Main(
            private val name: String? = null,
            private val exception: String? = null
        ) : ErrorParsingStrategy() {
            override fun invoke(): JSONObject = runCatching {
                JSONObject().apply {
                    val mainThread = Looper.getMainLooper().thread
                    name?.let { put("name", it) }
                    exception?.let { put("exception", it) }
                    mainThread.stackTrace.getOrNull(0)?.let { latest ->
                        latest.fileName?.let { "$it:${latest.lineNumber}" }?.let { location ->
                            put("location", location)
                        }
                    }
                    mainThread.getFormattedStackTrace(CommonsLocator.threadingLimitsProvider.provideErrorThreadFramesLimit()) {
                        exception?.let { append(it).append("\n") }
                    }.also { put("stackTrace", it) }
                }
            }.getOrReportError(JSONObject(), "Failed parsing main thread error")
        }

        /**
         * Generates an error object with the stacktrace of crash throwable
         */
        class Crashing @JvmOverloads constructor(
            private val throwable: Throwable,
            private val identifier: String? = null
        ) : ErrorParsingStrategy() {
            override fun invoke(): JSONObject =
                ExceptionFormatter.createExceptionJson(throwable, identifier)
        }
    }
}