package com.instabug.commons.snapshot

import com.instabug.commons.logging.logVerbose

class CaptorsRegistry {

    private val registry: MutableMap<Int, MutableSet<Int>> = mutableMapOf()
    private val running: MutableList<Captor> = mutableListOf()

    fun start(launcherId: Int, captor: Captor): Unit = synchronized(this) {
        "Starting captor ${captor.id} for Launcher: $launcherId".logVerbose()
        val isRegisteredForLauncher = isCaptorRegisteredForLauncher(launcherId, captor.id)
        val hasRunningCaptor = isInstanceOfCaptorRunning(captor.id)
        if (isRegisteredForLauncher && hasRunningCaptor) return
        registerCaptorForLauncher(launcherId, captor)
        if (hasRunningCaptor) return
        with(running) { captor.start(); find { it.id == captor.id }?.let(::remove); add(captor) }
    }

    fun stop(launcherId: Int, captorId: Int): Unit = synchronized(this) {
        "Stopping captor $captorId for launcher: $launcherId".logVerbose()
        if (!isCaptorRegisteredForLauncher(launcherId, captorId)) return
        removeCaptorForLauncher(launcherId, captorId)
        if (isCaptorRegisteredForAnotherLauncher(launcherId, captorId)) return
        running.find { it.id == captorId }
            ?.also { it.shutdown() }
            ?.let { running.remove(it) }
    }

    fun force(launcherId: Int, captorId: Int): Unit = synchronized(this) {
        "Forcing captor $captorId for launcher: $launcherId".logVerbose()
        if (!isCaptorRegisteredForLauncher(launcherId, captorId)) return
        getRunningCaptorInstance(captorId)?.force()
    }

    fun forceOnAvailable(captorId: Int): Unit = synchronized(this) {
        "Forcing captor $captorId if available".logVerbose()
        getRunningCaptorInstance(captorId)?.force()
    }

    private fun isCaptorRegisteredForLauncher(launcherId: Int, captorId: Int): Boolean =
        registry.any { it.key == launcherId && it.value.contains(captorId) }

    private fun isInstanceOfCaptorRunning(captorId: Int): Boolean =
        running.any { it.id == captorId && !it.isShutdown }

    private fun isCaptorRegisteredForAnotherLauncher(launcherId: Int, captorId: Int): Boolean =
        registry.any { it.key != launcherId && it.value.contains(captorId) }

    private fun getRunningCaptorInstance(captorId: Int): Captor? =
        running.firstOrNull { it.id == captorId && !it.isShutdown }

    private fun registerCaptorForLauncher(launcherId: Int, captor: Captor) =
        registry.run { get(launcherId)?.add(captor.id) ?: put(launcherId, mutableSetOf(captor.id)) }

    private fun removeCaptorForLauncher(launcherId: Int, captorId: Int) =
        registry.run { get(launcherId)?.remove(captorId) }
}
