package com.instabug.commons.session

import com.instabug.commons.models.Incident.Type
import com.instabug.library.model.v3Session.IBGSessionData
import com.instabug.library.sessionV3.providers.SessionID
import org.json.JSONArray
import org.json.JSONObject

//region Session Request constants

private const val crashesSessionDataKey = "cd"
private const val fatalCrashesKey = "f"
private const val nonFatalCrashesKey = "nf"
private const val anrKey = "anr"
private const val bgAnrKey = "bg_anr"
private const val fatalHangKey = "fh"
private const val appTerminationKey = "ats"
private const val ndkKey = "ndkc"

//endregion
private fun incidentToRequestKey(type: Type) = when (type) {
    Type.ANR -> anrKey
    Type.BG_ANR -> bgAnrKey
    Type.FatalHang -> fatalHangKey
    Type.FatalCrash -> fatalCrashesKey
    Type.NonFatalCrash -> nonFatalCrashesKey
    Type.NDKCrash -> ndkKey
    Type.Termination -> appTerminationKey
}

fun incidentsToSessionsData(incidents: List<SessionIncident>): Map<String, IBGSessionData> =
    incidents
        .let(::groupIncidentsBySessionsIds)
        .mapValues { extractSessionData(it.value) }

fun groupIncidentsBySessionsIds(incidents: List<SessionIncident>): Map<String, List<SessionIncident>> =
    incidents.groupBy { it.sessionId }

fun complementSessionsMap(
    linkedSessions: Map<String, IBGSessionData>,
    fullSessions: List<SessionID>
) = (fullSessions - linkedSessions.keys)
    .associateWith { IBGSessionData(crashesSessionDataKey, JSONObject()) } + linkedSessions

fun extractSessionData(incidents: List<SessionIncident>): IBGSessionData = incidents
    .groupByIncidentsType()
    .mapValuesToIncidentsIds()
    .toSessionData()

private fun Map<String, List<String>>.toSessionData(): IBGSessionData {
    val jsonObject = constructCrashesJsonObject(this)
    return IBGSessionData(crashesSessionDataKey, jsonObject)
}

private fun constructCrashesJsonObject(incidents: Map<String, List<String>>): JSONObject {
    val jsonObject = JSONObject()
    incidents.forEach { incident ->
        incident.value.takeUnless { it.isEmpty() }
            ?.let { value -> jsonObject.put(incident.key, JSONArray(value)) }
    }
    return jsonObject
}

private fun Map<String, List<SessionIncident>>.mapValuesToIncidentsIds() =
    mapValues { entry -> entry.value.mapNotNull { incident -> incident.incidentId } }

private fun List<SessionIncident>.groupByIncidentsType() =
    groupBy { it.incidentType }
        .mapKeys { incident -> incidentToRequestKey(incident.key) }
