package com.instabug.commons.lifecycle

import com.instabug.commons.lifecycle.CompositeLifecycleOwner.CompositeLifecycleObserver
import com.instabug.library.core.CurrentFragmentLifeCycleEventBus
import com.instabug.library.core.eventbus.CurrentActivityLifeCycleEventBus
import com.instabug.library.core.eventbus.eventpublisher.IBGDisposable
import com.instabug.library.tracking.ActivityLifeCycleEvent
import com.instabug.library.tracking.FragmentLifeCycleEvent
import io.reactivexport.disposables.CompositeDisposable
import io.reactivexport.disposables.Disposable
import java.util.concurrent.ScheduledExecutorService
import java.util.concurrent.ScheduledFuture
import java.util.concurrent.TimeUnit

/**
 * A class adapting multiple lifecycle observer subscriptions to single output port [CompositeLifecycleObserver].
 * The class currently observes for ActivityLifecycle and FragmentLifecycle by default. Currently notifies
 * lifecycle as follows:
 * 1- When an activity started notifies immediately. [CompositeLifecycleObserver.onActivityStarted]
 * 2- When a fragment started notifies immediately. [CompositeLifecycleObserver.onFragmentStarted]
 * 3- When the started activities count reaches 0 notifies with a fixed delay (700ms), unless an activity
 * gets started within. [CompositeLifecycleObserver.onBackgrounded]
 */
class CompositeLifecycleOwner(
    private val notificationExecutor: ScheduledExecutorService
) {
    private var compositeDisposable: CompositeDisposable? = null
    private var currentActivityLifecycleDisposable: IBGDisposable? = null
    private val observers: MutableSet<CompositeLifecycleObserver> = mutableSetOf()

    /**
     * Initializes the observation of combined lifecycles.
     */
    fun init() {
        compositeDisposable = CompositeDisposable().apply {
            currentActivityLifecycleDisposable = subscribeToActivityLifecycle()
            add(this@CompositeLifecycleOwner.subscribeToFragmentLifecycle())
        }
    }

    /**
     * Disposes the observation base.
     */
    fun dispose() {
        compositeDisposable?.dispose()
        currentActivityLifecycleDisposable?.dispose()
        currentActivityLifecycleDisposable = null
        compositeDisposable = null
    }

    /**
     * Registers new observer to this composite lifecycle owner
     * @param observer the [CompositeLifecycleObserver] to be notified
     */
    fun register(observer: CompositeLifecycleObserver) {
        observers += observer
    }

    /**
     * Unregisters an observer
     * @param observer the [CompositeLifecycleObserver] to be unregistered
     */
    fun unregister(observer: CompositeLifecycleObserver) {
        observers -= observer
    }

    private fun subscribeToActivityLifecycle(): IBGDisposable =
        CurrentActivityLifeCycleEventBus.subscribe(this::handleActivityEvent)

    private fun subscribeToFragmentLifecycle(): Disposable =
        CurrentFragmentLifeCycleEventBus.getInstance().subscribe(this::handleFragmentEvent)

    private fun handleActivityEvent(event: ActivityLifeCycleEvent) = when (event) {
        ActivityLifeCycleEvent.STARTED -> {
            notifyActivityStarted()
        }

        else -> Unit // NoOp
    }

    private fun handleFragmentEvent(event: FragmentLifeCycleEvent) = when (event) {
        FragmentLifeCycleEvent.STARTED -> notifyFragmentStarted()
        else -> Unit // NoOp
    }

    private fun notifyActivityStarted() =
        synchronized(this) { observers.forEach { observer -> observer.onActivityStarted() } }

    private fun notifyFragmentStarted() =
        synchronized(this) { observers.forEach { observer -> observer.onFragmentStarted() } }


    interface CompositeLifecycleObserver {
        fun onActivityStarted()
        fun onFragmentStarted()
    }

    companion object {
        private const val EXECUTOR_NAME = "LifecycleNotifierExecutor"
    }
}
